#ifndef _AUDIO_GATE_H
#define _AUDIO_GATE_H

#include "chainop.h"
#include "samplebuffer.h"

class GATE_BASE : public CHAIN_OPERATOR {
  //
  // Interface to gate effects. When gate is open, the signal is let 
  // through untouched. When closed, signal is turned off complitely.
  //

private:

  bool gate_open;
  
protected:

  void gate_process(SAMPLE_BUFFER* sbuf);
  void close_gate(void) { gate_open = false; }
  void open_gate(void) { gate_open = true; }
  bool is_open(void) { return(gate_open); }

public:

  // Functions returning info about effect and its parameters.
  // ---
  void set_parameter(int param, double value) { }

  virtual GATE_BASE* clone(void) = 0;   
  virtual ~GATE_BASE(void) { }

  GATE_BASE(void) { support_for_dynamic_parameters(false); }
};

class TIME_CROP_GATE : public GATE_BASE {
  //
  // A time crop gate. Initially the gate is closed, but is opened after 
  // 'open_at' seconds has elapsed. Gate remains open for 
  // 'duration' seconds. If 'duration' is 0, gate will stay open
  // forever.
  //

public:

  // Functions returning info about effect and its parameters.
  // ---
  double get_parameter(int param) const;

  string label(void) { return("Time crop gate"); }
  virtual string id_string(void) { return("gc"); }
  virtual int number_of_params(void) { return(2); }


  void process(SAMPLE_BUFFER* sbuf);

  TIME_CROP_GATE* clone(void)  { return new TIME_CROP_GATE(*this); }
  TIME_CROP_GATE (double open_at, double duration);

private:

  double curtime, btime, etime; 
};

class THRESHOLD_GATE : public GATE_BASE {
  //
  // A threshold open gate. When the average volume goes 
  // over 'threshold_openlevel', gate is opened. In the 
  // same way, when the average volume drops below 
  // 'threshold_closelevel', gate is closed. Either 
  // peak or RMS level is used for calculating average 
  // volume. The thresholds are given in percents. Unlike
  // noise gates, threshold gate is opened and closed 
  // only once. 
  //

public:

  // Functions returning info about effect and its parameters.
  // ---
  double get_parameter(int param) const;

  string label(void) { return("Threshold gate"); }
  virtual string id_string(void) { return("ge"); }
  virtual int number_of_params(void) { return(3); }


  void process(SAMPLE_BUFFER* sbuf);

  THRESHOLD_GATE* clone(void)  { return new THRESHOLD_GATE(*this); }
  THRESHOLD_GATE (double threshold_openlevel, double
		  threshold_closelevel,  bool use_rms = false);

private:

    double openlevel, closelevel, avolume;
    bool rms;
    bool is_opened, is_closed;
};

#endif









