// ------------------------------------------------------------------------
// eca-session.cpp: Ecasound runtime setup and parameters.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>

#include <string>
#include <cstring>
#include <algorithm>
#include <vector>
#include <pthread.h>

#include <kvutils.h>

#include "eca-resources.h"

#include "chain.h"
#include "audiofx.h"
#include "audioio.h"

#include "error.h"
#include "debug.h"

#include "eca-session.h"
#include "eca-chainsetup.h"

ECA_SESSION::ECA_SESSION(void) {
  set_defaults();
  ecaresources.load();
  active_chainsetup = 0;
}

ECA_SESSION::~ECA_SESSION(void) {
  ecadebug->msg(1,"ECA_SESSION destructor!");
  ecaresources.save();

  for(vector<ECA_CHAINSETUP*>::iterator q = chainsetups.begin(); q != chainsetups.end(); q++) {
    delete *q;
  }

  ecadebug->control_flow("Closing session");
}

ECA_SESSION::ECA_SESSION(COMMAND_LINE& cline) {
  set_defaults();
  ecaresources.load();
  active_chainsetup = 0;

  interpret_general_options(cline);

  if (chainsetups.size() == 0) {
    ECA_CHAINSETUP* comline_setup = new ECA_CHAINSETUP(&ecaresources, cline);
    add_setup(comline_setup);
    set_active_setup(comline_setup->name());
  }

  // These were added for convenience...
  if (cline.size() < 2) {
    // No parameters, let's give some help.
    interpret_general_option("-h");
  }
  if (inputs->size() == 0) {
    // Still no inputs? If not in interactive mode, there really isn't
    // anything left to do.
    if (iactive == false) 
      throw(new ECA_ERROR("ECA_SESSION","Nothing to do!"));
  }
}

void ECA_SESSION::set_defaults(void) {
  //  buffersize = SB_BUFFERSIZE; // set to default buffersize

  if (ecaresources.resource("default-to-interactive-mode")
      == "true") iactive = true;
  else
    iactive = false;

  if (ecaresources.resource("default-to-raisepriority")
      == "true") raisepriority = true;
  else
    raisepriority = false;

  enable_outqueue = false;
  mthreaded_use_of_session = false;
  mthreaded_io = false;
  mixmode = ep_mm_auto;
  multitrack_mode = false;

  loop_active = false;
  loop_start_pos = 0;
  loop_end_pos = 0;

  sfx = true;
  loop_counter = 0;

  pthread_mutex_init(&status_lock, NULL);
  status(ep_status_notready);
}

void ECA_SESSION::init_mutexes(void) {
  ecadebug->msg(1,"(eca-main) Initializing chain mutexes.");
  while(chain_locks.size() != 0) {
    delete chain_locks.back();
    chain_locks.pop_back();
  }
  for(int n = 0; n < chains->size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    chain_locks.push_back(new_mutex);
  }

  ecadebug->msg(1,"(eca-main) Initializing input mutexes.");
  while(in_locks.size() != 0) {
    delete in_locks.back();
    in_locks.pop_back();
  }
  for(int n = 0; n < chains->size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    in_locks.push_back(new_mutex);
  }

  ecadebug->msg(1,"(eca-main) Initializing output mutexes.");
  while(out_locks.size() != 0) {
    delete out_locks.back();
    out_locks.pop_back();
  }
  for(int n = 0; n < outputs->size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    out_locks.push_back(new_mutex);
  }
}

void ECA_SESSION::add_setup(ECA_CHAINSETUP* comline_setup) {
  chainsetups.push_back(comline_setup);
}

void ECA_SESSION::set_active_setup(const string& name) {
  if (active_chainsetup != 0) active_chainsetup->disable();

  vector<ECA_CHAINSETUP*>::const_iterator p = chainsetups.begin();
  while(p != chainsetups.end()) {
    if ((*p)->name() == name) {
      ecadebug->msg("(eca-session) Setting chainsetup \"" + name + "\" active.");
      (*p)->enable();
      if ((*p)->is_valid() == false)
	throw(new ECA_ERROR("ECA-SESSION","Chainsetup not valid, can't set it active."));
      active_chainsetup = *p;
      connect_active_setup();
    }
    ++p;
  }
}

void ECA_SESSION::save_active_setup(const string& name) {
  active_chainsetup->save_to_file(name);
}

void ECA_SESSION::load_active_setup(const string& name) {
  vector<ECA_CHAINSETUP*>::const_iterator p = chainsetups.begin();
  while(p != chainsetups.end()) {
    if ((*p)->name() == name)
      throw(new ECA_ERROR("ECA-SESSION","Chainsetup \"" + name + 
			  "\" already exists.", ECA_ERROR::retry));
    ++p;
  }

  ECA_CHAINSETUP* new_setup = new ECA_CHAINSETUP(&ecaresources, name);
  add_setup(new_setup);
  set_active_setup(new_setup->name());
}

void ECA_SESSION::connect_active_setup(void) {
  inputs = &(active_chainsetup->inputs);
  select_master_input();

  outputs = &(active_chainsetup->outputs);
  chains = &(active_chainsetup->chains);

  init_mutexes();

  buffersize = active_chainsetup->buffersize;
  mixmode = active_chainsetup->mixmode;
  raisepriority = active_chainsetup->raisepriority;

  while(inslots.size() != 0) inslots.pop_back();
  while(inslots.size() != inputs->size()) inslots.push_back(SAMPLE_BUFFER(buffersize));

  while(outslots.size() != 0) outslots.pop_back();
  while(outslots.size() != outputs->size()) outslots.push_back(SAMPLE_BUFFER(buffersize));

}

void ECA_SESSION::select_master_input(void) {
  master_input_length = 0;
  CHAIN::aio_id_type p = 0;
  while (p < inputs->size()) {
    if ((*inputs)[p]->length_in_samples() > master_input_length) {
      master_input_length = (*inputs)[p]->length_in_samples();
      master_input_id = p;
    }
    ++p;
  }

  ecadebug->msg(1, "(eca-session) Setting master_input_id: " +
		kvu_numtostr(master_input_id) + ", length: " +
		kvu_numtostr(master_input_length) + ".");
}

void ECA_SESSION::interpret_general_options(COMMAND_LINE& cline) {
  cline.back_to_start();
  while(cline.ready()) {
    string temp = cline.next_argument();
    interpret_general_option(temp);
  }

 cline.back_to_start();    
 while(cline.ready()) {
   string argu = cline.next_argument();
   string argu_param = cline.next();
   if (argu_param.size() > 0) {
     if (argu_param[0] == '-') {
       cline.previous();
       argu_param == "";
     }
   }

    interpret_chainsetup(argu, argu_param);
  }
}

void ECA_SESSION::interpret_general_option (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECAPARAMS", "empty argument", retry));

  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'c':
    iactive = true;
    ecadebug->msg(0, "(eca-session) Interactive mode enabled."); 
    break;

  case 'd':
    {
      ecadebug->set_debug_level(atoi(get_argument_number(1, argu).c_str()));
      MESSAGE_ITEM mtempd;
      mtempd << "(eca-session) Set debug level to: " << ecadebug->get_debug_level();
      ecadebug->msg(mtempd.to_string());
      break;
    }
  case 'h':      // help!
    cout << ecasound_parameter_help();
    break;

  default: { }
  }
}

void ECA_SESSION::interpret_chainsetup (const string& argu,
					const string& toinen) {
  if (argu.size() == 0) return;
  
  string tname = get_argument_number(1, argu);
  if (tname == "") tname = toinen;
  else if (tname[0] == '-') tname = toinen;
  else tname = argu;

  if (argu.size() < 2) return;
  switch(argu[1]) {
  case 's':
    load_active_setup(tname);
    break;
  }
}

bool ECA_SESSION::is_slave_output(CHAIN::aio_id_type aiod) {
  if ((*outputs)[aiod]->is_realtime()) return(false);
  vector<CHAIN*>::iterator q = chains->begin();
  while(q != chains->end()) {
    if ((*q)->outputid == aiod) {
      if ((*inputs)[(*q)->inputid]->is_realtime()) {
	ecadebug->msg(2,"(eca-session) slave output detected: " + (*outputs)[(*q)->outputid]->label());
	return(true);
      }
    }
    ++q;
  }
  return(false);
}

void ECA_SESSION::status(const EP_STATUS temp) {
  pthread_mutex_lock(&status_lock);
  ep_status = temp;
  pthread_mutex_unlock(&status_lock);
}

EP_STATUS ECA_SESSION::status(void) {
  pthread_mutex_lock(&status_lock);
  EP_STATUS temp = ep_status;
  pthread_mutex_unlock(&status_lock);
  return(temp);
}

long ECA_SESSION::length_in_samples(void) const { 
  if (inputs->size() < master_input_id + 1) return(0);
  return((*inputs)[master_input_id]->length_in_samples()); 
}

double ECA_SESSION::length_in_seconds_exact(void) const { 
  return((double)length_in_samples() / SAMPLE_BUFFER::sample_rate); 
}

long ECA_SESSION::position_in_samples(void) const { 
  if (inputs->size() < master_input_id + 1) return(0);
  return((*inputs)[master_input_id]->position_in_samples()); 
}

double ECA_SESSION::position_in_seconds_exact(void) const {
    return((double)position_in_samples() / SAMPLE_BUFFER::sample_rate); 
}

string ECA_SESSION::ecasound_parameter_help(void) {
  MESSAGE_ITEM laite; 
  laite <<  "USAGE: (qt)ecasound [general options] [chain setup] {inputs] [outputs]";
  laite <<  "\n";
  laite <<  "\n   -c                       set interactive mode";
  laite <<  "\n   -d:debug_level           show debug info";
  laite <<  "\n   -q                       quiet mode, no output";
  laite <<  "\n   -s[:]file                load chainsetup from 'file'";
// ---------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -b:buffersize            size of sample buffer in samples";
  laite <<  "\n   -m:mixmode               mixmode; auto (default), simple, normal,";
  laite <<  "\n                            mthreaded";
  laite <<  "\n   -r                       raise runtime prioritary";
// ---------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -a:name1, name2, ...     set active chains ('all' reserved)";
  laite <<  "\n   -i[:]infile              specify a new infile (assigned to active chains)";
  laite <<  "\n   -o[:]outfile             specify a new outfile (assigned to active chains)";
  laite <<  "\n   -f:bits,channels,srate   file format (for all following inputs/outputs)";
// ---------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -ps:preset_name          insert a single-chain preset";
  laite <<  "\n   -ea:amplify_%, cmax      amplify_% (100% means no change), warns if over";
  laite <<  "\n                            cmax consecutive clipped samples";
  laite <<  "\n   -eas:amplify_%           amplify_% (100% means no change)";
  laite <<  "\n   -ec:c_rate,threshhold    compress (c_rate: -1.0 max. compress, 0.0 no";
  laite <<  "\n                            effect, >0.0 expand; doesn't affect if volume";
  laite <<  "\n                            under threshhold)";
  laite <<  "\n   -ef1:center_freq, width  resonant bandpass filter";
  laite <<  "\n   -ef3:cutoff_freq, reso,  resonant lowpass filter";
  laite <<  "\n        gain";
  laite <<  "\n   -efb:center_freq,width   bandpass filter";
  laite <<  "\n   -efh:cutoff_freq         highpass filter";
  laite <<  "\n   -efi:delay_in_sampl,rad  inverse comb filter";
  laite <<  "\n   -efl:cutoff_freq         lowpass filter";
  laite <<  "\n   -efr:center_freq,width   bandreject filter";
  laite <<  "\n   -efs:center_freq,width   resonator filter";
  laite <<  "\n   -enm:threshold_level_%,  noise gate, mono";
  laite <<  "\n        th_time, attack,    ([0,1], ms, ms, ms, ms)";
  laite <<  "\n        hold, release       ";
  laite <<  "\n   -epp:l_r_balance         normal pan (0 = left, 100 = right)";
  laite <<  "\n   -etd:delay_time,mode,    delay (mode: 0 = normal, ";
  laite <<  "\n        number_of_delays    1 = surround, 2 = stereo-spread)";
  laite <<  "\n        mix_%%              ";
  laite <<  "\n   -etf:delay_time          fake stereo (delay 1-40ms)";
  laite <<  "\n   -etr:reverb_time,mode,   reverb (mode: 0 = normal, ";
  laite <<  "\n       feedback_%%          1 = surround)";
  laite <<  "\n   -ev                      analyze/maximize volume (non-realtime)";
  laite <<  "\n   -ezf                     find optimal value for DC-fix";
  laite <<  "\n   -ezx:left,right          adjust DC";
// ---------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -gc:beg.time, len         time crop gate (gate open for 'len' seconds ";
  laite <<  "\n                             starting from 'beg.time')";
  laite <<  "\n   -ge:othreshold%, cthold%, threshold gate (open when volume goes over";
  laite <<  "\n       volume_mode           'othreshold' and closes when it drops below"; 
  laite <<  "\n                             'cthold'. If 'value_mode' is 'rms', use";
  laite <<  "\n                             average RMS volume, otherwise use peak";
// ---------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -kos:fx_param,low_range,  sine-oscillator";
  laite <<  "\n        high_rng,freq,i_phase  ";
  laite <<  "\n   -kf:fx_param, low_range,  file envelope (generic oscillator)";
  laite <<  "\n       genosc_number       ";
  laite <<  "\n   -km:fx_param, low_range,  MIDI controlled envelope";
  laite <<  "\n       high_rng, controller,";
  laite <<  "\n       channel";
  
  laite <<  "\n\n";

  return(laite.to_string());
}

