// ------------------------------------------------------------------------
// eca-controller.cpp: Class for controlling ECA_SESSION, ECA_PROCESSOR
//                     and other such objects.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <iostream.h>
#include <fstream.h>
#include <string>
#include <vector>
#include <pthread.h>
#include <unistd.h>

#include <kvutils.h>

#include "eca-main.h"
#include "eca-session.h"
#include "eca-controller.h"

#include "error.h"
#include "debug.h"

ECA_CONTROLLER::ECA_CONTROLLER (ECA_SESSION* psession) {
  session = psession;
  show_prompt = false;
}

void ECA_CONTROLLER::command(const string& cmd) {
  vector<string> cmds = string_to_words(cmd);
  vector<string>::const_iterator p = cmds.begin();
  while (p != cmds.end()) {
    if (*p == "") continue;
  
    // ---
    // General
    // ---
    if (*p == "help"
	|| *p == "h"
	|| *p == "?") 
      {
	++p;
	if (p == cmds.end() || *p != "more") {
	  --p;
	  show_controller_help();
	}
	else if (p != cmds.end() || *p == "more") {
	  show_controller_help_more();
	}
	show_prompt = true;
      }
    else if (*p == "exec") {
      start_engine();
      ++p;
      if (p == cmds.end()) return;
      else --p;
    }
    else if (*p == "quit" || *p == "q") {
      quit();
    }
    else if (*p == "chain" || *p == "c") {
      ++p;
      if (p == cmds.end()) continue;
      cqueue.push_back("c");
      cqueue.push_back(*p);
      while(cqueue.cmds_available()) usleep(500);
      print_chain_status();
      show_prompt = true;
    }
    // ---
    // Chainsetups
    // ---
    else if (*p == "load") {
      ++p;
      if (p == cmds.end()) break;
      load_chainsetup(*p);
    }
    else if (*p == "save") {
      ++p;
      if (p == cmds.end()) break;
      save_chainsetup(*p);
    }
    // ---
    // Session status
    // ---
    else if (*p == "status" 
	     || *p == "st"
	     || *p == "u") {
      print_general_status();
    }
    else if (*p == "cstatus" 
	     || *p == "cs"
	     || *p == "a") {
      print_chain_status();
    }
    else if (*p == "estatus" 
	     || *p == "es"
	     || *p == "x") {
      print_effect_status();
    }
    else if (*p == "fstatus" 
	     || *p == "fs"
	     || *p == "l") {
      print_file_status();
    }
    // ---
    // Send to ECA_PROCESSOR
    // ---
    else 
      cqueue.push_back(*p);

    ++p;
  }
  show_prompt = true;
}

bool ECA_CONTROLLER::prompt(void) {
  if (show_prompt == true) {
    show_prompt = false;
    return(true);
  }
  return(false);
}

void ECA_CONTROLLER::load_chainsetup(const string name) {
  ecadebug->msg("(eca-controller) Loading a new chainsetup from file \""
		+ name + "\".");
      
  bool was_running = false;
  if (session->status() == ep_status_running) was_running = true;

  cqueue.push_back("stop");
  while(session->status() == ep_status_running) {
    ecadebug->msg("(eca-controller) Waiting for the processing to stop...");
    usleep(1000);
  }
  try {
    close_engine();
    session->load_active_setup(name);
    start_engine(true);
  }
  catch (ECA_ERROR* e) {
    cerr << "---\n(eca-controller) FAILED: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  if (was_running) cqueue.push_back("start");
}

void ECA_CONTROLLER::save_chainsetup(const string name) {
  try {
    ecadebug->msg("(eca-controller) Saving active chainsetup to file \""
		  + name + "\".");
    session->save_active_setup(name);
  }
  catch (ECA_ERROR* e) {
    cerr << "---\n(eca-controller) FAILED: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
}

void ECA_CONTROLLER::quit(void) {
  close_engine();
  int n = ECA_QUIT;
  throw(n);
}

void ECA_CONTROLLER::start_engine(bool ignore_lock) {
  ifstream fin(ECASOUND_LOCK_FILE);
  if (!fin || ignore_lock)
    start_normal_thread(session, retcode, &th_cqueue);
  else {
    MESSAGE_ITEM mitem;
    mitem << "(eca-controller) Can't execute; processing module already running!" << 'c' << "\n";
    cerr << "\"" << mitem.to_string() << "\"";
  }
}

void ECA_CONTROLLER::close_engine(void) {
  cqueue.push_back("end");
  ifstream fin(ECASOUND_LOCK_FILE);
  while(fin) {
    fin.close();
    ecadebug->msg(1, "(eca-controller) Waiting for the processing thread...");
    usleep(1000);
    fin.open(ECASOUND_LOCK_FILE);
  }
  //  fin.close();
}

void ECA_CONTROLLER::print_general_status(void) {
  ecadebug->control_flow("Controller/General Status");
  MESSAGE_ITEM st_info_string;
  st_info_string << "Buffersize is " << session->buffersize << " samples.\n";
  st_info_string << "Internal sampling rate is " << SAMPLE_BUFFER::sample_rate << " samples per second.\n";
  if (session->raisepriority) st_info_string << "Priority risen.\n";
  else st_info_string << "Realtime-mode disabled.\n";
  if (session->multitrack_mode) st_info_string << "Multitrack-mode enabled.\n";
  else st_info_string << "Multitrack-mode disabled.\n";
  if (session->loop_active)
    st_info_string << "Looping enabled. Starts at "
		   << session->loop_start_pos
		   << " and ends at "
		   << session->loop_end_pos
		   << ".\n";
  else
    st_info_string << "Looping disabled.\n";
  if (session->sfx) st_info_string << "Effect section enabled.";
  else st_info_string << "Effect section disabled.";

  ecadebug->msg(st_info_string.to_string());
}

void ECA_CONTROLLER::print_chain_status(void) {
  ecadebug->control_flow("Controller/Chain Status");
  MESSAGE_ITEM mitem;

  vector<CHAIN*>::const_iterator chain_citer;
  vector<CHAIN_OPERATOR*>::const_iterator chainop_citer;

  for(chain_citer = session->chains->begin(); chain_citer != session->chains->end();) {
    mitem << "Chain \"" << (*chain_citer)->cname() << "\" [";
    if ((*chain_citer)->is_enabled()) mitem << "on] ";
    else mitem << "off] ";
    if (session->sfx) {
      for(chainop_citer = (*chain_citer)->chainops.begin(); chainop_citer != (*chain_citer)->chainops.end();) {
	mitem << "\"" << (*chainop_citer)->label() << "\"";
	++chainop_citer;
	if (chainop_citer != (*chain_citer)->chainops.end()) mitem << " -> ";   
      }
    }
    ++chain_citer;
    if (chain_citer != session->chains->end()) mitem << "\n";
  }
  ecadebug->msg(mitem.to_string());
}

void ECA_CONTROLLER::print_effect_status(void) {
  ecadebug->control_flow("Controller/Effect Status");
  MESSAGE_ITEM mitem;
  string st_info_string;
  vector<CHAIN*>::const_iterator chain_citer;
  vector<CHAIN_OPERATOR*>::const_iterator chainop_citer;

  for(chain_citer = session->chains->begin(); chain_citer != session->chains->end();) {
    for(chainop_citer = (*chain_citer)->chainops.begin(); chainop_citer != (*chain_citer)->chainops.end();) {
      mitem << "FX " << (*chainop_citer)->label();
      mitem << " with params \"" << (*chainop_citer)->params() << "\"";
      mitem << ", chain " << (*chain_citer)->cname() << ":\n";
      st_info_string = "";
      (*chainop_citer)->status(&st_info_string);
      ++chainop_citer;
      mitem << st_info_string;
      //      if (chainop_citer != (*chain_citer)->chainops.end()) mitem << " ";   
    }
    ++chain_citer;
    //    if (chain_citer == session->chains->end()) mitem << "\n";
  }
  if (mitem.to_string() != "") ecadebug->msg(mitem.to_string());
  st_info_string = "";
}

void ECA_CONTROLLER::print_file_status(void) {
  ecadebug->control_flow("Controller/File status");

  string st_info_string;
  vector<AUDIO_IO_DEVICE*>::const_iterator adev_citer;
  vector<AUDIO_IO_DEVICE*>::size_type adev_sizet = 0;

  adev_citer = session->inputs->begin();

  while(adev_citer != session->inputs->end()) {
    st_info_string += "Infile \"";
    st_info_string += (*adev_citer)->label();
    st_info_string += "\", connected to chains \"";
    vector<string> temp = session->get_connected_chains_to_input(adev_sizet);
    vector<string>::const_iterator p = temp.begin();
    while (p != temp.end()) {
      st_info_string += *p; 
      ++p;
      if (p != temp.end())  st_info_string += ",";
    }
    st_info_string += "\": ";
    (*adev_citer)->status(&st_info_string);
    st_info_string += "\n";
    ++adev_sizet;
    ++adev_citer;
  }

  adev_sizet = 0;
  adev_citer = session->outputs->begin();

  while(adev_citer != session->outputs->end()) {
    st_info_string += "Outfile \"";
    st_info_string += (*adev_citer)->label();
    st_info_string += "\", connected to chains \"";
    vector<string> temp = session->get_connected_chains_to_output(adev_sizet);
    vector<string>::const_iterator p = temp.begin();
    while (p != temp.end()) {
      st_info_string += *p; 
      ++p;
      if (p != temp.end())  st_info_string += ",";
    }
    st_info_string += "\": ";
    (*adev_citer)->status(&st_info_string);
    ++adev_sizet;
    ++adev_citer;
    if (adev_sizet < session->outputs->size()) st_info_string += "\n";
  }
  ecadebug->msg(st_info_string);
}

void show_controller_help(void) {
  MESSAGE_ITEM mitem; 

  mitem << "\n-------------------------------------------------------------------";
  mitem << "\n(qt)ecasound - command reference\n";

  mitem << "\n'q' - Quits ecasound";
 
  mitem << "\n'start', 't' - Processing is started (play)";
 
  mitem << "\n'stop', 's' - Stops processing"; 
 
  mitem << "\n'rewind time-in-seconds', 'rw time-in-seconds' - Rewind";
 
  mitem << "\n'forward time-in-seconds', 'fw time-in-seconds' - Forward";
 
  mitem << "\n'setpos time-in-seconds' - Sets the current position to 'time-in-seconds' seconds from the beginning.";
 
  mitem << "\n'status','st','u' - General status info";
 
  mitem << "\n'cstatus','cs','a' - Status info about the effect chains";
 
  mitem << "\n'estatus', 'es','x' - Status info about effects and controller envelopes";
 
  mitem << "\n'fstatus', 'fs','l' - Status info about open files and devices";
 
  mitem << "\n'sfx' - Enable/disable the sound effects section";
 
  mitem << "\n'chain chainname', 'c chainname' - Enable/disable the the chain 'chainname'";
 
  mitem << "\n'help more' - More commands";

  ecadebug->msg(mitem.to_string());
}

void show_controller_help_more(void) {
  MESSAGE_ITEM mitem; 

  mitem << "\n-------------------------------------------------------------------";
  mitem << "\n(qt)ecasound - command reference (more)\n";

  mitem << "\n'exec' - Starts the processing engine. Normally you don't need to use this.";
  mitem << "\n'end' - Exits the processing engine, but doesn't exit the interactive mode. Normally you don't need to use this.";

  mitem << "\n'loop' - Start/stop looping";
 
  mitem << "\n'loop_start seconds_from_start' - Set the start point for looping";
  mitem << "\n'loop_end seconds_from_start' - Set the end point for looping";

  mitem << "\n'load file_name' - Load a chainsetup from file 'file_name' and set it active.";

  mitem << "\n'save file_name' - Save the active chainsetup to file 'file_name'.";

  ecadebug->msg(mitem.to_string());
}

void start_normal_thread(ECA_SESSION* param, int retcode, pthread_t* th_cqueue) {
  retcode = pthread_create(th_cqueue, NULL, start_normal, (void*)param);
  if (retcode != 0)
    throw(new ECA_ERROR("ECA-CONTROLLER", "Unable to create a new thread (start_normal)."));
}

void* start_normal(void* param) {
  ofstream fout(ECASOUND_LOCK_FILE);
  fout << getpid();
  fout.close();
  ecadebug->msg(1,"(eca-controller) Engine-thread pid: " + kvu_numtostr(getpid()));
  start_normal((ECA_SESSION*)param);
  remove(ECASOUND_LOCK_FILE);
}

void start_normal(ECA_SESSION* param) {
  try {
    ECA_PROCESSOR epros (param);
    epros.exec();
  }
  catch(ECA_ERROR* e) {
    cerr << "---\n(eca-controller) ERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "---\n(eca-controller) Caught an unknown exception!\n";
  }
}



