// ------------------------------------------------------------------------
// audioio-oss.cpp: OSS (/dev/dsp) input/output.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>
#ifdef COMPILE_OSS

#include <string>
#include <cstring>
#include <cstdio>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-oss_impl.h"
#include "audioio-oss.h"

#include "error.h"
#include "debug.h"

OSSDEVICE::OSSDEVICE(const string& name, const SIMODE mode, const AIO_PARAMS& fmt, int bsize) 
  :  AUDIO_IO_DEVICE(name, mode, fmt) 
{
    buffersize = bsize;

    //    format(fmt);
    finished(false);

    open_device();

    is_triggered = false;
}

void OSSDEVICE::open_device(void) {
  if (is_open == true) return;
  if (io_mode() == si_read) {
    //        ecadebug->msg("(audioio-oss) Opening device for reading " + label());
    if ((audio_fd = open(label().c_str(), O_RDONLY, 0)) == -1) {
      throw(new ECA_ERROR("ECA-OSS", "unable to open OSS-device to O_RDONLY"));
    }
    
    int enable_bits = ~PCM_ENABLE_INPUT; // This disables recording
    if (ioctl(audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1)
      throw(new ECA_ERROR("ECA-OSS", "OSS-device doesn't support SNDCTL_DSP_SETTRIGGER"));
  }
  else if (io_mode() == si_write) {
    //        ecadebug->msg("(audioio-oss) Opening device for writing " + label());
    
    if ((audio_fd = open(label().c_str(), O_WRONLY, 0)) == -1) {
      // Opening device failed
      perror("(eca-oss)");
      throw(new ECA_ERROR("ECA-OSS", "unable to open OSS-device to O_WRONLY, " + label() + "."));
    }
    
    int enable_bits = ~PCM_ENABLE_OUTPUT; // This disables playback
    if (ioctl(audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1)
            throw(new ECA_ERROR("ECA-OSS", "OSS-device doesn't support SNDCTL_DSP_SETTRIGGER"));
  }
  
  // ---
  // Set fragment size.
  // ---
  
    if (buffersize == 0) 
      throw(new ECA_ERROR("ECA-OSS", "Buffersize is 0!"));
    
    int fragsize, fragtotal = 16;
    unsigned short int fr_size, fr_count = 0x7fff; // 0x7fff = not limited
    
    MESSAGE_ITEM m;
    m << "setting OSS fragment size according to buffersize " << buffersize << ".\n";
    m << "setting OSS fragment size to " << buffersize * format().align << ".";
    ecadebug->msg(1, m.to_string());

    // fr_size == 4  -> the minimum fragment size: 2^4 = 16 bytes
    
    for(fr_size = 4; fragtotal < buffersize * format().align; fr_size++)
        fragtotal = fragtotal * 2;

    fragsize = ((fr_count << 16) | fr_size);
    
    if (ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &fragsize)==-1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS-error SNDCTL_DSP_SETFRAGMENT"));

    MESSAGE_ITEM mi;
    mi << "set OSS fragment size to (2^x) " << fr_size << ".";
    ecadebug->msg(1, mi.to_string());
    
    // ---
    // Select audio format
    // ---

    if (format().bits == 16) {
        int format = AFMT_S16_LE;
        if (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &format)==-1)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_S16_LE"));
        if (format != AFMT_S16_LE)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_S16_LE"));
        ecadebug->msg(1,"(audioio-oss) Opened device in mode AFMT_S16_LE");
    }
    else {
        int form = AFMT_U8;
        if (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &form)==-1)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_U8"));
        if (form != AFMT_U8)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_U8"));
        ecadebug->msg(1, "(audioio-oss) Opened device in mode AFMT_U8");
    }

    // ---
    // Select number of channels
    // ---

    int stereo = format().channels - 1;     /* 0=mono, 1=stereo */
    if (ioctl(audio_fd, SNDCTL_DSP_STEREO, &stereo)==-1)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported STEREO"));

    if (stereo != format().channels - 1)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported STEREO"));        

    // ---
    // Select sample rate
    // ---
    int speed = format().srate;
    if (ioctl(audio_fd, SNDCTL_DSP_SPEED, &speed) == -1)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported SRATE"));
   
    if (speed != (int)format().srate)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported SRATE"));

    AIO_PARAMS stemp = format();
    format(stemp);

    // ---
    // Get fragment size.
    // ---

    if (ioctl(audio_fd, SNDCTL_DSP_GETBLKSIZE, &fragment_size) == -1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS error SNDCTL_DSP_GETBLKSIZE"));
    MESSAGE_ITEM mitem;
    mitem << "OSS set to use fragment size of " << fragment_size << ".";
    ecadebug->msg(1, mitem.to_string());

    is_open = true;
}

void OSSDEVICE::rt_stop(void) {
  if (is_open) close_device();
  ecadebug->msg("(audioio-oss) Audio device \"" + label() + "\" disabled.");
  is_triggered = false;
}

void OSSDEVICE::close_device(void) {
  if (is_open) { 
    is_open = false;
    if (close(audio_fd) == -1) 
      throw(new ECA_ERROR("ECA-OSS", "error while closing OSS device"));
  }
}

void OSSDEVICE::rt_ready(void) {
  ecadebug->msg("(audioio-oss) Audio device \"" + label() + "\" ready.");
  if (is_open == false) {
    open_device();
  }    
}

void OSSDEVICE::rt_activate(void) {
  if (is_triggered == false) {
    ecadebug->msg("(audioio-oss) Audio device \"" + label() + "\" activated.");
    int enable_bits;
    if (io_mode() == si_read) enable_bits = PCM_ENABLE_INPUT;
    else if (io_mode() == si_write) enable_bits = PCM_ENABLE_OUTPUT;
    if (ioctl(audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS-error SNDCTL_DSP_SETTRIGGER"));
    
    is_triggered = true;
  }
}

void OSSDEVICE::get_sample(SAMPLE_BUFFER* t) {
  //  if (t->length_in_samples() * format().align != fragment_size)
  //    throw(new ECA_ERROR("AUDIOIO-OSS", "get_sample(): sample_buffer and oss-buffer-setup have different sizes!", stop));
  //  t->length_in_samples(fragment_size / format().align);

  if (!is_open) throw(new ECA_ERROR("AUDIOIO-OSS","get_sample(): trying to read from a closed device!"));
  if (read(audio_fd,t->iobuf_uchar, format().align * t->length_in_samples()) == -1) {
    if (is_triggered) {
      throw(new ECA_ERROR("AUDIOIO-OSS", "general OSS error: read from audio device"));
    }
  }
  t->iobuf_to_buffer(format().bits, true, format().channels);
  //  t->iobuf_to_buffer(format().bits, SAMPLE_BUFFER::is_system_littleendian, format().channels);
}

void OSSDEVICE::put_sample(SAMPLE_BUFFER* t) {
  //  if (t->length_in_samples() * format().align != fragment_size)
  //    throw(new ECA_ERROR("ECA-OSS", "put_sample(): sample_buffer and oss-buffer-setup have different sizes!", stop));

  if (!is_open) throw(new ECA_ERROR("AUDIOIO-OSS","get_sample(): trying to write to a closed device!"));
  
  //  t->buffer_to_iobuf(format().bits,  SAMPLE_BUFFER::is_system_littleendian,
  //	     format().channels);
  t->buffer_to_iobuf(format().bits, true, format().channels);
  if (write(audio_fd, t->iobuf_uchar, format().align *
	    t->length_in_samples()) == -1)
    if (is_triggered) throw(new ECA_ERROR("ECA-OSS", "general OSS error: write to audio device (8bit)"));
}

OSSDEVICE::~OSSDEVICE(void) { close_device(); }

#endif // COMPILE_OSS
