// ------------------------------------------------------------------------
// eca-text.cpp: Textmode user-interface routines for ecasound.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <iostream.h>
#include <fstream.h>
#include <string>
#include <vector>
#include <algorithm>
#include <unistd.h>
#include <signal.h>

#include <kvutils.h>

#include "eca-session.h"
#include "eca-main.h"
#include "version.h"
#include "debug.h"
#include "textdebug.h"
#include "error.h"
#include "eca-text.h"

ECA_SESSION* global_pointer_to_ecaparams; 

int main(int argc, char *argv[])
{

  ECA_SESSION* ecaparams;
  ecadebug->set_debug_level(0);

  signal(SIGTERM, signal_handler);
  signal(SIGINT, signal_handler);
  signal(SIGQUIT, signal_handler);

  try {
    print_header();
    
    COMMAND_LINE cline = COMMAND_LINE (argc, argv);
    ecaparams = new ECA_SESSION(cline);
    global_pointer_to_ecaparams = ecaparams;  // used only for signal handling! 
    if (ecaparams->iactive) start_iactive(ecaparams);
    else start_normal(ecaparams);
  }
  catch(ECA_ERROR* e) {
    cerr << "\n---\nERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "\n---\nCaught an unknown exception!\n";
  }

  //  ecaparams->~ECA_SESSION();
  ecadebug->flush();
    
  return(0); // everything ok!
}

void signal_handler(int signum) {
  cerr << "Unexpected interrupt... cleaning up.\n";
  delete global_pointer_to_ecaparams;
  //  global_pointer_to_ecaparams->~ECAPARAMS();
  remove(ECASOUND_LOCK_FILE);
  exit(1);
}

void print_header(void) 
{
    cout << "****************************************************************************\n";
    cout << "*                [1mECASound " << ecasound_version << " (C) 1997-1999 Kai Vehmanen[0m              *\n";
    cout << "****************************************************************************\n";
    cout << "\n";
}

void* start_normal(void* param) {
  try {
    ofstream fout(ECASOUND_LOCK_FILE);
    fout << getpid();
    fout.close();
    ecadebug->msg(1,"ECA_PROCESSOR -pid: " + kvu_numtostr(getpid()));
    start_normal((ECA_SESSION*)param);
    remove(ECASOUND_LOCK_FILE);
  }
  catch(ECA_ERROR* e) {
    cerr << "\n---\nERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "\n---\nCaught an unknown exception!\n";
  }
}

void start_normal(ECA_SESSION* param) {
  // Creates and uses ECA_PROCESSOR.

  ECA_PROCESSOR epros (param);
  epros.exec();
}

void start_iactive(ECA_SESSION* param) {
  // Creates a new thread, in which 'start_normal' is run.
  // Asks user for input. Some of the commands are intepreteded here, while 
  // the rest is sent to a COMMAND_QUEUE object.

  int retcode = 0;
  pthread_t th_cqueue;

  start_normal_thread(param, retcode, &th_cqueue);
    
  //    for(ifstream fin(ECASOUND_LOCK_FILE); !fin;) {
  //        fin.close();
  //        sleep(2);
  //        fin.open(ECASOUND_LOCK_FILE);
  //    }
    
  string cmd;
  do {
    if (cmd.size() > 0) {
      vector<string> temp = string_to_words(cmd);
      vector<string>::const_iterator p = temp.begin();
      while (p != temp.end()) {
	cmd = *p;
	if (cmd == "quit" || cmd == "q") {
	  cqueue.push_back("end");
	  break;
	}
	else if (cmd == "exec") {
	  ifstream fin(ECASOUND_LOCK_FILE);
	  if (!fin)
	    start_normal_thread(param, retcode, &th_cqueue);
	  else {
	    MESSAGE_ITEM mitem;
	    mitem << "IA-MODE: Can't execute; processing module already running!" << 'c' << "\n";
	    cerr << "\"" << mitem.to_string() << "\"";
	  }
	}
	cqueue.push_back(cmd);
	//            cout << "\necasound> ";
	cmd = "";
	++p;
      }
      if (cmd == "quit" || cmd == "q") break;
    }
    else {
      cout << "\nCommand 'help' shows the online-help.\n";
      cout << "ecasound> "; 
    }
  }
  while(getline(cin,cmd));
	//  while(cin >> cmd);

    // ---
    // Don't exit if the processing thread hasn't yet
    // deleted the lock file.

  ifstream fin(ECASOUND_LOCK_FILE);
  while(fin) {
    fin.close();
    ecadebug->msg("IA-MODE: Waiting for the processing thread...");
    usleep(1000);
    ifstream fin(ECASOUND_LOCK_FILE);
  }
  fin.close();
  ecadebug->msg("IA-MODE: Exiting...");
}

void start_normal_thread(ECA_SESSION* param, int retcode, pthread_t* th_cqueue) {
  retcode = pthread_create(th_cqueue, NULL, start_normal, (void*)param);
  if (retcode != 0)
    throw(new ECA_ERROR("ECA-TEXT", "Unable to create a new thread (start_normal).", stop));
}
