#ifndef _ECA_SESSION_H
#define _ECA_SESSION_H

#include <vector>
#include <string>
#include <map>
#include <pthread.h>

#include "audioio.h"
#include "chain.h"
#include "chainop.h"
#include "samplebuffer.h"
#include "ctrl_source.h"

#include "eca-resources.h"
#include "eca-session_impl.h"

enum EP_MM_MODE { ep_mm_auto, ep_mm_simple_iactive, ep_mm_simple_passive, ep_mm_normal_iactive, ep_mm_normal_passive, ep_mm_mthreaded_iactive, ep_mm_mthreaded_passive };
enum EP_STATUS { ep_status_running, ep_status_stopped, ep_status_notready };

class ECA_SESSION {

 private:

  ECA_RESOURCES ecaresources;

  // ---
  // Setup data
  // ---
  vector<string> active_chainids;
  int chaincount;
  AIO_PARAMS active_sinfo;
  bool active_sinfo_explicit;

  // ---
  // Status data
  // ---
  pthread_mutex_t status_lock;
  EP_STATUS ep_status;
  map<CHAIN::aio_id_type,vector<string> > chains_assigned_to_idev, chains_assigned_to_odev;
  map<CHAIN::aio_id_type,int> number_of_chains_assigned_to_idev,number_of_chains_assigned_to_odev;
  long master_input_length;
  int master_input_id;

  // ---
  // Setup interpretation
  // ---
  void set_defaults (void);
  void interpret_general_options(COMMAND_LINE& cline);
  void interpret_setup(COMMAND_LINE& cline);
  void interpret_general_option (const string& argu);
  void interpret_mixmode (const string& argu);
  void interpret_setup_option (const string& argu);
  void interpret_audioio_option (const string& argu, const string& argu_param);
  void interpret_chainop_option (const string& argu);
  void process_singlechain_preset(const string& preset_name);

  int get_type_fromext (const string& teksti);
  string get_argument_number(int number, const string& argu);
  vector<string> get_arguments(const string& argu);

  // ---
  // Setup helper functions
  // ---
  AUDIO_IO_DEVICE* new_aio_device(const string& tname, const SIMODE mode, const AIO_PARAMS& format, int buffersize);
  void add_default_chain(void);

 public:

  // ---
  // Setup functions (except chains themselves, all additions are done
  // to all active chains)
  // ---
  void add_new_chains(const vector<string>& newchains);
  void add_chainop(CHAIN_OPERATOR* cotmp);
  void add_gcontroller(CONTROLLER_SOURCE* csrc, int fxparam, double low, double high);
  void add_input(AUDIO_IO_DEVICE* aiod);
  void add_output(AUDIO_IO_DEVICE* aiod);
  void set_active_chains(vector<string> chains) { active_chainids = chains; }

  // ---
  // Control functions
  // ---
  void update_controller_sources(void);
  void status(const EP_STATUS);

  // ---
  // Functions for handling master input
  // ---
  void lock_master_input(void) { pthread_mutex_lock(in_locks[master_input_id]); }
  void release_master_input(void) { pthread_mutex_unlock(in_locks[master_input_id]); }
  AUDIO_IO_DEVICE* master_input(void) { return(inputs[master_input_id]); }

  // ---
  // Status/info functions
  // ---
  string ecasound_parameter_help(void);
  EP_STATUS status(void);

  vector<string> get_connected_chains_to_input(CHAIN::aio_id_type aiod)  { return(chains_assigned_to_idev[aiod]); }
  vector<string> get_connected_chains_to_output(CHAIN::aio_id_type aiod)  { return(chains_assigned_to_odev[aiod]); }
  int number_of_connected_chains_to_input(CHAIN::aio_id_type aiod)  {
    return(number_of_chains_assigned_to_idev[aiod]); }
  int number_of_connected_chains_to_output(CHAIN::aio_id_type aiod)  {
    return(number_of_chains_assigned_to_odev[aiod]); }

  bool is_slave_output(CHAIN::aio_id_type aiod);

  long length_in_samples(void) const;
  double length_in_seconds_exact(void) const;
  long position_in_samples(void) const;
  double position_in_seconds_exact(void) const;

  // ---
  // Public thread locks
  // ---
  vector<pthread_mutex_t*> in_locks;
  vector<pthread_mutex_t*> chain_locks;
  vector<pthread_mutex_t*> out_locks;

  // ---
  // Public data objects
  // ---
  map<int,bool> chain_ready_for_submix;
  vector<SAMPLE_BUFFER> inslots;
  vector<SAMPLE_BUFFER> outslots;
  vector<AUDIO_IO_DEVICE*> inputs;
  vector<AUDIO_IO_DEVICE*> outputs;
  vector<CHAIN*> chains;

  // ---
  // Public status data
  // ---
  int buffersize;        // Buffersize in samples.
  bool iactive;          // Should engine use 'cqueue'?
  bool mthreaded_io;     // Should engine use multithreaded io-routines?
  bool enable_outqueue;  // Should engine use 'outputqueue'?
  bool raisepriority;    // Should engine raise its process priority?
  bool mthreaded_use_of_session;
  bool loop_active;
  int loop_start_pos, loop_end_pos;
  bool sfx;              // Is SFX-section enabled?
  long loop_counter;
  enum EP_MM_MODE mixmode;

  // --
  // Constructors and destructors
  // --
  ECA_SESSION(void);
  ECA_SESSION(COMMAND_LINE& cline);
  ~ECA_SESSION(void);

 private:

  // --
  // Make sure that objects of this class aren't copied
  // --
  ECA_SESSION (const ECA_SESSION& x) { }
  ECA_SESSION& operator=(const ECA_SESSION& x) { }
};

#endif
