// ------------------------------------------------------------------------
// eca-session.cpp: Ecasound runtime setup and parameters.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>

#include <string>
#include <cstring>
#include <algorithm>
#include <vector>
#include <pthread.h>

#include <kvutils.h>

#include "eca-resources.h"

#include "chain.h"
#include "audiofx.h"
#include "audiogate.h"

#include "audioio.h"
#include "audioio-cdr.h"
#include "audioio-wave.h"
#include "audioio-oss.h"
#include "audioio-oss_dma.h"
#include "audioio-ewf.h"
#include "ecasynth.h"
#include "audioio-mp3.h"
#include "audioio-alsa.h"

#include "osc-gen.h"
#include "osc-sine.h"
#include "midi-cc.h"

#include "error.h"
#include "debug.h"
#include "eca-session_impl.h"
#include "eca-session.h"

ECA_SESSION::ECA_SESSION(void) {
  set_defaults();
  ecaresources.load();
}

ECA_SESSION::~ECA_SESSION(void) {
  ecadebug->msg(1,"ECA_SESSION destructor!");

  for(vector<AUDIO_IO_DEVICE*>::iterator q = inputs.begin(); q != inputs.end(); q++) {
    ecadebug->msg("Closing audio device/file \"" + (*q)->label() + "\".");
    delete *q;
  }
  for(vector<AUDIO_IO_DEVICE*>::iterator q = outputs.begin(); q != outputs.end(); q++) {
    ecadebug->msg("Closing audio device/file \"" + (*q)->label() + "\".");
    delete *q;
  }
  ecaresources.save();
}

void ECA_SESSION::set_defaults(void) {
  //  buffersize = SB_BUFFERSIZE; // set to default buffersize
  buffersize = atoi(ecaresources.resource("default-buffersize").c_str());

  if (ecaresources.resource("default-to-interactive-mode")
      == "true") iactive = true;
  else
    iactive = false;

  if (ecaresources.resource("default-to-raisepriority")
      == "true") raisepriority = true;
  else
    raisepriority = false;

  enable_outqueue = false;
  mthreaded_use_of_session = false;
  mthreaded_io = false;
  mixmode = ep_mm_auto;
  chaincount = 0;

  loop_active = false;
  loop_start_pos = 0;
  loop_end_pos = 0;

  sfx = true;
  loop_counter = 0;
  active_sinfo_explicit = false;

  pthread_mutex_init(&status_lock, NULL);
  status(ep_status_notready);
}

ECA_SESSION::ECA_SESSION(COMMAND_LINE& cline) {
  set_defaults();
  ecaresources.load();

  interpret_general_options(cline);
  interpret_setup(cline);

  // These were added for convenience...
  if (cline.size() < 2) {
    // No parameters, let's give some help.
    interpret_general_option("-h");
  }
  if (inputs.size() == 0) {
    // No -i[:] options specified; let's try to be artificially intelligent 
    cline.back_to_start();
    cline.next_non_argument(); // skip the program name
    string temp = cline.next_non_argument();  
    if (temp != "")
      interpret_audioio_option("-i", temp);
  }
  if (inputs.size() > 0 && outputs.size() == 0) {
    // No -o[:] options specified; let's use the default output 
    interpret_audioio_option("-o", ecaresources.resource("default-output"));
  }
  if (inputs.size() == 0) {
    // Still no inputs? If not in interactive mode, there really isn't
    // anything left to do.
    if (iactive == false) 
      throw(new ECA_ERROR("ECA_SESSION","Nothing to do!"));
  }
}

void ECA_SESSION::interpret_general_options(COMMAND_LINE& cline) {
  cline.back_to_start();
  while(cline.ready()) {
    string temp = cline.next_argument();
    interpret_general_option(temp);
    interpret_mixmode(temp);
  }
}

void ECA_SESSION::interpret_setup(COMMAND_LINE& cline) {
  cline.back_to_start();    
    
  while(cline.ready()) {
    string argu = cline.next_argument();
    string argu_param = cline.next();
    if (argu_param[0] == '-') {
      cline.previous();
      argu_param == "";
    }

    interpret_setup_option(argu);
    interpret_audioio_option(argu, argu_param);
    interpret_chainop_option(argu);
  }
}

void ECA_SESSION::interpret_general_option (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECAPARAMS", "empty argument", retry));

  if (argu.size() < 2) return;
  switch(argu[1]) {
  case 'b':
    {
      buffersize =  atoi(get_argument_number(1, argu).c_str());
      MESSAGE_ITEM mitemb;
      mitemb << "Setting buffersize to (samples) " << buffersize << ".";
      ecadebug->msg(0, mitemb.to_string()); 
      break;
    }
  case 'c':
    iactive = true;
    ecadebug->msg(0, "Interactive mode enabled."); 
    break;

  case 'd':
    {
      ecadebug->set_debug_level(atoi(get_argument_number(1, argu).c_str()));
      MESSAGE_ITEM mtempd;
      mtempd << "Set debug level to: " << ecadebug->get_debug_level();
      ecadebug->msg(mtempd.to_string());
      break;
    }
  case 'h':      // help!
    cout << ecasound_parameter_help();
    break;
  case 'r':
    {
      ecadebug->msg("Raised-priority mode enabled.");
      raisepriority = true;
      break;
    }
  default: { }
  }
}

void ECA_SESSION::interpret_mixmode (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECA-TEXT", "empty argument", retry));
  if (argu.size() < 2) return;
  switch(argu[1]) {
  case 'm':      // mixmode
    {
      string temp = get_argument_number(1, argu);
      if (temp == "auto") {
	mixmode = ep_mm_auto;
	ecadebug->msg("Mix-mode is selected automatically.");
      }
      else {
	if (iactive == true) {
	  if (temp == "multi") {
	    ecadebug->msg("Using mix-mode \"multithreaded-interactive\".");
	    mixmode = ep_mm_mthreaded_iactive; 
	  }
	  else if (temp == "simple") {
	    ecadebug->msg("Using mix-mode \"simple-interactive \".");	
	    mixmode = ep_mm_simple_iactive; 
	  }
	  else {
	    ecadebug->msg("Using mix-mode \"normal-interactive \".");
	    mixmode = ep_mm_normal_iactive;
	  }
	}
	else {
	  if (temp == "multi") {
	    ecadebug->msg("Using mix-mode \"multithreaded-passive\".");	
	    mixmode = ep_mm_mthreaded_passive; 
	  }
	  else if (temp == "simple") {
	    ecadebug->msg("Using mix-mode \"simple-passive\".");
	    mixmode = ep_mm_simple_passive; 
	  }
	  else {
	    ecadebug->msg("Using mix-mode \"normal-passive \".");
	    mixmode = ep_mm_normal_passive;
	  }
	}
      }
      break;
    }
  }
}

void ECA_SESSION::interpret_setup_option (const string& argu) {
  if (argu.size() < 2) return;  
  switch(argu[1]) {
  case 'a':
    {
      active_chainids = get_arguments(argu);
      add_new_chains(active_chainids);
      MESSAGE_ITEM mtempa;
      mtempa << "Active chain ids: ";
      for (vector<string>::const_iterator p = active_chainids.begin(); p != active_chainids.end(); p++) {
	mtempa << *p << " ";
      }
      ecadebug->msg(1, mtempa.to_string());
      break;
    }
  case 'f':
    {
      active_sinfo.bits = (unsigned short int) atoi(get_argument_number(1, argu).c_str());
      active_sinfo.channels = (unsigned short int) atoi(get_argument_number(2, argu).c_str());
      active_sinfo.srate = (unsigned short int) atoi(get_argument_number(3, argu).c_str());
      active_sinfo_explicit = true;
      
      MESSAGE_ITEM ftemp;
      ftemp << "Set active format to (bits/channels/srate): ";
      ftemp << (int)active_sinfo.bits << "/" << (int)active_sinfo.channels << "/" << active_sinfo.srate;
      ecadebug->msg(ftemp.to_string());
      break;
    }
  case 'p':
    {
      if (argu.size() < 3) return;  
      switch(argu[2]) {
      case 'm':
	{
	  break;
	}

      case 's': 
	{
	  process_singlechain_preset(get_argument_number(1,argu));
	  break;
	}
	
      default: { }
      }
      break;
    }
  default: { }
  }
}

void ECA_SESSION::process_singlechain_preset(const string&  preset_name) {
  ecadebug->msg(1,"(eca-session) Opening sc-preset file.");
  string filename =
    ecaresources.resource("resource-directory") + "/" + ecaresources.resource("resource-file-single-effect-presets");
  ifstream fin (filename.c_str());

  if (!fin) {
    //    throw(new ECA_ERROR("ECA_SESSION", "Unable to open single-chain preset file: " + filename + "."));
    cerr << "Tried to open: " << filename << ".\n";
    return;
  }
  
  cerr << "a";
  //  ecadebug->msg(1,"(eca-session) Starting to process sc-preset file. Trying to find \"" + preset_name + "\".");
  string sana;
  while(fin >> sana) {
    if (sana.size() > 0 && sana[0] == '#') {
      while(fin.get() != '\n' && fin.eof() == false);
      continue;
    }
    else {
      ecadebug->msg(5, "(eca-session) Next sc-preset is " + sana + ".");
      if (preset_name == sana) {
	ecadebug->msg(5, "(eca-session) Found the right preset!");

	getline(fin,sana);
	vector<string> chainops = string_to_words(sana);
	vector<string>::const_iterator p = chainops.begin();
	while (p != chainops.end()) {
	  ecadebug->msg(5, "(eca-session) Adding chainop " + *p + ".");
	  interpret_chainop_option(*p);
	  ++p;
	}
	break;
      }
      else 
	while(fin.get() != '\n' && fin.eof() == false);
    }
  }

  fin.close();
}

void ECA_SESSION::interpret_audioio_option (const string& argu, const string& argu_param) {
 
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECA-TEXT", "empty argument", retry));
  if (argu.size() == 0) return;
  
  string tname = get_argument_number(1, argu);
  if (tname[0] == '-' || tname == "") tname = argu_param;
  else tname = argu;

  if (argu.size() < 2) return;  
  switch(argu[1]) {
  case 'i':
    {
      ecadebug->control_flow("Main program/Adding a new input");
      AUDIO_IO_DEVICE* aiod;
      if (active_sinfo_explicit == true) {
	aiod = new_aio_device(tname, si_read, active_sinfo, buffersize);
      }
      else {
	AIO_PARAMS temp;
	aiod = new_aio_device(tname, si_read, temp, buffersize);
      }
	
      add_input(aiod);
	
      //      MESSAGE_ITEM itemp;
      //      itemp << "Infile (" << inputs.size() - 1 << ") \"" << inputs.back()->label() << "\".";
      //      ecadebug->msg(itemp.to_string());
      break;
    }

  case 'o':
    {
      ecadebug->control_flow("Main program/Adding a new output");
      AUDIO_IO_DEVICE* aiod;
      if (active_sinfo_explicit == true)
	aiod = new_aio_device(tname, si_write, active_sinfo, buffersize);
      else {
	AIO_PARAMS default_sinfo;
	default_sinfo.bits = 16;
	default_sinfo.srate = 44100;
	default_sinfo.channels = 2;
	default_sinfo.byte_second = 176400;
	aiod = new_aio_device(tname, si_write, default_sinfo, buffersize);
      }
	
      add_output(aiod);
	
      //      MESSAGE_ITEM otemp;
      //      otemp << "Outfile (" << outputs.size() - 1 << ") \"" << outputs.back()->label() << "\".";
      //      ecadebug->msg(otemp.to_string());
      break;
    }
  default: { }
  }
}

int ECA_SESSION::get_type_fromext (const string& teksti) {
  int typevar;

  if (strstr(teksti.c_str(),".wav") != 0) typevar = TYPE_WAVE;
  else if (strstr(teksti.c_str(),".cdr") != 0) typevar = TYPE_CDR;
  else if (strstr(teksti.c_str(),".ewf") != 0) typevar = TYPE_EWF;
  else if (strstr(teksti.c_str(),"/dev/dsp") != 0) typevar = TYPE_OSS;
  else if (strstr(teksti.c_str(),".ess") != 0) typevar = TYPE_ESS;
  else if (strstr(teksti.c_str(),".mp3") != 0) typevar = TYPE_MP3;
  else if (strstr(teksti.c_str(),"alsa") != 0) typevar = TYPE_ALSA;
  else typevar = TYPE_FROM_EXT;

  return(typevar);
}

AUDIO_IO_DEVICE* ECA_SESSION::new_aio_device(const string& argu, const SIMODE mode, const AIO_PARAMS& format, int buffersize) {

  string tname = get_argument_number(1, argu);

  AUDIO_IO_DEVICE* main_file;
  int type = get_type_fromext(tname);
  switch(type) {
  case TYPE_WAVE:
    main_file = new WAVEFILE (tname, mode, format);
    break;
    
  case TYPE_CDR:
    main_file = new CDRFILE (tname, mode, format);
    break;
    
  case TYPE_OSS:
#ifdef COMPILE_OSS
    main_file = new OSSDEVICE (tname, mode, format, buffersize);
#endif
    break;
    
  case TYPE_EWF:
    main_file = new EWFFILE (tname, mode, format);
    break;
    
  case TYPE_ESS:
    main_file = new ECASYNTH (tname, mode, format);
    break;
    
  case TYPE_OSSDMA:
#ifdef COMPILE_OSS
    main_file = new OSSDMA (tname, mode, format, buffersize);
#endif
    break;
    
  case TYPE_MP3:
    main_file = new MP3FILE (tname, mode, format);
    break;

  case TYPE_ALSA:
    {
      int card = atoi(get_argument_number(2, argu).c_str());
      int device = atoi(get_argument_number(3, argu).c_str());

#ifdef COMPILE_ALSA
      main_file = new ALSADEVICE (card, device, mode, format, buffersize);
#endif
      break;
    }

  default:
    throw(new ECA_ERROR("ECA_SESSION", "unknown file format; unable to open file " + tname + ".", stop));
  }
  return(main_file);
}

void ECA_SESSION::interpret_chainop_option (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECA-TEXT", "empty argument", retry));
  CHAIN_OPERATOR* cotmp;
  
  if (argu.size() < 2) return;  
  switch(argu[1]) {
  case 'e':
    ecadebug->control_flow("Main program/Adding a new chain operator");
  if (argu.size() < 3) return;
    switch(argu[2]) {
    case 'a':
      if (argu[3] == ':') {
	cotmp = new EFFECT_AMPLIFY (atof(get_argument_number(1, argu).c_str()),
				    atoi(get_argument_number(2, argu).c_str()));
      }
      else {
	if (argu.size() < 4) return;
	switch(argu[3]) {
	case 's':
	  cotmp = new EFFECT_AMPLIFY_SIMPLE (atof(get_argument_number(1, argu).c_str()));
	  break;
	}
      }
      break;
    case 'c':
      cotmp = new EFFECT_COMPRESS (atof(get_argument_number(1, argu).c_str()),
				   atof(get_argument_number(2, argu).c_str()));
      break;
    case 'f':
      ecadebug->msg(4, "entering filter-fx-init section");
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case '1':
	cotmp = new EFFECT_RESONANT_BANDPASS (atof(get_argument_number(1, argu).c_str()),
					      atof(get_argument_number(2, argu).c_str()));
	break;

      case '3':
	cotmp = new EFFECT_RESONANT_LOWPASS (atof(get_argument_number(1, argu).c_str()),
					     atof(get_argument_number(2, argu).c_str()),
					     atof(get_argument_number(3, argu).c_str()));
	break;
      case 'b':
	cotmp = new EFFECT_BANDPASS (atof(get_argument_number(1, argu).c_str()),
				     atof(get_argument_number(2, argu).c_str()));
	break;
      case 'h':
	cotmp = new EFFECT_HIGHPASS (atof(get_argument_number(1, argu).c_str()));

	break;
      case 'i': 
	cotmp = new EFFECT_INVERSE_COMB_FILTER (atoi(get_argument_number(1, argu).c_str()),
						atof(get_argument_number(2, argu).c_str()));
	break;

      case 'l':
	cotmp = new EFFECT_LOWPASS (atof(get_argument_number(1, argu).c_str()));
	break;

      case 'r':
	cotmp = new EFFECT_BANDREJECT (atof(get_argument_number(1, argu).c_str()),
				       atof(get_argument_number(2, argu).c_str()));
	break;
      case 's':
	cotmp = new EFFECT_RESONATOR (atof(get_argument_number(1, argu).c_str()),
				      atof(get_argument_number(2, argu).c_str()));
	break;
      }
      break;
    case 'n':
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'm':
	cotmp = new EFFECT_NOISEGATE_MONO (atof(get_argument_number(1, argu).c_str()),
					   atof(get_argument_number(2, argu).c_str()),
					   atof(get_argument_number(3, argu).c_str()),
					   atof(get_argument_number(4, argu).c_str()),
					   atof(get_argument_number(5, argu).c_str()));
	break;
      }
      break;
    case 'p':
      ecadebug->msg(4, "entering pan-fx-init section");
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case '3':
	//	  cotmp = new EFFECT_3D_PAN (atof(get_argument_number(1, argu).c_str()));
	break;
      case 'p':
	cotmp = new EFFECT_NORMAL_PAN (atof(get_argument_number(1, argu).c_str()));
	break;
      }
      break;
    case 't':
      ecadebug->msg(4, "entering timebased-fx-init section");
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'd': 	   
	cotmp = new EFFECT_DELAY (atof(get_argument_number(1, argu).c_str()),
				  get_argument_number(2, argu),
				  atoi(get_argument_number(3, argu).c_str()));
	break;
      case 'f':
	cotmp = new EFFECT_FAKE_STEREO (atof(get_argument_number(1, argu).c_str()));
	break;
      case 'r': 
	cotmp = new EFFECT_REVERB (atof(get_argument_number(1, argu).c_str()),
				   get_argument_number(2, argu),
				   atof(get_argument_number(3, argu).c_str()));
	break;
      }
      break;
    case 'v':
      cotmp = new EFFECT_ANALYZE ();
      break;
    case 'z':
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'f':
	cotmp = new EFFECT_DCFIND ();
	break;
	
      case 'x':
	cotmp = new EFFECT_DCFIX (atof(get_argument_number(1, argu).c_str()),
				  atof(get_argument_number(2, argu).c_str()));
	break;
      }
      break;
    default:
      throw(new ECA_ERROR("ECA_SESSION", "unknown effect parametre", stop));
    }
      
    add_chainop(cotmp);
    
    break; 
    // --- effect-section ends
      
  case 'g':
    ecadebug->msg(1, "entering gate-init section.\n");
    ecadebug->control_flow("Main program/Adding chainop gate");
    if (argu.size() < 3) return;
    switch(argu[2]) {
    case 'c':
      cotmp = new TIME_CROP_GATE(atof(get_argument_number(1,
							  argu).c_str()),
				 atof(get_argument_number(2,
							  argu).c_str()));
      break;
    case 't':
      if (get_argument_number(3,argu) == "rms") {
	cotmp = new
	  THRESHOLD_GATE(atof(get_argument_number(1,argu).c_str()),
			 atof(get_argument_number(2, argu).c_str()),
			 true);
      }
      else {
	cotmp = new
	  THRESHOLD_GATE(atof(get_argument_number(1,argu).c_str()),
			 atof(get_argument_number(2, argu).c_str()),
			 false);
      }
      break;
    default:
      throw(new ECA_ERROR("ECA-TEXT", "unknown gate parametre", stop));
    }

    add_chainop(cotmp);
    break;

  case 'k':
    {

    ecadebug->control_flow("Main program/Adding controller source");

    int fpar = atoi(get_argument_number(1, argu).c_str());
    double lowr = atof(get_argument_number(2, argu).c_str());
    double highr = atof(get_argument_number(3, argu).c_str());
    double step = (double)buffersize / SAMPLE_BUFFER::sample_rate;

    if (argu.size() < 3) return;
    switch(argu[2]) {
    case 'a':
      // KTYPE_VOLUME;
      break;

    case 'f':
      {
      //	aparam->active_fx->kenve.back()->type =
      //	KTYPE_FILE_ENVELOPE;
        string genosc_conf = ecaresources.resource("resource-directory") + "/" + ecaresources.resource("resource-file-genosc-envelopes");
	add_gcontroller(new GENERIC_OSCILLATOR(step,
								atof(get_argument_number(4, argu).c_str()), atoi(get_argument_number(5, argu).c_str()), genosc_conf), fpar, lowr, highr);
      break;
      }
    case 'm':
      // KTYPE_MIDI;
	add_gcontroller(new
					 MIDI_CONTROLLER(atoi(get_argument_number(4, argu).c_str()), atoi(get_argument_number(5, argu).c_str())), fpar, lowr, highr);
      break;
    case 'o':
      // KTYPE_OSCILLATOR;
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'q':
	// KTYPE_OSCILLATOR_SQUARE
	break;
      case 's':
	add_gcontroller(new
					 SINE_OSCILLATOR(step, atof(get_argument_number(4, argu).c_str()), atof(get_argument_number(5, argu).c_str())), fpar, lowr, highr); 
	break;
      case 't':
	// KTYPE_OSKILLATOR_TRIANGLE
	break;
      }
      break;
    case 'p':
      // KTYPE_PITCH;
      break;
    default:
      throw(new ECA_ERROR("ECA_SESSION", "invalid kenve parametres", stop));
    }
    break;
    }
    
  default: { }
  }
}

string ECA_SESSION::get_argument_number(int number, const string& argu) {
    int curnro = 1;
    string::const_iterator e;
    string::const_iterator b = find(argu.begin(), argu.end(), ':');

    string target;

    if (b == argu.end()) {
      if (argu.size() > 0) b = argu.begin();
      else {
	MESSAGE_ITEM mtemp;
	mtemp << "hae_arvo() returned argument: \"\".";
	ecadebug->msg(5, mtemp.to_string());
	return("");
      }
    }
    else 
      ++b;

    do {
        e = find(b, argu.end(), ',');

        if (number == curnro) {
            target = string(b, e);
            MESSAGE_ITEM mtemp;
            mtemp << "hae_arvo() returned argument: " << target;
            ecadebug->msg(5, mtemp.to_string());
            break;
        }
        curnro++;

        b = e;
        ++b;
    } while( b < argu.end());

    return(target);
}

vector<string> ECA_SESSION::get_arguments(const string& argu) {
  string::const_iterator b, e;
  vector<string> rvalue;

  for(b = find(argu.begin(), argu.end(), ':'); b != argu.end();) {
    ++b;
    e = find(b, argu.end(), ',');
    string target = string(b, e);
    if (target.size() > 0) {
      rvalue.push_back(target);
      
      MESSAGE_ITEM mtemp;
      mtemp << "get_arguments(), added: " << target;
      ecadebug->msg(5, mtemp.to_string());
    }
    if (e == argu.end()) break;
    b = e;
  }
  return(rvalue);
}

void ECA_SESSION::add_default_chain() {
  chaincount++;
  chains.push_back(new CHAIN(buffersize, chaincount));
  chains.back()->cname("default");
  ecadebug->msg(1,"add_default_chain() ");
  active_chainids.push_back("default");
}

void ECA_SESSION::add_new_chains(const vector<string>& newchains) {
  for(vector<string>::const_iterator p = newchains.begin(); p != newchains.end(); p++) {
    bool exists = false;
    if (*p == "all") continue;
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname()) exists = true;
    }
    if (exists == false) {
      chaincount++;
      //      chains.push_back(new CHAIN(buffersize, chaincount, (*p)));
      //      chains.push_back(new CHAIN(buffersize, chaincount, (*p)));
      chains.push_back(new CHAIN(buffersize, chaincount));
      chains.back()->cname(*p);
      ecadebug->msg(1,"add_new_chains() added chain " + *p);
    }
  }
}

void ECA_SESSION::add_gcontroller(CONTROLLER_SOURCE* csrc, int fxparam, double low, double high) {
  GCONTROLLER* gtmp;
  for(vector<string>::const_iterator a = active_chainids.begin(); a != active_chainids.end(); a++) {
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*a == (*q)->cname() || *a == "all") {
	if ((*q)->chainops.size() == 0) continue;
	if ((*q)->chainops.back()->support_for_dynamic_parameters() == false) {
	  //	  throw(new ECA_ERROR("ECAPARAMS","Chainop type"));
	  ecadebug->msg("WARNING: Chainop type \"" + 
                        (*q)->chainops.back()->label() + 
			"\" doesn't yet support dynamic parameter controlling.");
	}
	gtmp = new GCONTROLLER((*q)->chainops.back(),
			       csrc,
			       (char)fxparam, low, high);
	(*q)->gcontrollers.push_back(gtmp);
      }
    }
  }
}

void ECA_SESSION::add_chainop(CHAIN_OPERATOR* cotmp) {
  if (chains.size() == 0) add_default_chain();
  for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname() || *p == "all") {
	(*q)->chainops.push_back(cotmp);
      }
    }
  }
}

void ECA_SESSION::add_input(AUDIO_IO_DEVICE* aiod) {
  if (chains.size() == 0) add_default_chain();
  inputs.push_back(aiod);
  inslots.push_back(SAMPLE_BUFFER(buffersize));
  
  if (inputs.size() == 1 || aiod->length_in_samples() > master_input_length) {
    master_input_length = aiod->length_in_samples();
    master_input_id = inputs.size() - 1;
    ecadebug->msg(1, "(eca-session) Setting master_input_id: " +
		  kvu_numtostr(master_input_id) + ", length: " +
		  kvu_numtostr(master_input_length) + ".");
  }
  string temp = "(eca-session) Opening file \"" + aiod->label();
  temp += "\" for ";
  if (aiod->io_mode() == si_read) temp += "reading";
  if (aiod->io_mode() == si_write) temp += "writing";
  if (aiod->io_mode() == si_readwrite) temp += "reading and writing";
  temp += ".\n";
  temp += aiod->format_info();
  temp += "\n(eca-session) Assigning file to chains:";
  for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
    if (*p == "all") temp += " all";
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname() || *p == "all") {
	(*q)->inputid = (CHAIN::aio_id_type)(inputs.size() - 1);
	chains_assigned_to_idev[(*q)->inputid].push_back((*q)->cname());
	if (*p == "all")
	  number_of_chains_assigned_to_idev[(*q)->inputid] = (int)chains.size();
	else {
	  number_of_chains_assigned_to_idev[(*q)->inputid] =
	    (int)active_chainids.size();
	  temp += " " + *p;
	}
      }
    }
  }
  ecadebug->msg(temp);
}

void ECA_SESSION::add_output(AUDIO_IO_DEVICE* aiod) {
  if (chains.size() == 0) add_default_chain();
  string temp = "(audio-io) Opening file \"" + aiod->label();
  temp += "\" for ";
  if (aiod->io_mode() == si_read) temp += "reading";
  if (aiod->io_mode() == si_write) temp += "writing";
  if (aiod->io_mode() == si_readwrite) temp += "reading and writing";
  temp += ".\n";
  temp += aiod->format_info();

  outputs.push_back(aiod);
  outslots.push_back(SAMPLE_BUFFER(buffersize));

  temp += "\n(eca-session) Assigning file to chains:";
  for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
    if (*p == "all") temp += " all";
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->cname() || *p == "all") {
	(*q)->outputid = (CHAIN::aio_id_type)(outputs.size() - 1);
	chains_assigned_to_odev[(*q)->outputid].push_back((*q)->cname());
	if (*p == "all")
	  number_of_chains_assigned_to_odev[(*q)->outputid] = (int)chains.size();
	else {
	  number_of_chains_assigned_to_odev[(*q)->outputid] =
	    (int)active_chainids.size();
	  temp += " " + *p;
	}
      }
    }
  }
  ecadebug->msg(temp);
}

bool ECA_SESSION::is_slave_output(CHAIN::aio_id_type aiod) {
  if (outputs[aiod]->is_realtime()) return(false);
  vector<CHAIN*>::iterator q = chains.begin();
  while(q != chains.end()) {
    if ((*q)->outputid == aiod) {
      if (inputs[(*q)->inputid]->is_realtime()) {
	ecadebug->msg(2,"(eca-session) slave output detected: " + outputs[(*q)->outputid]->label());
	return(true);
      }
    }
    ++q;
  }
  return(false);
}

void ECA_SESSION::status(const EP_STATUS temp) {
  pthread_mutex_lock(&status_lock);
  ep_status = temp;
  pthread_mutex_unlock(&status_lock);
}

EP_STATUS ECA_SESSION::status(void) {
  pthread_mutex_lock(&status_lock);
  EP_STATUS temp = ep_status;
  pthread_mutex_unlock(&status_lock);
  return(temp);
}

long ECA_SESSION::length_in_samples(void) const { 
  if (inputs.size() < master_input_id + 1) return(0);
  return(inputs[master_input_id]->length_in_samples()); 
}

double ECA_SESSION::length_in_seconds_exact(void) const { 
  return((double)length_in_samples() / SAMPLE_BUFFER::sample_rate); 
}

long ECA_SESSION::position_in_samples(void) const { 
  if (inputs.size() < master_input_id + 1) return(0);
  return(inputs[master_input_id]->position_in_samples()); 
}

double ECA_SESSION::position_in_seconds_exact(void) const {
    return((double)position_in_samples() / SAMPLE_BUFFER::sample_rate); 
}

string ECA_SESSION::ecasound_parameter_help(void) {
  MESSAGE_ITEM laite; 
//             -------------------------------------------------------------------------------
  laite <<  "USAGE: (qt)ecasound [general options] [chain setup] {inputs] [outputs]";
  laite <<  "\n";
  laite <<  "\n   -b:buffersize            size of sample buffer in samples";
  laite <<  "\n   -c                       set interactive mode";
  laite <<  "\n   -d:debug_level           show debug info";
  laite <<  "\n   -m:mixmode               mixmode; auto (default), simple, normal,";
  laite <<  "\n                            mthreaded";
  laite <<  "\n   -r                       raise runtime prioritary";
//          -------------------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -a:name1, name2, ...     set active chains ('all' reserved)";
  laite <<  "\n   -i[:]infile              specify a new infile (assigned to active chains)";
  laite <<  "\n   -o[:]outfile             specify a new outfile (assigned to active chains)";
  laite <<  "\n   -f:bits,channels,srate   file format (for all following inputs/outputs)";
//          -------------------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -ps:preset_name          insert a single-chain preset";
  laite <<  "\n   -ea:amplify_%, cmax      amplify_% (100% means no change), warns if over";
  laite <<  "\n                            cmax consecutive clipped samples";
  laite <<  "\n   -eas:amplify_%           amplify_% (100% means no change)";
  laite <<  "\n   -ec:c_rate,threshhold    compress (c_rate: -1.0 max. compress, 0.0 no";
  laite <<  "\n                            effect, >0.0 expand; doesn't affect if volume";
  laite <<  "\n                            under threshhold)";
  laite <<  "\n   -ef1:center_freq, width  resonant bandpass filter";
  laite <<  "\n   -ef3:cutoff_freq, reso,  resonant lowpass filter";
  laite <<  "\n        gain";
  laite <<  "\n   -efb:center_freq,width   bandpass filter";
  laite <<  "\n   -efh:cutoff_freq         highpass filter";
  laite <<  "\n   -efi:delay_in_sampl,rad  inverse comb filter";
  laite <<  "\n   -efl:cutoff_freq         lowpass filter";
  laite <<  "\n   -efr:center_freq,width   bandreject filter";
  laite <<  "\n   -efs:center_freq,width   resonator filter";
  laite <<  "\n   -enm:threshold_level_%,  noise gate, mono";
  laite <<  "\n        th_time, attack,    ([0,1], ms, ms, ms, ms)";
  laite <<  "\n        hold, release       ";
  laite <<  "\n   -epp:l_r_balance         normal pan (0 = left, 100 = right)";
  laite <<  "\n   -etd:delay_time,mode,     delay (mode: 'surround', ";
  laite <<  "\n        number_of_delays     'stereo-spread')";
  laite <<  "\n   -etf:delay_time          fake stereo (delay 1-40ms)";
  laite <<  "\n   -etr:reverb_time,mode,   reverb (mode: 'surround')";
  laite <<  "\n       feedback_%%          ";
  laite <<  "\n   -ev                      analyze/maximize volume (non-realtime)";
  laite <<  "\n   -ezf                     find optimal value for DC-fix";
  laite <<  "\n   -ezx:left,right          adjust DC";
//          -------------------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -gc:beg.time, len         time crop gate (gate open for 'len' seconds ";
  laite <<  "\n                             starting from 'beg.time')";
  laite <<  "\n   -ge:othreshold%, cthold%, threshold gate (open when volume goes over";
  laite <<  "\n       volume_mode           'othreshold' and closes when it drops below"; 
  laite <<  "\n                             'cthold'. If 'value_mode' is 'rms', use";
  laite <<  "\n                             average RMS volume, otherwise use peak";
//          -------------------------------------------------------------------------------
  laite <<  "\n";
  laite <<  "\n   -kos:fx_param,low_range,  sine-oscillator";
  laite <<  "\n        high_rng,freq,i_phase  ";
  laite <<  "\n   -kf:fx_param, low_range,  file envelope (generic oscillator)";
  laite <<  "\n       envelope_number       ";
  laite <<  "\n   -km:fx_param, low_range,  MIDI controlled envelope";
  laite <<  "\n       high_rng, controller,";
  laite <<  "\n       channel";
  
  laite <<  "\n\n";
  return(laite.to_string());
}
