// ------------------------------------------------------------------------
// eca-main.cpp: Main processing engine
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <vector>
#include <map>
#include <ctime>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <pthread.h>

#include <kvutils.h>

#include "chain.h"
#include "ecaparams.h"
#include "samplebuffer.h"
#include "chainop.h"
#include "audioio.h"
#include "error.h"
#include "debug.h"
#include "eca-main.h"

COMMAND_QUEUE cqueue;

ECA_PROCESSOR::ECA_PROCESSOR(ECAPARAMS* params) 
{
  ecadebug->control_flow("Main program/Init");
  eparams = params;
  st_info_string = "";
  init_mutexes();
  eparams->status(ep_status_stopped);

  // ---
  // Handle general parameters.
  // ---
  
  if (eparams->raisepriority) {
    getpriority(PRIO_PROCESS, 0);
    if (setpriority(PRIO_PROCESS, 0, -10) == -1)
      ecadebug->msg("Unable to change priority.");
    
    MESSAGE_ITEM mtemp;
    mtemp << "Raised process priority (for PID): " << (int)getpid();
    ecadebug->msg(1, mtemp.to_string());        
  }
 
  init_status_variables();
  init_mix_method();
}

void ECA_PROCESSOR::init_mutexes(void) {
  ecadebug->msg(1,"(eca-main) Initializing chain mutexes.");
  for(int n = 0; n < eparams->chains.size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    eparams->chain_locks.push_back(new_mutex);
  }

  ecadebug->msg(1,"(eca-main) Initializing input mutexes.");
  for(int n = 0; n < eparams->chains.size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    eparams->in_locks.push_back(new_mutex);
  }

  ecadebug->msg(1,"(eca-main) Initializing output mutexes.");
  for(int n = 0; n < eparams->outputs.size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    eparams->out_locks.push_back(new_mutex);
  }
}

void ECA_PROCESSOR::init_status_variables(void) {
  end_request = false;

  // ---
  // Are we doing multitrack-recording?
  // ---    
    
  rt_infiles = rt_outfiles = false;
    
  for (adev_citer = eparams->inputs.begin(); adev_citer != eparams->inputs.end(); adev_citer++) {
    if ((*adev_citer)->is_realtime()) rt_infiles = true;
  }

  for (adev_citer = eparams->outputs.begin(); adev_citer != eparams->outputs.end(); adev_citer++) {
    if ((*adev_citer)->is_realtime()) rt_outfiles = true;
  }

  multitrack_mode = false;
  multitrack_sync_phase = false;
   
  if (rt_infiles == true && rt_outfiles == true &&
      eparams->chains.size() > 1)  {
    ecadebug->msg("Multitrack-mode enabled.");
    multitrack_mode = true;
  }
}


void ECA_PROCESSOR::init_mix_method(void) { 
  if (eparams->mixmode == ep_mm_auto) {
    if (eparams->iactive == true || 
	eparams->mthreaded_use_of_ecaparams == true ||
	rt_infiles == true) {
	if (eparams->chains.size() == 1 &&
	    eparams->inputs.size() == 1 &&
	    eparams->outputs.size() == 1 &&
	    eparams->mthreaded_use_of_ecaparams == false)
	  eparams->mixmode = ep_mm_simple_iactive;
	else if (eparams->buffersize >= 1024)
	  eparams->mixmode = ep_mm_mthreaded_iactive;
	else 
	  eparams->mixmode = ep_mm_normal_iactive;
    }
    else {
	if (eparams->chains.size() == 1 &&
	    eparams->inputs.size() == 1 &&
	    eparams->outputs.size() == 1)
	  eparams->mixmode = ep_mm_simple_passive;
	else if (eparams->buffersize >= 1024) 
	  eparams->mixmode = ep_mm_mthreaded_passive;
	else 
	  eparams->mixmode = ep_mm_normal_passive;
    }
  }
}


void ECA_PROCESSOR::exec(void) {
  switch(eparams->mixmode) {
  case ep_mm_simple_iactive:
    {
      exec_simple_iactive();
      break;
    }
  case ep_mm_simple_passive:
    {
      exec_simple_passive();
      break;
    }
  case ep_mm_normal_iactive:
    {
      exec_normal_iactive();
      break;
    }
  case ep_mm_normal_passive:
    {
      exec_normal_passive();
      break;
    }
  case ep_mm_mthreaded_iactive:
    {
      exec_mthreaded_iactive();
      break;
    }
  case ep_mm_mthreaded_passive:
    {
      exec_mthreaded_passive();
      break;
    }
  default: 
    {
      exec_normal_iactive();
    }
  }
}

void ECA_PROCESSOR::exec_normal_iactive(void) {
  // ---
  // Enable devices (non-multitrack-mode)
  // ---

  stop();
  
  // ---
  // The main processing loop.
  // ---
  
  ecadebug->control_flow("Main program/Starting to process in \"normal iactive\" mode");
  
  while (true) {
    if (finished() == true) {
      if (eparams->loop_active == true) {
	set_position(eparams->loop_start_pos);
	stop();
	start();
      }
      else if (end_request) {
	ecadebug->control_flow("Main program/Exiting");
	break;
      }
      else stop();
    }

    if (eparams->iactive) {
      interprete_cqueue();
      if (end_request) {
	ecadebug->control_flow("Main program/Exiting");
	break;
      }
      if (eparams->status() == ep_status_stopped) continue;
    }

    // ---
    // Get input samples.
    // ---
   
    for(adev_sizet = 0; adev_sizet < eparams->inputs.size(); adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      eparams->inputs[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }

    // ---
    // Mix from eparams->inslots to chains.
    // ---

    for (chain_sizet = 0; chain_sizet < eparams->chains.size(); chain_sizet++)
      pthread_mutex_lock(eparams->chain_locks[chain_sizet]);

    mix_to_chains();
    
    // ---
    // Chainoperator processing phase.
    // ---

    if (eparams->sfx) {
      chain_iter = eparams->chains.begin();
      while(chain_iter != eparams->chains.end()) {
    	(*chain_iter)->chainop_process();
	//	cerr << "C";
    	++chain_iter;
      }
    }
    
    // ---
    // Mix from chains to eparams->outslots.
    // ---
    for(adev_sizet = 0; adev_sizet < eparams->outputs.size(); adev_sizet++)
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    mix_from_chains();

    for (chain_sizet = 0; chain_sizet < eparams->chains.size(); chain_sizet++)
      pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);

    // ---
    // Write to output file(s).
    // ---

    for(adev_sizet = 0; adev_sizet < eparams->outputs.size();
	adev_sizet++) {
      // ---
      // NOTICE! Add support for rt-synch:
      // skip output if: sync_phase + realtime-input +
      // non-realtime-output
      // ---
      eparams->outputs[adev_sizet]->put_sample(&eparams->outslots[adev_sizet]);
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }

    // ---
    // Enable devices (multitrack-mode)
    // ---        
	
    start_synced();
    ++(eparams->loop_counter);
       
  }
}

void ECA_PROCESSOR::exec_normal_passive(void) {
  // ---
  // Enable devices.
  // ---
  start();
  
  ecadebug->control_flow("Main program/Starting to process in \"normal passive\" mode");
  while (finished() == false) {
    for(adev_sizet = 0; adev_sizet < eparams->inputs.size(); adev_sizet++) {
      eparams->inputs[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
    }

    mix_to_chains();

    chain_iter = eparams->chains.begin();
    while(chain_iter != eparams->chains.end()) {
      (*chain_iter)->chainop_process();
      ++chain_iter;
    }
    
    mix_from_chains();

    for(adev_sizet = 0; adev_sizet < eparams->outputs.size(); adev_sizet++) {
      eparams->outputs[adev_sizet]->put_sample(&eparams->outslots[adev_sizet]);
    }
  }
}


void ECA_PROCESSOR::exec_simple_iactive(void) {
  // ---
  // Enable devices (non-multitrack-mode)
  // ---
  stop();  

  ecadebug->control_flow("Main program/Starting to process in \"simple iactive\" mode");
  while (true) {
    if (finished() == true) {
      if (eparams->loop_active == true) {
	set_position(eparams->loop_start_pos);
	stop();
	start();
      }
      else if (end_request) {
	ecadebug->control_flow("Main program/Exiting");
	break;
      }
      else stop();
    }

    interprete_cqueue();
    if (end_request) {
      ecadebug->control_flow("Main program/Exiting");
      break;
    }
    if (eparams->status() == ep_status_stopped) continue;

    pthread_mutex_lock(eparams->in_locks[0]);
    eparams->inputs[0]->get_sample(&eparams->inslots[0]);
    pthread_mutex_unlock(eparams->in_locks[0]);

    if (eparams->sfx) {
	pthread_mutex_lock(eparams->chain_locks[0]);

      eparams->chains[0]->gcontroller_update();
      chainop_citer = eparams->chains[0]->chainops.begin();
      while(chainop_citer != eparams->chains[0]->chainops.end()) {
	(*chainop_citer)->process(&eparams->inslots[0]);
	++chainop_citer;
      }
      eparams->inslots[0].limit_values();
      pthread_mutex_unlock(eparams->chain_locks[0]);
    }
    pthread_mutex_lock(eparams->out_locks[0]);
    eparams->outputs[0]->put_sample(&eparams->inslots[0]);
    pthread_mutex_unlock(eparams->out_locks[0]);
  }
}

void ECA_PROCESSOR::exec_simple_passive(void) {
  // ---
  // Enable devices.
  // ---
  start();
  ecadebug->control_flow("Main program/Starting to process in \"simple passive\" mode");
  while (!finished()) {
    eparams->inputs[0]->get_sample(&eparams->inslots[0]);
    eparams->chains[0]->gcontroller_update();
    chainop_citer = eparams->chains[0]->chainops.begin();
    while(chainop_citer != eparams->chains[0]->chainops.end()) {
      (*chainop_citer)->process(&eparams->inslots[0]);
      ++chainop_citer;
    }
    
    eparams->outputs[0]->put_sample(&eparams->inslots[0]);
  }
}

void ECA_PROCESSOR::print_status(void) {
    MESSAGE_ITEM st_info_string;
    st_info_string << "Buffersize is " << eparams->buffersize << " samples.\n";
    st_info_string << "Internal sampling rate is " << SAMPLE_BUFFER::sample_rate << " samples per second.\n";
    if (eparams->raisepriority) st_info_string << "Priority risen.\n";
    else st_info_string << "Realtime-mode disabled.\n";
    if (multitrack_mode) st_info_string << "Multitrack-mode enabled.\n";
    else st_info_string << "Multitrack-mode disabled.\n";
    if (eparams->loop_active)
        st_info_string << "Looping enabled. Starts at "
            << eparams->loop_start_pos
            << " and ends at "
            << eparams->loop_end_pos
            << ".\n";
    else
        st_info_string << "Looping disabled.";
    ecadebug->msg(st_info_string.to_string());
}

void ECA_PROCESSOR::print_cstatus(void) {
  ecadebug->control_flow("Main program/Chains/Status");
  MESSAGE_ITEM mitem;
  for(chain_citer = eparams->chains.begin(); chain_citer != eparams->chains.end();) {
    mitem << "Chain \"" << (*chain_citer)->cname() << "\" [";
    if ((*chain_citer)->is_enabled()) mitem << "on] ";
    else mitem << "off] ";
    for(chainop_citer = (*chain_citer)->chainops.begin(); chainop_citer != (*chain_citer)->chainops.end();) {
      mitem << "\"" << (*chainop_citer)->label() << "\"";
      ++chainop_citer;
      if (chainop_citer != (*chain_citer)->chainops.end()) mitem << " -> ";   
    }
    ++chain_citer;
    if (chain_citer != eparams->chains.end()) mitem << "\n";
  }
  ecadebug->msg(mitem.to_string());
  st_info_string = "";
}

void ECA_PROCESSOR::print_estatus(void) {
  ecadebug->control_flow("Main program/FX/Status");
  MESSAGE_ITEM mitem;
  for(chain_citer = eparams->chains.begin(); chain_citer != eparams->chains.end();) {
    for(chainop_citer = (*chain_citer)->chainops.begin(); chainop_citer != (*chain_citer)->chainops.end();) {
      mitem << "FX " << (*chainop_citer)->label();
      mitem << " with params \"" << (*chainop_citer)->params() << "\"";
      mitem << ", chain " << (*chain_citer)->cname() << ":\n";
      st_info_string = st_info_string + mitem.to_string();
      (*chainop_citer)->status(&st_info_string);
      ++chainop_citer;
      if (chainop_citer != (*chain_citer)->chainops.end()) mitem << " ";   
    }
    ++chain_citer;
    if (chain_citer != eparams->chains.end()) mitem << "\n";
  }
  if (mitem.to_string() != "") ecadebug->msg(mitem.to_string());
  st_info_string = "";
}

void ECA_PROCESSOR::print_fstatus(void) {
    ecadebug->control_flow("Main program/Files/Status");
    adev_citer = eparams->inputs.begin();
    while(adev_citer != eparams->inputs.end()) {
      st_info_string += "Infile \"";
      st_info_string += (*adev_citer)->label();
      st_info_string += "\", connected to chains \"";
      vector<string> temp = eparams->get_connected_chains(*adev_citer);
      vector<string>::const_iterator p = temp.begin();
      while (p != temp.end()) {
	st_info_string += *p; 
	++p;
	if (p != temp.end())  st_info_string += ",";
      }
      st_info_string += "\": ";
      (*adev_citer)->status(&st_info_string);
      st_info_string += "\n";
      ++adev_citer;
    }
    adev_citer = eparams->outputs.begin();
    while(adev_citer != eparams->outputs.end()) {
      st_info_string += "Outfile \"";
      st_info_string += (*adev_citer)->label();
      st_info_string += "\", connected to chains \"";
      vector<string> temp = eparams->get_connected_chains(*adev_citer);
      vector<string>::const_iterator p = temp.begin();
      while (p != temp.end()) {
	st_info_string += *p; 
	++p;
	if (p != temp.end())  st_info_string += ",";
      }
      st_info_string += "\": ";
      (*adev_citer)->status(&st_info_string);
      ++adev_citer;
      if (adev_citer != eparams->outputs.end()) st_info_string += "\n";
    }
    ecadebug->msg(st_info_string);
    st_info_string = "";
}

void ECA_PROCESSOR::set_position(double seconds) {
  bool was_running;
  if (eparams->status() == ep_status_running) was_running = true;
  else was_running = false;
  stop();

  for (adev_sizet = 0; adev_sizet != eparams->inputs.size(); adev_sizet++) {
    pthread_mutex_lock(eparams->in_locks[adev_sizet]);
    eparams->inputs[adev_sizet]->position_in_seconds(seconds);
    pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
  }

  for (adev_sizet = 0; adev_sizet != eparams->outputs.size(); adev_sizet++) {
    pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    eparams->outputs[adev_sizet]->position_in_seconds(seconds);
    pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
  }
  if (was_running == true) start();
}

void ECA_PROCESSOR::change_position(int seconds) {
  bool was_running;
  if (eparams->status() == ep_status_running) was_running = true;
  else was_running = false;
  stop();

  for (adev_sizet = 0; adev_sizet != eparams->inputs.size(); adev_sizet++) {
    pthread_mutex_lock(eparams->in_locks[adev_sizet]);
    eparams->inputs[adev_sizet]->position_in_seconds(eparams->inputs[adev_sizet]->position_in_seconds()
                                           + seconds);
    pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
  }

  for (adev_sizet = 0; adev_sizet != eparams->outputs.size(); adev_sizet++) {
    pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    eparams->outputs[adev_sizet]->position_in_seconds(eparams->outputs[adev_sizet]->position_in_seconds()
                                           + seconds);
    pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
  }
  if (was_running == true) start();
}

void ECA_PROCESSOR::stop(void) { 
  if (eparams->status() == ep_status_stopped) return;
  ecadebug->msg(2, "(eca-main) stop()");
  for (adev_sizet = 0; adev_sizet != eparams->inputs.size(); adev_sizet++) {
    pthread_mutex_lock(eparams->in_locks[adev_sizet]);
    eparams->inputs[adev_sizet]->rt_stop();
    pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
  }

  for (adev_sizet = 0; adev_sizet != eparams->outputs.size(); adev_sizet++) {
    pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    eparams->outputs[adev_sizet]->rt_stop();
    pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
  }

  eparams->status(ep_status_stopped);
}

void ECA_PROCESSOR::start(void) {
  if (finished() || eparams->status() == ep_status_running) return;
  ecadebug->msg(2, "(eca-main) start()");
  if (multitrack_mode == true) {
    multitrack_sync_phase = true;

    for (adev_sizet = 0; adev_sizet != eparams->outputs.size(); adev_sizet++) {
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
      eparams->outputs[adev_sizet]->rt_ready();
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }
  }
  else {
    for (adev_sizet = 0; adev_sizet != eparams->inputs.size(); adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      eparams->inputs[adev_sizet]->rt_ready();
      eparams->inputs[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }
    for (adev_sizet = 0; adev_sizet != eparams->outputs.size(); adev_sizet++) {
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
      eparams->outputs[adev_sizet]->rt_ready();
      eparams->outputs[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }
  }
  eparams->status(ep_status_running);
}

void ECA_PROCESSOR::start_synced(void) {
  if (multitrack_mode == true && multitrack_sync_phase == true) {
    multitrack_sync_phase = false;
    for (adev_sizet = 0; adev_sizet != eparams->outputs.size(); adev_sizet++) {
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
      eparams->outputs[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }
    for (adev_sizet = 0; adev_sizet != eparams->inputs.size(); adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      eparams->inputs[adev_sizet]->rt_ready();
      eparams->inputs[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }
  }
}

ECA_PROCESSOR::~ECA_PROCESSOR(void) {
  ecadebug->msg(1,"ECA_PROCESSOR destructor!");

  ecadebug->control_flow("Main program/Closing files and devices");

  if (!eparams->iactive)
        print_estatus();

  ecadebug->control_flow("Main program/Exiting");
}
    
void ECA_PROCESSOR::interprete_cqueue(void) {
    while (cqueue.cmds_available() == true) {
        ecadebug->msg(2,"cqueue: cmds available - front " + cqueue.front());
        //
        // Basic control commands.
        //
        if (cqueue.front() == "end" || cqueue.front() == "e") {
            ecadebug->msg(2,"cqueue: end!");
            stop();
            end_request = true;
        }
        else if (cqueue.front() == "start" || cqueue.front() == "t") start();
        else if (cqueue.front() == "stop" || cqueue.front() == "s") stop();
        //
        // Section/chain (en/dis)abling commands.
        //                
        else if (cqueue.front() == "chain" || cqueue.front() == "c") {
            cqueue.pop_front();
            if (cqueue.cmds_available() != true) return;
	    if (is_chain_enabled(cqueue.front())) {
	      disable_chain(cqueue.front());
	      ecadebug->msg("IA-MODE: Chain " + cqueue.front() + " disabled.");
	    }
	    else {
	      enable_chain(cqueue.front());
	      ecadebug->msg("IA-MODE: Chain " + cqueue.front() + " enabled.");
	    }
        }
        //
        // Status requests.
        //
        else if (cqueue.front() == "status" || cqueue.front() == "st") {
	  print_status();
	}
        else if (cqueue.front() == "cstatus" || cqueue.front() == "cs") {
	  print_cstatus();
	}
        else if (cqueue.front() == "estatus" || cqueue.front() == "es") {
	  print_estatus();
	}
        else if (cqueue.front() == "fstatus" || cqueue.front() == "fs") {
	  print_fstatus();
	}
        else if (cqueue.front() == "sfx") {
            if (eparams->sfx == true) {
                eparams->sfx = false;
                ecadebug->msg("IA-MODE: Effect section disabled.");
            }
            else {
                eparams->sfx = true;
                ecadebug->msg("IA-MODE: Effect section enabled.");
            }
        }
        else if (cqueue.front() == "rewind" || cqueue.front() == "rw") {
            cqueue.pop_front();
            if (cqueue.cmds_available() != true) return;
            change_position(-(int)ceil(atof(cqueue.front().c_str())));
            MESSAGE_ITEM mtemp;
            mtemp << "IA-MODE: Rewinding (seconds) " << cqueue.front();
            ecadebug->msg(mtemp.to_string());
        }
        else if (cqueue.front() == "forward" || cqueue.front() == "fw") {
            cqueue.pop_front();
            if (cqueue.cmds_available() != true) return;
            change_position((int)ceil(atof(cqueue.front().c_str())));
            MESSAGE_ITEM mtemp;
            mtemp << "IA-MODE: Forward (seconds) " << cqueue.front();
            ecadebug->msg(mtemp.to_string());                        
        }
        else if (cqueue.front() == "setpos") {
            cqueue.pop_front();
            if (cqueue.cmds_available() != true) return;
	    set_position((int)ceil(atof(cqueue.front().c_str())));
            MESSAGE_ITEM mtemp;
            mtemp << "IA-MODE: Position from beginning (seconds) " << cqueue.front();
            ecadebug->msg(mtemp.to_string());                                    
        }
        //
        // Loop commands.
        //
        else if (cqueue.front() == "loop") {
            ecadebug->msg("IA-MODE: Enabled looping.");
            eparams->loop_active = true;
        }
        else if (cqueue.front() == "loop_start") {
            cqueue.pop_front();
            if (cqueue.cmds_available() != true) return;
            eparams->loop_start_pos = (int)ceil(atof(cqueue.front().c_str()));
            MESSAGE_ITEM mtemp;
            mtemp << "IA-MODE: Looping starts at position " << eparams->loop_start_pos;
            ecadebug->msg(mtemp.to_string());            
        }
        else if (cqueue.front() == "loop_end") {
            cqueue.pop_front();
            if (cqueue.cmds_available() != true) return;
	    eparams->loop_end_pos = (int)ceil(atof(cqueue.front().c_str()));
            MESSAGE_ITEM mtemp;
            mtemp << "IA-MODE: Looping ends at position " << eparams->loop_end_pos;
            ecadebug->msg(mtemp.to_string());
        }
        cqueue.pop_front();
    }
}

bool ECA_PROCESSOR::finished(void) {
  finished_result = true;
  for (adev_citer = eparams->outputs.begin(); adev_citer != eparams->outputs.end(); adev_citer++) {
    if ((*adev_citer)->finished() == true) {
      ecadebug->msg(2, "Finished... outfile.");
      stop();
      return(true);
    }
  }
  
  for (adev_sizet = 0; adev_sizet != eparams->inputs.size(); adev_sizet++) {
    if (!(eparams->inputs[adev_sizet]->finished())) {
      finished_result = false;
      break;
    }
    else finished_result = true;
  }
  
  if (finished_result == true && eparams->iactive == false)
    ecadebug->msg(2, "Finished... through.");
  if (finished_result) stop();
  return(finished_result);
}

void ECA_PROCESSOR::mix_to_chains(void) {
  chain_citer = eparams->chains.begin();
  while (chain_citer != eparams->chains.end()) {
    for(audioslot_sizet = 0; audioslot_sizet < eparams->inslots.size(); audioslot_sizet++) {
      if ((*chain_citer)->inputid == audioslot_sizet) {
	(*chain_citer)->audioslot.operator=(eparams->inslots[audioslot_sizet]);
	// --- for debugging signal flow
	//	cerr << "[1]Mixing from sbuf eparams->inslots[]" << " nro " << eparams->inslots[audioslot_sizet]->nro <<  " " << eparams->inslots[audioslot_sizet]->average_volume() <<".\n";
	//	cerr << "[1]Mixing to sbuf audioslot[c]" << " nro " << eparams->chains[c]->audioslot->nro << " " << eparams->chains[c]->audioslot->average_volume() << ".\n";
	// -----------------------------
      }
    }
    ++chain_citer;
  }
}

void ECA_PROCESSOR::mix_from_chains(void) {
  for(audioslot_sizet = 0; audioslot_sizet < eparams->outslots.size(); audioslot_sizet++) {
    eparams->outslots[audioslot_sizet].make_silent();
    chain_citer = eparams->chains.begin();
    while (chain_citer != eparams->chains.end()) {
      if ((*chain_citer)->outputid == audioslot_sizet) {
	if ((*chain_citer)->is_enabled() == true) {
	  eparams->outslots[audioslot_sizet].add_with_weight((*chain_citer)->audioslot, eparams->number_of_connected_chains(eparams->outputs[audioslot_sizet]));
	}
	// --- for debugging signal flow
	//	cerr << "[2]Mixing from sbuf audioslot[c]" << " nro " << eparams->chains[c]->audioslot->nro <<  " " << eparams->chains[c]->audioslot->average_volume() <<".\n";
	//	cerr << "[2]Mixing to sbuf outslot[]" << " nro " << eparams->outslots[audioslot_sizet]->nro <<  " " << eparams->outslots[audioslot_sizet]->average_volume() << ".\n";
	// ------------------------------
      }
      ++chain_citer;
    }
  }
}

bool ECA_PROCESSOR::is_chain_enabled(const string& cname) {
  for(chain_citer = eparams->chains.begin(); chain_citer != eparams->chains.end(); chain_citer++) {
    if ((*chain_citer)->cname() == cname) {
      return((*chain_citer)->is_enabled());
    }
  }
}

void ECA_PROCESSOR::enable_chain(const string& cname) {
  for(chain_iter = eparams->chains.begin(); chain_iter != eparams->chains.end(); chain_iter++) {
    if ((*chain_iter)->cname() == cname) {
      (*chain_iter)->enable();
      return;
    }
  }
}

void ECA_PROCESSOR::disable_chain(const string& cname) {
  for(chain_iter = eparams->chains.begin(); chain_iter != eparams->chains.end(); chain_iter++) {
    if ((*chain_iter)->cname() == cname) {
      (*chain_iter)->disable();
      return;
    }
  }
}

void ECA_PROCESSOR::exec_mthreaded_iactive(void) {

  ecadebug->control_flow("Main program/Starting to process in \"multithreaded interactive\" mode");
  stop();

  for (chain_sizet = 0; chain_sizet < eparams->chains.size(); chain_sizet++)
    eparams->chain_ready_for_submix[chain_sizet] = false;

  pthread_t chain_thread;
  int submix_pid = pthread_create(&chain_thread, NULL, mthread_process_chains, ((void*)eparams));

  if (submix_pid != 0)
    throw(new ECA_ERROR("ECA-MAIN", "Unable to create a new thread (mthread_process_chains)."));

  unsigned long usleep_count;
  while (true) {
    usleep_count = 5;
    if (finished() == true) {
      if (eparams->loop_active == true) {
	set_position(eparams->loop_start_pos);
	stop();
    	start();
      }
      else if (end_request) {
	ecadebug->control_flow("Main program/Exiting");
	break;
      }
      else stop();
    }

    
    interprete_cqueue();
    if (end_request) {
      ecadebug->control_flow("Main program/Exiting");
      break;
    }
    if (eparams->status() == ep_status_stopped) {
      if (usleep_count < 1000) ++usleep_count;
      usleep(usleep_count);
      continue;
    }

    for(adev_sizet = 0; adev_sizet < eparams->inputs.size(); adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      eparams->inputs[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
    }

    chain_sizet = 0;
    usleep_count = 5;
    while (chain_sizet < eparams->chains.size()) {
      pthread_mutex_lock(eparams->chain_locks[chain_sizet]);
      if (eparams->chain_ready_for_submix[chain_sizet] == true) {
	pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
	//	usleep(20);
	usleep(++usleep_count);
	continue;
      }

      for(audioslot_sizet = 0; audioslot_sizet < eparams->inslots.size(); audioslot_sizet++) {
	if (eparams->chains[chain_sizet]->inputid == audioslot_sizet) {
	  eparams->chains[chain_sizet]->audioslot.operator=(eparams->inslots[audioslot_sizet]);
	}
      }
	
      eparams->chain_ready_for_submix[chain_sizet] = true;
      pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
      ++chain_sizet;
    }

    for(adev_sizet = 0; adev_sizet < eparams->inputs.size(); adev_sizet++) {
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }


    ++(eparams->loop_counter);

    start_synced();
  }
  pthread_cancel(chain_thread);
  pthread_join(chain_thread,NULL);
}

void ECA_PROCESSOR::exec_mthreaded_passive(void) {

  ecadebug->control_flow("Main program/Starting to process in \"multithreaded passive\" mode");
  start();

  for (chain_sizet = 0; chain_sizet < eparams->chains.size(); chain_sizet++)
    eparams->chain_ready_for_submix[chain_sizet] = false;

  pthread_t chain_thread;
  int submix_pid = pthread_create(&chain_thread, NULL, mthread_process_chains, ((void*)eparams));

  if (submix_pid != 0)
    throw(new ECA_ERROR("ECA-MAIN", "Unable to create a new thread (mthread_process_chains)."));

  unsigned long usleep_count;
  while (!finished()) {
    for(adev_sizet = 0; adev_sizet < eparams->inputs.size(); adev_sizet++) {
      eparams->inputs[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
    }

    chain_sizet = 0;
    usleep_count = 5;
    while (chain_sizet < eparams->chains.size()) {
      pthread_mutex_lock(eparams->chain_locks[chain_sizet]);
      if (eparams->chain_ready_for_submix[chain_sizet] == true) {
	pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
	//	usleep(20);
	usleep(++usleep_count);
	continue;
      }

      for(audioslot_sizet = 0; audioslot_sizet < eparams->inslots.size(); audioslot_sizet++) {
	if (eparams->chains[chain_sizet]->inputid == audioslot_sizet) {
	  eparams->chains[chain_sizet]->audioslot.operator=(eparams->inslots[audioslot_sizet]);
	}
      }
      eparams->chain_ready_for_submix[chain_sizet] = true;
      pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
      ++chain_sizet;
    }
  }
  pthread_cancel(chain_thread);
  pthread_join(chain_thread,NULL);
}


void *mthread_process_chains(void* params) {

  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);  // other
  // threads can stop this one
  pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);

  ECAPARAMS* ecaparams = (ECAPARAMS*)params; 
  
  ecadebug->control_flow("Submix-thread ready.");
  ecadebug->msg(1,"Submix-pid: " + kvu_numtostr(getpid()));

  int n;
  unsigned long usleep_count;
  int chaincount  = ecaparams->chains.size();
  vector<CHAIN*>::const_iterator chain_citer;
  vector<SAMPLE_BUFFER>::size_type audioslot_sizet;

  while(true) {
    n = 0;
    usleep_count = 5;
    while (n < chaincount) {
      pthread_mutex_lock(ecaparams->chain_locks[n]);
      if (ecaparams->chain_ready_for_submix[n] == false) {
	pthread_mutex_unlock(ecaparams->chain_locks[n]);
	usleep(++usleep_count);
	continue;
      }

      if (ecaparams->sfx) {
	ecaparams->chains[n]->chainop_process();
      }

      ecaparams->chain_ready_for_submix[n] = false;
      ++n;
    }

    for(audioslot_sizet = 0; audioslot_sizet < ecaparams->outputs.size(); audioslot_sizet++) {
      pthread_mutex_lock(ecaparams->out_locks[audioslot_sizet]);
      ecaparams->outslots[audioslot_sizet].make_silent();
      chain_citer = ecaparams->chains.begin();
      while (chain_citer != ecaparams->chains.end()) {
	if ((*chain_citer)->outputid == audioslot_sizet) {
	  if ((*chain_citer)->is_enabled() == true) {
	    ecaparams->outslots[audioslot_sizet].add_with_weight((*chain_citer)->audioslot, ecaparams->number_of_connected_chains(ecaparams->outputs[audioslot_sizet]));
	  }
	}
	++chain_citer;
      }

      for (n = 0; n < chaincount; n++)
	pthread_mutex_unlock(ecaparams->chain_locks[n]);

      if (ecaparams->status() != ep_status_running) {
	pthread_mutex_unlock(ecaparams->out_locks[audioslot_sizet]);
	break;
      }

      // ---
      // NOTICE! Add support for rt-synch:
      // skip output if: sync_phase + realtime-input +
      // non-realtime-output
      // ---
      ecaparams->outputs[audioslot_sizet]->put_sample(&ecaparams->outslots[audioslot_sizet]);
      pthread_mutex_unlock(ecaparams->out_locks[audioslot_sizet]);
    }
    pthread_testcancel();
  }
}
