// ------------------------------------------------------------------------
// audiofx_filter.cpp: Routines for filter effects.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>

#include <kvutils.h>

#include "audiofx.h"
#include "samplebuffer.h"
#include "debug.h"

EFFECT_BANDPASS::EFFECT_BANDPASS (double centerf, double w) {
    set_parameter(1, centerf);
    set_parameter(2, w);

    init_values();

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "FX:    Bandpass filter enabled, center frequency is ";
    otemp << center << " and width " << width << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_BANDPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    center = value;
    D = 2 * cos(2 * M_PI * center / (double)SAMPLE_BUFFER::sample_rate);
    b[0] = -C * D * a[0];
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2;
    C = 1.0 / tan(M_PI * width / (double)SAMPLE_BUFFER::sample_rate);
    D = 2 * cos(2 * M_PI * center / (double)SAMPLE_BUFFER::sample_rate);
    a[0] = 1.0 / (1.0 + C);
    a[1] = 0.0;
    a[2] = -a[0];
    b[1] = (C - 1.0) * a[0];
    break;
  }
}

double EFFECT_BANDPASS::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
}

EFFECT_BANDREJECT::EFFECT_BANDREJECT (double centerf, double w) {
    set_parameter(1, centerf);
    set_parameter(2, w);

    init_values();

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "FX:    Bandreject filter enabled, center frequency is ";
    otemp << center << " and width " << width << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_BANDREJECT::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    center = value;
    D = 2 * cos(2 * M_PI * center / (double)SAMPLE_BUFFER::sample_rate);
    a[1] = -D * a[0];
    b[0] = a[1];
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2;
    C = tan(M_PI * width / (double)SAMPLE_BUFFER::sample_rate);
    a[0] = 1.0 / (1.0 + C);
    a[1] = -D * a[0];
    a[2] = a[0];
    b[0] = a[1];
    b[1] = (1.0 - C) * a[0];
    break;
  }
}

double EFFECT_BANDREJECT::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
}

void EFFECT_FILTER::process(SAMPLE_BUFFER::sample_type *insample) {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
        outputSample.put_channel(nm, a[0] * insample->get_channel(nm) + a[1]
                   * sin[0].get_channel(nm) + a[2] * sin[1].get_channel(nm)
    		   - b[0] * sout[0].get_channel(nm) - b[1] * sout[1].get_channel(nm));
        sin[1].put_channel(nm, sin[0].get_channel(nm));
        sin[0].put_channel(nm, insample->get_channel(nm));

        sout[1].put_channel(nm, sout[0].get_channel(nm));
        sout[0].put_channel(nm, outputSample.get_channel(nm));
    }
    insample->operator=(outputSample);
}

void EFFECT_FILTER::init_values(void) {
    for(int j = 0; j < 2;j++) {
	    sin[j].sample[SAMPLE_BUFFER::ch_left] = 0.0;
	    sin[j].sample[SAMPLE_BUFFER::ch_right] = 0.0;
	    sout[j].sample[SAMPLE_BUFFER::ch_left] = 0.0;
	    sout[j].sample[SAMPLE_BUFFER::ch_right] = 0.0;
    }
}

EFFECT_HIGHPASS::EFFECT_HIGHPASS (double cutoff) {
    set_parameter(1, cutoff);

    init_values();

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "FX:    Highpass filter enabled, cutoff frequency is ";
    otemp << cutOffFreq << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_HIGHPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    cutOffFreq = value;
    C = tan(M_PI * cutOffFreq / (double)SAMPLE_BUFFER::sample_rate);
    a[0] = 1.0 / (1.0 + sqrt(2.0) * C + C * C);
    a[1] = -2.0 * a[0];
    a[2] = a[0];
    b[0] = 2 * (C * C - 1.0) * a[0];
    b[1] = (1.0 - sqrt(2.0) * C + C * C) * a[0];
    break;
  }
}

double EFFECT_HIGHPASS::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(cutOffFreq);
  }
}

EFFECT_INVERSE_COMB_FILTER::EFFECT_INVERSE_COMB_FILTER (int delay_in_samples, double constant) 
{
  // 
  // delay in number of samples
  // circle radius
  //

  set_parameter(1, (double)delay_in_samples);
  set_parameter(2, constant);
    
  laskuri = 0.0;
    
  buffer = new SINGLE_BUFFER[1];
    
  init_values();

  MESSAGE_ITEM otemp;
  otemp.setprecision(0);
  otemp << "FX:    Inverse comb filter enabled with delay of ";
  otemp << C << " samples and circle radius of " << D << ".";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_INVERSE_COMB_FILTER::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    C = value;
    break;
  case 2: 
    D = value;
    break;
  }
}

double EFFECT_INVERSE_COMB_FILTER::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(C);
  case 2: 
    return(D);
  }
}

void EFFECT_INVERSE_COMB_FILTER::process(SAMPLE_BUFFER::sample_type *insample) {

    buffer[0].push_back(*insample);
    //   buffer.push_back(*insample);
    
    if (laskuri >= C) {
      //        insample->put_left(insample->get_left() - (pow(D, C) * (buffer.front().get_left())));
      //        insample->put_right(insample->get_right() - (pow(D, C) * (buffer.front().get_right())));
        insample->put_left(insample->get_left() - (pow(D, C) * (buffer[0].front().get_left())));
        insample->put_right(insample->get_right() - (pow(D, C) * (buffer[0].front().get_right())));
	buffer[0].pop_front();
    }
    else {
        insample->put_left(insample->get_left() / 2.0 );
        insample->put_right(insample->get_right() / 2.0);
        laskuri++;
    }
}

EFFECT_LOWPASS::EFFECT_LOWPASS (double cutoff) {
    set_parameter(1, cutoff);

    init_values();

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "FX:    Lowpass filter enabled, cutoff frequency is ";
    otemp << cutOffFreq << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_LOWPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    cutOffFreq = value;
    C = 1.0 / tan(M_PI * cutOffFreq / (double)SAMPLE_BUFFER::sample_rate);
    a[0] = 1.0 / (1.0 + sqrt(2.0) * C + C * C);
    a[1] = 2.0 * a[0];
    a[2] = a[0];
    b[0] = 2 * (1.0 - C * C) * a[0];
    b[1] = (1.0 - sqrt(2.0) * C + C * C) * a[0];
    break;
  }
}

double EFFECT_LOWPASS::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(cutOffFreq);
  }
}

EFFECT_RESONANT_BANDPASS::EFFECT_RESONANT_BANDPASS (double centerf, double w) {
    set_parameter(1, centerf);
    set_parameter(2, w);

    for(int n = 0; n < 2; n++) outhist[n].make_silent();

    support_for_dynamic_parameters(true);

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "FX:    Resonant bandpass filter enabled, center frequency is ";
    otemp << center << " and width " << width << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_RESONANT_BANDPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    center = value;
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2.0;
    break;
  }
  R = 1.0 - M_PI * width / (double)SAMPLE_BUFFER::sample_rate;
  c = R * R;
  pole_angle = acos((2.0 * R) / (1.0 + c) * cos(center));
  a = (1.0 - c) * sin(pole_angle);
  b = 2.0 * R * cos(center);
}

double EFFECT_RESONANT_BANDPASS::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
}

void EFFECT_RESONANT_BANDPASS::process(SAMPLE_BUFFER::sample_type *insample) {
    insample->put_left(a * insample->get_left()  + b * outhist[0].get_left() * - c * outhist[1].get_left() - outhist[1].get_left());
    insample->put_right(a * insample->get_right() + b * outhist[0].get_right() - c * outhist[1].get_right() - outhist[1].get_right());
    
    outhist[1] = outhist[0];
    outhist[0] = *insample;
}

EFFECT_RESONANT_LOWPASS::EFFECT_RESONANT_LOWPASS (double co, double res, double g) {

    cutoff = co;
    Q = res;

    gain_orig = gain = g;

    laskuri = 0.0;
    
    pi = 4.0 * atan(1.0);
    
    for(int n = 0; n < 4; n++) outhist[n].make_silent();
    for(int n = 0; n < 2; n++) newhist[n].make_silent();

//
// Setup filter s-domain coefficients
//

    ProtoCoef[0].a0 = 1.0;
    ProtoCoef[0].a1 = 0;
    ProtoCoef[0].a2 = 0;
    ProtoCoef[0].b0 = 1.0;
    ProtoCoef[0].b1 = 0.765367 / Q;      // Divide by resonance or Q
    ProtoCoef[0].b2 = 1.0;

    ProtoCoef[1].a0 = 1.0;
    ProtoCoef[1].a1 = 0;
    ProtoCoef[1].a2 = 0;
    ProtoCoef[1].b0 = 1.0;
    ProtoCoef[1].b1 = 1.847759 / Q;      // Divide by resonance or Q
    ProtoCoef[1].b2 = 1.0;

    szxform(0);
    szxform(1);

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "FX:    Resonant lowpass filter enabled, cutoff frequency is ";
    otemp << cutoff << ", resonance " << Q;
    otemp << " and overall gain of " << gain << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_RESONANT_LOWPASS::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    cutoff = value;
    break;
  case 2: 
    Q = value;
    break;
  case 3: 
    gain = value;
    break;
  }
  refresh_values();
}

double EFFECT_RESONANT_LOWPASS::get_parameter(int param) {
  switch (param) {
  case 1: 
    return(cutoff);
  case 2: 
    return(Q);
  case 3: 
    return(gain);
  }
}

void EFFECT_RESONANT_LOWPASS::refresh_values(void) {
    if (cutoff == 0.0) cutoff = 0.1;
    
    gain = gain_orig;
    
//    ProtoCoef[0].a0 = 1.0;
    ProtoCoef[0].a1 = 0;
    ProtoCoef[0].a2 = 0;
//    ProtoCoef[0].b0 = 1.0;
    ProtoCoef[0].b1 = 0.765367 / Q;      // Divide by resonance or Q
    ProtoCoef[0].b2 = 1.0;

//    ProtoCoef[1].a0 = 1.0;
    ProtoCoef[1].a1 = 0;
    ProtoCoef[1].a2 = 0;
//    ProtoCoef[1].b0 = 1.0;
    ProtoCoef[1].b1 = 1.847759 / Q;      // Divide by resonance or Q
    ProtoCoef[1].b2 = 1.0;

    szxform(0);
    szxform(1);
}

void EFFECT_RESONANT_LOWPASS::szxform(int section) {
    wp = 2.0 * (double)SAMPLE_BUFFER::sample_rate * tan(pi * cutoff / (double)SAMPLE_BUFFER::sample_rate);

    // a0 and b0 are presumed to be 1, so...

    ProtoCoef[section].a2 = ProtoCoef[section].a2 / (wp * wp);
    ProtoCoef[section].a1 = ProtoCoef[section].a1 / wp;

    ProtoCoef[section].b2 = ProtoCoef[section].b2 / (wp * wp);
    ProtoCoef[section].b1 = ProtoCoef[section].b1 / wp;

                 // alpha (Numerator in s-domain)
    ad = 4.0 * ProtoCoef[section].a2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate + 2.0 * ProtoCoef[section].a1
         * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].a0;
                 // beta (Denominator in s-domain)
    bd = 4.0 * ProtoCoef[section].b2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate + 2.0 * ProtoCoef[section].b1
         * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].b0;

                 /* update gain constant for this section */
    gain *= ad/bd;

                 // Denominator
    Coef[section].A = (2.0 * ProtoCoef[section].b0 - 8.0 * ProtoCoef[section].b2
                       * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate) / bd;
                      // beta1
    Coef[section].B = (4.0 * ProtoCoef[section].b2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate - 2.0 * ProtoCoef[section].b1
                       * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].b0) / bd;
                      // beta2

                 // Nominator
    Coef[section].C = (2.0 * ProtoCoef[section].a0 - 8.0 * ProtoCoef[section].a2
                       * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate) / ad;
                      // alpha1
    Coef[section].D = (4.0 * ProtoCoef[section].a2 * (double)SAMPLE_BUFFER::sample_rate * (double)SAMPLE_BUFFER::sample_rate - 2.0
                       * ProtoCoef[section].a1 * (double)SAMPLE_BUFFER::sample_rate + ProtoCoef[section].a0) / ad;
                      // alpha2

}

void EFFECT_RESONANT_LOWPASS::process(SAMPLE_BUFFER::sample_type *insample) {

    insample->put_left(insample->get_left() * gain);
    insample->put_right(insample->get_right() * gain);

    // first section:
    // --------------
    
    // poles:
    insample->put_left(insample->get_left() - outhist[0].get_left() * Coef[0].A);
    insample->put_right(insample->get_right() - outhist[0].get_right() * Coef[0].A);

    newhist[0].put_left(insample->get_left() - outhist[1].get_left() * Coef[0].B);
    newhist[0].put_right(insample->get_right() - outhist[1].get_right() * Coef[0].B);
        
    // zeros:
    insample->put_left(newhist[0].get_left() + outhist[0].get_left() * Coef[0].C);
    insample->put_right(newhist[0].get_right() + outhist[0].get_right() * Coef[0].C);
    
    insample->put_left(insample->get_left() + outhist[1].get_left() * Coef[0].D);
    insample->put_right(insample->get_right() + outhist[1].get_right() * Coef[0].D);
        
    outhist[1] = outhist[0];
    outhist[0] = newhist[0];
        
    // second section:
    // --------------
    
    // poles:
    insample->put_left(insample->get_left() - outhist[2].get_left() * Coef[1].A);
    insample->put_right(insample->get_right() - outhist[2].get_right() * Coef[1].A);

    newhist[1].put_left(insample->get_left() - outhist[3].get_left() * Coef[1].B);
    newhist[1].put_right(insample->get_right() - outhist[3].get_right() * Coef[1].B);
        
    // zeros:
    insample->put_left(newhist[1].get_left() + outhist[2].get_left() * Coef[1].C);
    insample->put_right(newhist[1].get_right() + outhist[2].get_right() * Coef[1].C);
    
    insample->put_left(insample->get_left() + outhist[3].get_left() * Coef[1].D);
    insample->put_right(insample->get_right() + outhist[3].get_right() * Coef[1].D);
        
    outhist[3] = outhist[2];
    outhist[2] = newhist[1];
}


EFFECT_RESONATOR::EFFECT_RESONATOR (double centerf, double w) {
    set_parameter(1, centerf);
    set_parameter(2, w);

    init_values();

    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "FX:    Resonator filter enabled, center frequency is ";
    otemp << center << " and width " << width << ".";
    ecadebug->msg(otemp.to_string());
}

void EFFECT_RESONATOR::set_parameter(int param, double value) {
  switch (param) {
  case 1: 
    center = value;
    break;
  case 2: 
    if (value != 0) width = value;
    else width = center / 2;
    break;
  }
  b[1] = exp(-(2 * M_PI) * (width / (double)SAMPLE_BUFFER::sample_rate));
  b[0] = (-4.0 * b[1]) / (1.0 + b[1]) * cos(2 * M_PI * (center / (double)SAMPLE_BUFFER::sample_rate));
  a[0] = (1.0 - b[1]) * sqrt(1.0 - (b[0] * b[0]) / (4.0 * b[1]));
}

double EFFECT_RESONATOR::get_parameter(int param) { 
  switch (param) {
  case 1: 
    return(center);
  case 2: 
    return(width);
  }
}

void EFFECT_RESONATOR::process(SAMPLE_BUFFER::sample_type *insample) {

    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
        temp.put_channel(nm, a[0] * insample->get_channel(nm)
    		   - b[0] * sout[0].get_channel(nm) - b[1] * sout[1].get_channel(nm));

        sout[1].put_channel(nm, sout[0].get_channel(nm));
        sout[0].put_channel(nm, temp.get_channel(nm));

    }
    *insample = temp;
}

EFFECT_SIMPLE_FILTER::EFFECT_SIMPLE_FILTER (double cnst) {
  C = cnst;
  support_for_dynamic_parameters(false);
  
  MESSAGE_ITEM otemp;
  otemp.setprecision(0);
  otemp << "FX:    Simple lowpass filter enabled, constant value is ";
  otemp << C << ".";
  ecadebug->msg(otemp.to_string());
}

void EFFECT_SIMPLE_FILTER::process(SAMPLE_BUFFER::sample_type *insample) {
  
  outputSample.put_left((insample->get_left() - a[0] + b[0]) / 3.0);
  outputSample.put_right((insample->get_right() - a[1] + b[1]) / 3.0);
  
  b[0] = a[0];
  b[1] = a[1];
  
  a[0] = insample->get_left();
  a[1] = insample->get_right();
  
  insample->operator=(outputSample);
}

