#ifndef _AUDIOIO_H
#define _AUDIOIO_H

#include <vector>
#include <stdio.h>
#include <math.h>
#include <string>

#include <kvutils.h>

#include "samplebuffer.h"
#include "debug.h"

enum SIMODE { si_read, si_write, si_readwrite };

class AIO_PARAMS {
 public:

  char channels;
  long int srate;
  long int byte_second;         // bytes per seconds
  short int align;              // the size of one sample in bytes
  char bits;

  AIO_PARAMS (void) { 
    channels = 2;
    srate = 44100;
    bits = 16;
    byte_second = srate * channels * bits / 8;
    align = channels * bits / 8;
  }
};

class AUDIO_IO_DEVICE {
  //
  // Abstract class representing sample inteAIO_PARAMSaces (files, audio devices,
  // sound producing program modules, etc.)
  //

 private:

  AIO_PARAMS sf_fmt;
  SIMODE si_mode; 
  bool fin;
  string id_label;
    
 public:

  // --
  // IO-routines (remember SIMODE);
  // --

  virtual void put_sample(SAMPLE_BUFFER* t) = 0;
  virtual void get_sample(SAMPLE_BUFFER* t) = 0;
    
  // --
  // Device position/state (virtual)
  // --

  virtual void first(void) = 0;
  virtual void last(void) = 0;
  virtual void position_in_samples(long pos) = 0;
  virtual long position_in_samples(void) const = 0;
  virtual long length_in_samples(void) const = 0;
  virtual void status(string* output);

  // --
  // Device position/state (non-virtual)
  // --

  int length_in_seconds(void) const { return((int)ceil((double)length_in_samples() / (double)format().srate));  }
  double length_in_seconds_exact(void) const { return((double)length_in_samples() / (double)format().srate);  }
  int position_in_seconds(void) const { return((int)ceil((double)position_in_samples() / (double)format().srate)); }
  double position_in_seconds_exact(void) const { return((double)position_in_samples() / (double)format().srate); }
  void position_in_seconds(int pos_in_seconds) { position_in_seconds((double)pos_in_seconds); }
  void position_in_seconds(double pos_in_seconds) { position_in_samples(pos_in_seconds * format().srate);  }

  string format_info(void);
  inline SIMODE io_mode(void) { return(si_mode); }    
  inline const AIO_PARAMS& format(void) const { return(sf_fmt); }
  const string& label(void) const { return(id_label); }
  inline bool finished(void) const { return(fin); }
      // If mode 'si_read', finished() means that end of stream has been reached
      // If mode 'si_write', finished() means an error.
  
  // --
  // Realtime related functions
  // --
  
  virtual bool is_realtime(void) const = 0;
  virtual void rt_stop(void) { }   // Device will not be used in a while (i/o not allowed)
  virtual void rt_ready(void) { }  // Makes device ready for use (i/o allowed; useful for
                                   // preloading data)
  virtual void rt_activate(void) { }  // Activates the device.

  virtual ~AUDIO_IO_DEVICE(void) { }
  AUDIO_IO_DEVICE(const string& name, const SIMODE mode, const AIO_PARAMS& fmt) {
    label(name);
    io_mode(mode);
    format(fmt);
  }
  AUDIO_IO_DEVICE(void) {
    label("unknown");
    io_mode(si_read);
    format(AIO_PARAMS()); // use default params
  }
    
 protected:

  virtual AUDIO_IO_DEVICE* clone(void) = 0;

  // --
  // Device info (non-virtual, protected)
  // --

  void io_mode(SIMODE newmode) { si_mode = newmode; }
  void format(const AIO_PARAMS& sffmt) { sf_fmt = sffmt; fix_format(sf_fmt); }
  void label(const string& newlabel) { id_label = newlabel; }
  void finished(bool value) { fin = value; }

 private:

  void fix_format(AIO_PARAMS& format) {
    format.byte_second = format.channels * format.bits * format.srate / 8;
    format.align = format.channels * format.bits / 8;
  }
};

class NULLFILE : public AUDIO_IO_DEVICE {
  //
  // Null AUDIO_IO_DEVICE derivative;
  //
 public:

  void   get_sample(SAMPLE_BUFFER* t) { }
  void   put_sample(SAMPLE_BUFFER* t) { }
    
  void first(void) { }
  void last(void) { }

  bool is_realtime(void) const { return(false); }

  long   length_in_samples(void) const { return(0); }
  long   position_in_samples(void) const { return(0); }
  void   position_in_samples(long pos) { }
    
  NULLFILE(void) { }
  ~NULLFILE(void) { }
  NULLFILE* new_expr(void) { return new NULLFILE(); }
  NULLFILE* clone(void) { return new NULLFILE(*this); }
};

#endif






