#ifndef _AUDIOFX_FILTER_H
#define _AUDIOFX_FILTER_H

#include <vector>
// #include <deque>

#include <kvutils.h>

#include "audiofx.h"
#include "samplebuffer.h"

class EFFECT_FILTER : public AUDIOFX_BASE {
  //
  // Base class for filter effects.
  //
private:
    int nm;
    SAMPLE_BUFFER::sample_type outputSample;
    
public:

  // Functions returning info about effect and its parameters.
  // ---

  virtual void set_parameter(int param, double value) { }
  virtual double get_parameter(int param) { }

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_FILTER(void) { support_for_dynamic_parameters(true); }

protected:

  void init_values(void);

protected:
  
  SAMPLE_BUFFER::sample_type sin[2];
  SAMPLE_BUFFER::sample_type sout[2];
  
  // Temporary variables needed by derived classes:
  double C;
  double D;
  
  SAMPLE_BUFFER::single_type a[3];
  SAMPLE_BUFFER::single_type b[2];    

};

class EFFECT_BANDPASS: public EFFECT_FILTER {
  //
  // A bandpass filter.
  //

private:
    
  double center;
  double width;

public:

  string label(void) { return("Bandpass filter"); }
  string params(void) { return("center-freq, width"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_BANDPASS::EFFECT_BANDPASS (double centerf, double width);
};

class EFFECT_BANDREJECT: public EFFECT_FILTER {
  //
  // Esitt bandreject -filtteri.
  //

private:
    
  double center;
  double width;

public:

  string label(void) { return("Bandreject filter"); }
  string params(void) { return("center-freq, width"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_BANDREJECT::EFFECT_BANDREJECT (double centerf, double width);

};


class EFFECT_HIGHPASS : public EFFECT_FILTER {
  //
  // A highpass filter.
  //
  
 private:

    double cutOffFreq;
    
public:
    
  string label(void) { return("Highpass filter"); }
  string params(void) { return("cutoff-freq"); }
  int number_of_params(void) { return(1); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_HIGHPASS::EFFECT_HIGHPASS (double cutoff);

};

class EFFECT_INVERSE_COMB_FILTER: public EFFECT_FILTER {
  //
  // Inverse comb filter. 
  //
  // ---
  // The basic theory behind this can be found from Ken Steiglitz's book 
  // "A digital signal processing primer", page 77.
  // ---

  //  typedef RING_BUFFER<SAMPLE_BUFFER::sample_type> FSINGLE_BUFFER;
  typedef deque<SAMPLE_BUFFER::sample_type> FSINGLE_BUFFER;
  //  typedef vector<SAMPLE_BUFFER::sample_type> SSINGLE_BUFFER;
  FSINGLE_BUFFER *buffer;
  SAMPLE_BUFFER::sample_type outputSample;
  double laskuri;
  SAMPLE_BUFFER::sample_type temp;

public:

  string label(void) { return("Inverse comb filter"); }
  string params(void) { return("delay/radius"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_INVERSE_COMB_FILTER::EFFECT_INVERSE_COMB_FILTER (int delay_in_samples, double constant);
};

class EFFECT_LOWPASS: public EFFECT_FILTER {
  //
  // A lowpass filter.
  //
  // ---
  //   Algorithm:  1nd order filter.
  //   From Fugue source code:
  //  
  //    output[N] = input[N] * A + input[N-1] * B
  //  
  //    A = 2.0 * pi * center
  //    B = exp(-A / frequency)
  //  
  // ---

private:

  double cutOffFreq;

public:

  string label(void) { return("Lowpass filter"); }
  string params(void) { return("cutoff-freq"); }
  int number_of_params(void) { return(1); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_LOWPASS::EFFECT_LOWPASS (double cutoff);
};

class EFFECT_RESONANT_BANDPASS: public AUDIOFX_BASE {
//
// A resonant bandpass filter. 
//

private:

  SAMPLE_BUFFER::sample_type outhist[2];
  
  double center;
  double width;
  
  double a, b, c, R;
  double pole_angle;

public:

  string label(void) { return("Resonant bandpass filter"); }
  string params(void) { return("center-freq, width"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_BANDPASS::EFFECT_RESONANT_BANDPASS (double centerf, double width);
};

class EFFECT_RESONANT_LOWPASS: public EFFECT_FILTER {
  //
  // A resonant lowpass filter.
  //
  // ---
  // Algorithm is based on a sample filter-routine (iir_filter) posted to comp.dsp.
  // ---
    
  SAMPLE_BUFFER::sample_type outhist[4];
  SAMPLE_BUFFER::sample_type newhist[2];
    
  typedef struct {
    double a0, a1, a2;       // numerator coefficients
    double b0, b1, b2;       // denominator coefficients
  } BIQUAD;

  typedef struct {
    double A, B, C, D;       // filter coefficients
  } FILTER_COEF;
    
  BIQUAD ProtoCoef[2];         // Filter prototype coefficients,
                               // for each filter section
  FILTER_COEF Coef[2];
    
  double cutoff, Q, gain, gain_orig;
  double pi;
  double laskuri;

  double ad, bd, wp;      // for szxform()

  void szxform(int section);
  void refresh_values(void);

public:

  string label(void) { return("Lowpass filter"); }
  string params(void) { return("cutoff-freq, Q, gain"); }
  int number_of_params(void) { return(3); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_LOWPASS::EFFECT_RESONANT_LOWPASS (double cutoff, double resonance, double gain);
};

class EFFECT_RESONATOR: public EFFECT_FILTER {
  //
  // A resonating bandpass filter.
  //
  // ---
  // Based on a second order all-pole (IIR) band-pass filter from SPKit 
  // (for more info, see: http://www.music.helsinki.fi/research/spkit)
  // --

private:

  int nm; // temporary value 
  SAMPLE_BUFFER::sample_type temp;        // temporary values
    
  double center;
  double width;

public:

  string label(void) { return("Resonator filter"); }
  string params(void) { return("center-freq, width"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONATOR::EFFECT_RESONATOR (double center, double width);
};

class EFFECT_SIMPLE_FILTER : public EFFECT_FILTER {
  //
  // Simple filter.
  //
  // ---
  // The basic theory behind this can be found from Ken Steiglitz' book 
  // "A digital signal processing primer", pages 66 and 73.
  // ---

   SAMPLE_BUFFER::sample_type outputSample;    

 public:

   string label(void) { return("Simple filter"); }
   string params(void) { return("constant"); }
   int number_of_params(void) { return(1); }

   void set_parameter(int param, double value) { }
   double get_parameter(int param) { }

   void process(SAMPLE_BUFFER::sample_type *insample);

   EFFECT_SIMPLE_FILTER::EFFECT_SIMPLE_FILTER (double cnst);
};

#endif

