// ------------------------------------------------------------------------
// audioio-oss_dma.cpp: OSS (/dev/dsp) input/output using direct DMA-access
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <cstring>
#include <errno.h>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-oss_dma.h"

#include "error.h"
#include "debug.h"

OSSDMA::OSSDMA(const string& name, const SIMODE mode, const AIO_PARAMS& fmt, int bsize) 
  :  AUDIO_IO_DEVICE(name, mode, fmt)
{
    buffersize = bsize;

    format(fmt);
    finished(false);

    open_device();

    is_open = true;
    is_triggered = false;
}

void OSSDMA::open_device(void) {
    if (io_mode() == si_read) {
        ecadebug->msg("(dsp_dma) Opening device for reading " + label());
        if ((audio_fd = open(label().c_str(), O_RDONLY, 0)) == -1) {
            throw(new ECA_ERROR("ECA-OSS", "unable to open OSS-device to O_RDONLY", stop));
        }
        
        int enable_bits = ~PCM_ENABLE_INPUT; // This disables recording
        if (ioctl(audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1)
            throw(new ECA_ERROR("ECA-OSS", "OSS-device doesn't support SNDCTL_DSP_SETTRIGGER", stop));
    }
    else if (io_mode() == si_write) {
        ecadebug->msg("(dsp_dma) Opening device for writing " + label());

        if ((audio_fd = open(label().c_str(), O_WRONLY, 0)) == -1) {
//        if ((audio_fd = open(label().c_str(), O_RDWR, 0)) == -1) {
            // Opening device failed
            throw(new ECA_ERROR("ECA-OSS", "unable to open OSS-device to O_WRONLY", stop));
        }

        int enable_bits = ~PCM_ENABLE_OUTPUT; // This disables playback
        if (ioctl(audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1)
            throw(new ECA_ERROR("ECA-OSS", "OSS-device doesn't support SNDCTL_DSP_SETTRIGGER", stop));
    }

    // ---
    // Set fragment size.
    // ---

    if (buffersize == 0) 
      throw(new ECA_ERROR("ECA-OSS", "Buffersize is 0!", stop));
    
    int fragsize, fragtotal = 16;
    unsigned short int fr_size, fr_count = 0x7fff; // 0x7fff = not limited
    
    //    ecadebug->msg(1, "setting OSS fragment size according to buffersize", buffersize);
    //    ecadebug->msg(1, "setting OSS fragment size to", buffersize * format().align);

    // fr_size == 4  -> the minimum fragment size: 2^4 = 16 bytes
    
    for(fr_size = 4; fragtotal < buffersize * format().align; fr_size++)
        fragtotal = fragtotal * 2;

    fragsize = ((fr_count << 16) | fr_size);
    
    if (ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &fragsize)==-1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS-error SNDCTL_DSP_SETFRAGMENT", stop));
    //    ecadebug->msg(1, "set OSS fragment size to (2^x)", fr_size);
    
    // ---
    // Select audio format
    // ---

    if (format().bits == 16) {
        int format = AFMT_S16_LE;
        if (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &format)==-1)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_S16_LE", stop));
        if (format != AFMT_S16_LE)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_S16_LE", stop));
        ecadebug->msg(1,"(dsp_dma) Opened device in mode AFMT_S16_LE");
    }
    else {
        int form = AFMT_U8;
        if (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &form)==-1)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_U8", stop));
        if (form != AFMT_U8)
            throw(new ECA_ERROR("ECA-OSS", "audio format not supported aFMT_U8", stop));
        ecadebug->msg(1, "(dsp_dma) Opened device in mode AFMT_U8");
    }

    // ---
    // Select number of channels
    // ---

    int stereo = format().channels - 1;     /* 0=mono, 1=stereo */
    if (ioctl(audio_fd, SNDCTL_DSP_STEREO, &stereo)==-1)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported STEREO", stop));

    if (stereo != format().channels - 1)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported STEREO", stop));        

    // ---
    // Select sample rate
    // ---
    int speed = format().srate;
    if (ioctl(audio_fd, SNDCTL_DSP_SPEED, &speed) == -1)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported SRATE", stop));
   
    if (speed != (int)format().srate)
        throw(new ECA_ERROR("ECA-OSS", "audio format not supported SRATE", stop));

    AIO_PARAMS stemp = format();
    format(stemp);

    // ---
    // Get fragment size.
    // ---

    if (ioctl(audio_fd, SNDCTL_DSP_GETBLKSIZE, &fr_size) == -1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS error: SNDCTL_DSP_GETBLKSIZE", stop));
    //    ecadebug->msg(1, "OSS set to use fragment size of", fr_size);

    // ---
    // Check capabilities.
    // ---
    
    int caps;
    if (ioctl(audio_fd, SNDCTL_DSP_GETCAPS, &caps)==-1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS error: SNDCTL_DSP_GETCAPS", stop));
    if (!(caps & DSP_CAP_TRIGGER) || !(caps & DSP_CAP_MMAP))
        throw(new ECA_ERROR("ECA-OSS", "general OSS error: no support for DSP_CAP_TRIGGER and DSP_CAP_MMAP", stop));

    if (io_mode() == si_read) {    
        if (ioctl(audio_fd, SNDCTL_DSP_GETISPACE, &audiobuf)==-1)
            throw(new ECA_ERROR("ECA-OSS", "general OSS error: no support for SNDCTL_DSP_GETISPACE", stop));
    }
    else if (io_mode() == si_write) {
        if (ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &audiobuf)==-1)
            throw(new ECA_ERROR("ECA-OSS", "general OSS error: no support for SNDCTL_DSP_GETOSPACE", stop));
    }
    
    total_oss_buffersize = audiobuf.fragstotal * audiobuf.fragsize;
    fragment_size = audiobuf.fragsize;

    // ---
    // Mmap setup.
    // ---
    
    if (io_mode() == si_read) {
        // not implemented! yeah! motha'fucka!
    }
    else if (io_mode() == si_write) {
//        if ((buf=mmap(NULL, total_oss_buffersize, PROT_WRITE, MAP_FILE|MAP_SHARED, audio_fd, 0))==(caddr_t)-1)
        op =(unsigned char *)mmap(NULL, (size_t)total_oss_buffersize, PROT_WRITE, MAP_FILE|MAP_SHARED, audio_fd, 0);
        cerr << "Erno: " << errno << " buffersize " << total_oss_buffersize << ".\n";
        perror("erno-perno!");
        sleep(100);
        if ((int)op == -1)
            throw(new ECA_ERROR("ECA-OSS", "general OSS error: mmap", stop));
//	printf("mmap (out) returned %08x\n", buf);
//	op=buf;
    }
}

void OSSDMA::close_device(void) {
    ecadebug->msg("(dsp_dma) Closing audio device.");
    close(audio_fd);
}

void OSSDMA::enable(void) {
    if (!is_open) {
        open_device();
        is_open = true;
    }
    
    int enable_bits;
    if (io_mode() == si_read) enable_bits = PCM_ENABLE_INPUT;
    else if (io_mode() == si_write) enable_bits = PCM_ENABLE_OUTPUT;
    
    if (ioctl(audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS-error SNDCTL_DSP_SETTRIGGER", stop));
    
    is_triggered = true;
}

void OSSDMA::disable(void) {
    if (is_open) {
        close_device();
        is_open = false;
    }
}

void OSSDMA::get_sample(SAMPLE_BUFFER* t) { }
void OSSDMA::put_sample(SAMPLE_BUFFER* t) {
    if (t->length_in_samples() * format().align != fragment_size)
        throw(new ECA_ERROR("ECA-OSS", "put_sample(): sample_buffer and oss-buffer-setup have different sizes!", stop));
    
    FD_ZERO(&writeset);
    FD_SET(audio_fd, &writeset);

    tim.tv_sec = 10;
    tim.tv_usec= 0;

    if (format().bits == 16) {
      //        t->buffer_to_iobuf(SBTYPE_SSINT, format().channels);
    } else {
      //        t->buffer_to_iobuf(SBTYPE_UCHAR, format().channels);
    }    
    
    select(audio_fd+1, &writeset, &writeset, NULL, NULL);

    if (ioctl(audio_fd, SNDCTL_DSP_GETOPTR, &count)==-1)
        throw(new ECA_ERROR("ECA-OSS", "general OSS error: error in SNDCTL_DSP_GETOPTR", stop));

    count.ptr = (count.ptr/fragment_size)*fragment_size;
    
    if ((count.ptr + fragment_size + 16) < total_oss_buffersize)	// Last fragmento?
        extrabits = 16;
    else
        extrabits = 0;

    //    memcpy(op+count->ptr, t->iobuf_ssint, fragment_size + extrabits);
}

OSSDMA::~OSSDMA(void) { close_device(); }
