// ------------------------------------------------------------------------
// audioio-typeselect.cpp: A proxy class for overriding default keyword
//                         and filename associations.
// Copyright (C) 2001,2002 Kai Vehmanen (kai.vehmanen@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include "eca-logger.h"
#include "eca-object-factory.h"
#include "audioio-null.h"
#include "audioio-typeselect.h"

/**
 * Constructor.
 */
AUDIO_IO_TYPESELECT::AUDIO_IO_TYPESELECT (void) { 

  //  ECA_LOG_MSG(ECA_LOGGER::user_objects, "(audioio-typeselect) constructor " + label() + ".");  

  child_repp = new NULLFILE("uninitialized");
  init_rep = false;
}

/**
 * Destructor.
 */
AUDIO_IO_TYPESELECT::~AUDIO_IO_TYPESELECT (void)
{
  //  ECA_LOG_MSG(ECA_LOGGER::user_objects, "(audioio-typeselect) destructor " + label() + ".");  
  delete child_repp; // either null or the actual child object
}

AUDIO_IO_TYPESELECT* AUDIO_IO_TYPESELECT::clone(void) const
{
  AUDIO_IO_TYPESELECT* target = new AUDIO_IO_TYPESELECT();
  for(int n = 0; n < number_of_params(); n++) {
    target->set_parameter(n + 1, get_parameter(n + 1));
  }
  return(target);
}

void AUDIO_IO_TYPESELECT::open(void) throw(AUDIO_IO::SETUP_ERROR&)
{
  ECA_LOG_MSG(ECA_LOGGER::user_objects, "(audioio-typeselect) open " + label() + ".");  

  if (init_rep != true) {
    AUDIO_IO* tmp = 0;
    if (params_rep.size() > 1) {
      string& type = params_rep[1];
      if (type.size() > 0) {
	//  cerr << "typeselect: " << type << endl;
	tmp = ECA_OBJECT_FACTORY::create_audio_object(type);
      }
    }

    if (tmp != 0) {
      delete child_repp; /* the placeholder null object */
      child_repp = tmp;

      int numparams = child_repp->number_of_params();
      for(int n = 0; n < numparams; n++) {
	child_repp->set_parameter(n + 1, get_parameter(n + 3));
	numparams = child_repp->number_of_params(); // in case 'n_o_p()' varies
      }

      init_rep = true; /* must be set after dyn. parameters */
    }
  }
  
  child_repp->set_buffersize(buffersize());
  child_repp->set_io_mode(io_mode());
  child_repp->set_audio_format(audio_format());
  child_repp->open();
  if (child_repp->locked_audio_format() == true) {
    set_audio_format(child_repp->audio_format());
  }
  set_label(child_repp->label());

  AUDIO_IO::open();
}

void AUDIO_IO_TYPESELECT::close(void)
{ 
  if (child_repp->is_open() == true) child_repp->close();

  AUDIO_IO::close();
}

string AUDIO_IO_TYPESELECT::parameter_names(void) const
{ 
  return(string("typeselect,format,") + child_repp->parameter_names()); 
}

void AUDIO_IO_TYPESELECT::set_parameter(int param, string value)
{ 
  ECA_LOG_MSG(ECA_LOGGER::user_objects, 
		"(audioio-typeselect) set_parameter "
		+ label() + ".");  

  if (param > static_cast<int>(params_rep.size())) params_rep.resize(param);

  if (param > 0) {
    params_rep[param - 1] = value;
  }
  
  if (param > 2 && 
      init_rep == true) {
    child_repp->set_parameter(param - 2, value);
  }
}

string AUDIO_IO_TYPESELECT::get_parameter(int param) const
{
  ECA_LOG_MSG(ECA_LOGGER::user_objects, 
		"(audioio-typeselect) get_parameter "
		+ label() + ".");  

  if (param > 0 && param < static_cast<int>(params_rep.size()) + 1) {
    if (param > 2 &&
	init_rep == true) {
      params_rep[param - 1] = child_repp->get_parameter(param - 2);
    }
    return(params_rep[param - 1]);
  }

  return(""); 
}

void AUDIO_IO_TYPESELECT::set_position_in_samples(SAMPLE_SPECS::sample_pos_t pos)
{
  /* only way to update the current position */
  child_repp->seek_position_in_samples(pos);
  AUDIO_IO::set_position_in_samples(pos);
}

void AUDIO_IO_TYPESELECT::set_length_in_samples(SAMPLE_SPECS::sample_pos_t pos)
{
  // child_repp->set_length_in_samples(pos);
  AUDIO_IO::set_length_in_samples(pos);
}
