// ------------------------------------------------------------------------
// qeoperatorconfiguration.cpp: Input widget for configuring ecasound 
//                              operators
// Copyright (C) 2000,2001 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <iostream>
#include <vector>
#include <string>

#include <qaccel.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qhgroupbox.h>
#include <qvgroupbox.h>
#include <qhbox.h>
#include <qlineedit.h>
#include <qgrid.h>
#include <qslider.h>

#include "qeoperatorconfiguration.h"

using std::string;
using std::cerr;
using std::endl;

/**
 * Class constructor
 */
QEOperatorConfiguration::QEOperatorConfiguration (OPERATOR* op, QWidget *parent, const char *name) 
  : QEInput(parent, name),
    operator_repp(op) {
  init_layout();
}

QEOperatorConfiguration::~QEOperatorConfiguration (void) {
  for(size_t n = 0; n < pardesclist_rep.size(); n++) {
    delete pardesclist_rep[n];
    pardesclist_rep[n] = 0;
  }
}

/**
 * Creates and initializes widget layout
 */
void QEOperatorConfiguration::init_layout(void) {
  QBoxLayout* top = new QVBoxLayout(this);

  QGroupBox* descgroup = new QVGroupBox(this, "descgroup");
  //obj_name_repp = new QLabel(operator_repp->name().c_str(), descgroup, "name2");
  obj_desc_repp = new QLabel(operator_repp->description().c_str(), descgroup, "desc2");
  top->addWidget(descgroup);

  /* create fields for individual operator parameters */
  paramgrid_repp = new QGrid(6, this);

  top->addWidget(paramgrid_repp, 1);
  
  if (operator_repp != 0) change_operator(operator_repp);

}

/** 
 * Initilizes a new operator object
 */
void QEOperatorConfiguration::change_operator(OPERATOR* op) {
  assert(op != 0);

  operator_repp = op;
  //obj_name_repp->setText((operator_repp->name()+":").c_str());
  obj_desc_repp->setText(operator_repp->description().c_str());

  int params_height = 0;
  for(int n = 0; n < operator_repp->number_of_params(); n++) {
    if (n == static_cast<int>(paramlist_rep.size())) {
      paramlist_rep.push_back(new QLabel("", paramgrid_repp));
      pardesclist_rep.push_back(new OPERATOR::PARAM_DESCRIPTION());
      inputlist_rep.push_back(new QLineEdit(paramgrid_repp));
      sliderlist_rep.push_back(new QSlider(QSlider::Horizontal, paramgrid_repp));
      sliderlist_rep.back()->setMinimumWidth(100);      
      lowlimitlist_rep.push_back(new QLineEdit(paramgrid_repp));
      highlimitlist_rep.push_back(new QLineEdit(paramgrid_repp));
      suggestlist_rep.push_back(new QLabel("", paramgrid_repp));
    }

    /* label */
    paramlist_rep[n]->setMargin(5);
    paramlist_rep[n]->setText(QString(QString::number(n + 1) + 
				  QString(". ") + 
				  QString(operator_repp->get_parameter_name(n + 1).c_str())).leftJustify(12));

    /* value textarea input */
    inputlist_rep[n]->setText(QString::number(operator_repp->get_parameter(n + 1)));
    inputlist_rep[n]->installEventFilter(this);
    QObject::connect(inputlist_rep[n], 
		     SIGNAL(returnPressed()), 
		     this, 
		     SLOT(parameter_change_event()));

    /* slider input */
    slidervalues_rep.resize(operator_repp->number_of_params());
    QObject::connect(sliderlist_rep[n],
		     SIGNAL(valueChanged(int)),
		     this,
		     SLOT(slider_change_event(int)));

    /* textarea for setting low limit for range */
    lowlimitlist_rep[n]->installEventFilter(this);
    QObject::connect(lowlimitlist_rep[n],
		     SIGNAL(returnPressed()), 
		     this, 
		     SLOT(range_change_event()));

    /* textarea for setting high limit for range */
    highlimitlist_rep[n]->installEventFilter(this);
    QObject::connect(highlimitlist_rep[n], 
		     SIGNAL(returnPressed()), 
		     this, 
		     SLOT(range_change_event()));

    /* suggested range */
    set_from_parameter_desc(n, false);
    
    params_height += paramlist_rep[n]->height();

    inputlist_rep[n]->show();
    paramlist_rep[n]->show();
    sliderlist_rep[n]->show();
    suggestlist_rep[n]->show();
    lowlimitlist_rep[n]->show();
    highlimitlist_rep[n]->show();
  }

  set_range_values();
  set_sliders();

  for(int n = operator_repp->number_of_params(); n < static_cast<int>(paramlist_rep.size()); n++) {
    inputlist_rep[n]->hide();
    paramlist_rep[n]->hide();
    sliderlist_rep[n]->hide();
    lowlimitlist_rep[n]->hide();
    highlimitlist_rep[n]->hide();
    suggestlist_rep[n]->hide();
  }

  //  paramgrid_repp->resize(paramgrid_repp->width(), params_height);
}

bool QEOperatorConfiguration::eventFilter(QObject *, QEvent *e) {
  if (e->type() == QEvent::FocusOut) {
    parameter_change_event();
    range_change_event();
    return(true);
  }
  else 
    return(false);
}

void QEOperatorConfiguration::range_change_event(void) {
  //  cerr << "range_change_event()" << endl;
  set_range_values();
}

void QEOperatorConfiguration::parameter_change_event(void) {
  //  cerr << "parameter_change_event()" << endl;
  emit parameters_changed();
}

void QEOperatorConfiguration::slider_change_event(int n) {
  //  cerr << "slider_change_event()" << endl;

  bool changes = false;

  for(int n = 0; n < operator_repp->number_of_params(); n++) {
    int value = sliderlist_rep[n]->value();
    //  cerr << "n=" << n << ", value=" << value << ", slidervalue=" << slidervalues_rep[n] << "." << endl;
    if (value != slidervalues_rep[n]) {
      double fvalue = value;

      if (pardesclist_rep[n]->integer != true) {
	fvalue /= integer_steps_const;
      }

      if (static_cast<int>(inputlist_rep.size()) > n) inputlist_rep[n]->setText(QString::number(fvalue));

      //  cerr << "Parameter " << n << " of op " << operator_repp->name() << " was changed from " << slidervalues_rep[n] << " to " << fvalue << " (" << value << ")." << endl;

      slidervalues_rep[n] = value;
      changes = true;
    }
  }
  
  if (changes == true) {
    //  cerr << "parameters_changed() signal" << endl;
    emit parameters_changed();
  }
}


void QEOperatorConfiguration::update_results(void) {
  for(int n = 0; n < operator_repp->number_of_params(); n++) {
    operator_repp->set_parameter(n + 1, inputlist_rep[n]->text().toFloat());
  }
}

void QEOperatorConfiguration::set_range_values(void) {
  for(int n = 0; n < operator_repp->number_of_params(); n++) {
    double minvalue = lowlimitlist_rep[n]->text().toFloat();
    double maxvalue = highlimitlist_rep[n]->text().toFloat();

    if (pardesclist_rep[n]->integer != true) {
      minvalue *= integer_steps_const;
      maxvalue *= integer_steps_const;
    }

    sliderlist_rep[n]->setMinValue(static_cast<int>(minvalue));
    sliderlist_rep[n]->setMaxValue(static_cast<int>(maxvalue));
  }
}

void QEOperatorConfiguration::set_sliders(void) {
  for(int n = 0; n < operator_repp->number_of_params(); n++) {
    double curvalue = operator_repp->get_parameter(n + 1);

    double minvalue = lowlimitlist_rep[n]->text().toFloat();
    double maxvalue = highlimitlist_rep[n]->text().toFloat();
    double slidervalue = 0;

    if (curvalue < minvalue) slidervalue = minvalue;
    else if (curvalue > maxvalue) slidervalue = maxvalue;
    else {
      slidervalue = curvalue;
    }

    if (pardesclist_rep[n]->integer != true) {
      slidervalue *= integer_steps_const;
    }

    sliderlist_rep[n]->setValue(static_cast<int>(slidervalue));
    //  cerr << "Setting slidervalue to " << static_cast<int>(slidervalue) << " cop-param is " << curvalue << "." << endl;
  }
}

void QEOperatorConfiguration::set_from_parameter_desc(int n, bool create) {
  operator_repp->parameter_description(n + 1, pardesclist_rep[n]);
  double param = operator_repp->get_parameter(n + 1);

  QString lowbound = "n/a";
  QString highbound = "n/a";

  if (pardesclist_rep[n]->bounded_below == true) {
    lowbound = QString::number(pardesclist_rep[n]->lower_bound);
    lowlimitlist_rep[n]->setText(lowbound);
  }
  else {
    if (param < 0)
      lowlimitlist_rep[n]->setText(QString::number(param + param * 2));
    else 
      lowlimitlist_rep[n]->setText("0");
  }
  
  if (pardesclist_rep[n]->bounded_above == true) {
    highbound = QString::number(pardesclist_rep[n]->upper_bound);
    highlimitlist_rep[n]->setText(highbound);
  }
  else {
    if (param > 0) 
      highlimitlist_rep[n]->setText(QString::number(param + param * 2));
    else
      highlimitlist_rep[n]->setText("1");
  }
 
  QString text =  "[" + lowbound + " -> " + highbound + "]";
  if (create == true) {
    suggestlist_rep[n] = new QLabel(text, paramgrid_repp);
  }
  else {
    suggestlist_rep[n]->setText(text);
  }
} 
