/* pluck.c -- plucked string class definitions */

/*
 * Code conversion from C++ to C (October 1994)
 * Author: Michael A. Casey MIT Media Labs
 * Language: C
 * Copyright (c) 1994 MIT Media Lab, All Rights Reserved
 */

#include "cs.h"
#include "entry.h"
#include "wavegde.h"
#include "pluck.h"
#include "auxfd.h"

/* external prototypes */

/* ***** plucked string class member function definitions ***** */

/* ::pluck -- create the plucked-string instrument */
void
pluckPluck(WGPLUCK * plk)
{

	/* ndelay = total required delay - 1.0 */
	len_t ndelay = (len_t) (esr / *plk->freq - 1.0);

#ifdef WG_VERBOSE
	info << "pluckPluck -- allocating memory..." << endmsg;

#endif

	/* Allocate auxillary memory or reallocate if size has changed */
	auxalloc((len_t) (ndelay / 2.0) * sizeof (sampT), &plk->upperData);
	auxalloc((len_t) (ndelay / 2.0) * sizeof (sampT), &plk->lowerData);
	auxalloc(3L * sizeof (paramT), &plk->bridgeCoeffs);
	auxalloc(3L * sizeof (sampT), &plk->bridgeData);

#ifdef WG_VERBOSE
	info << "done.\n" << endmsg;

#endif

	/* construct waveguide object */
#ifdef WG_VERBOSE
	info << "Constructing waveguide..." << endmsg;

#endif

	waveguideWaveguide((waveguide *) & plk->wg,	/* waveguide      
							 */
			   (paramT) * plk->freq,	/* f0 frequency    */
			   (sampT *) plk->upperData.auxp,	/* upper rail
								   data */
			   (sampT *) plk->lowerData.auxp);	/* lower rail 
								   data */
#ifdef WG_VERBOSE
	info << "done.\n" << endmsg;

#endif
	/* Allocate memory to bridge data and coeffs */
#ifdef WG_VERBOSE
	info << "Initializing bridge filters..." << endmsg;

#endif
	plk->bridge.coeffs = (paramT *) plk->bridgeCoeffs.auxp;
	/* bridge coeffs */
	plk->bridge.buffer.data = (sampT *) plk->bridgeData.auxp;	/*
									   bridge data */
	filterFilter(&plk->bridge, 3);	/* construct bridge filter object 
					 */
#ifdef WG_VERBOSE
	info << "done\n" << endmsg;

#endif
	/* Excite the string with the input parameters */
#ifdef WG_VERBOSE
	info << "Exciting the string..." << endmsg;

#endif
	pluckExcite(plk);
#ifdef WG_VERBOSE
	info << "done\n" << endmsg;

#endif
}

/* pluck::excite -- excitation function for plucked string */
void
pluckExcite(WGPLUCK * plk)
{
	sampPtr shape;
	int i;

	/* set the delay element to pick at */
	plk->pickSamp = (len_t) (plk->wg.upperRail.size * *plk->pickPos);
	if (plk->pickSamp < 1)
		plk->pickSamp = 1;

	/* set the delay element to pickup at */
	plk->pickupSamp = (len_t) (plk->wg.upperRail.size * *plk->pickupPos);
	if (plk->pickupSamp < 1)
		plk->pickupSamp = 1;

	/* set the bridge filter coefficients for the correct magnitude
	   response */
	pluckSetFilters(plk, *plk->Aw0, *plk->AwPI);	/*attenuation in dB 
							   at w0 and PI */

	/* add the pick shape to the waveguide rails */
	shape = pluckShape(plk, *plk->amp);

	/* add shape to lower rail */
	for (i = 0; i < plk->wg.lowerRail.size; i++)
		plk->wg.lowerRail.data[i] += shape[i];

	/* flip shape and add to upper rail */
	pluckFlip(plk, shape);
	for (i = 0; i < plk->wg.upperRail.size; i++)
		plk->wg.upperRail.data[i] += shape[i];

	/* free the space used by the pluck shape */
	mfree((char *) shape);

	/* Reset the tuning and bridge filters */
	/*filterReset(&plk->wg.tnFIR); */
	/*filterReset(&plk->bridge); */

	/* set excitation flag */
	plk->wg.excited = 1;
}


/* pluck::setFilters -- frequency dependent filter calculations */
void
pluckSetFilters(WGPLUCK * plk, paramT A_w0, paramT A_PI)
{
	paramT coeffs[3];	/* array of updated coefficient values */

	/* Define the required magnitude response of H1 at w0 and PI */

	paramT tmpVal1 = (paramT) powf((float) 10.0, (float) (-A_w0 / 20));
	paramT tmpVal2 = (paramT) powf((float) 10.0, (float) (-A_PI / 20));

	/* Constrain attenuation specification to dB per second */
	paramT N = esr / plk->wg.f0;	/*  N=t*esr/f0  */
	paramT H1_w0 = (paramT) powf((float) tmpVal1, (float) (1 / N));
	paramT H1_PI = (paramT) powf((float) tmpVal2, (float) (1 / N));

	/* The tuning filter is allpass, so no dependency for H1 */
	/* therefore solve for the coefficients of the bridge filter directly
	 */
	paramT a1 = (H1_w0 + cosf((float) plk->wg.w0) * H1_PI) / (1 +
					       cosf((float) plk->wg.w0));
	paramT a0 = (a1 - H1_PI) / 2.0f;

	/* apply constraints on coefficients (see Sullivan) */
	if ((a0 < 0.0f) || (a1 < 2.0 * a0)) {
		a0 = 0.0f;
		a1 = H1_w0;
	}
	coeffs[0] = a0;
	coeffs[1] = a1;
	coeffs[2] = a0;
	filterSet(&plk->bridge, coeffs);	/* set the new bridge
						   coefficients */

/*  if (VERBOSE)
   info << "bridge :a0=" << a0 << ", a1=" << a1 << "\n" << endmsg;

 */
}

/* Reverse the positions of the values in 'a' */
sampPtr
pluckFlip(WGPLUCK * plk, sampPtr a)
{
	int i, j;
	int n = plk->wg.upperRail.size;
	int nh = (int) (n / 2.0f);
	sampT tmp;

	for (i = 0; i < nh; i++) {
		j = n - i - 1;
		tmp = a[i];
		a[i] = a[j];
		a[j] = tmp;
	}
	return a;
}

/* ::pluckShape -- the pluck function for a string */
sampPtr
pluckShape(WGPLUCK * plk, paramT amp)
{				/* AMP param is NOT USED */
	paramT scale = *plk->amp;
	sampPtr shape;
	len_t len = plk->wg.lowerRail.size;
	len_t i, M;

	/* This memory must be freed after use */
	shape = (sampPtr) mmalloc(len * sizeof (sampT));
	if (!shape)
		error << "Couldn't allocate for initial shape, <pluckShape>" << endmsg;


	for (i = 0; i < plk->pickSamp; i++)
		shape[i] = scale ** plk->amp * i / plk->pickSamp;

	M = len - plk->pickSamp;
	for (i = 0; i < M; i++)
		shape[plk->pickSamp + i] = scale ** plk->amp - (i * scale **
							   plk->amp / M);

	return shape;
}


/* ::getSamps -- the sample generating routine */
void
pluckGetSamps(WGPLUCK * plk)
{
	sampT yr0, yl0, yrM, ylM;	/* Key positions on the waveguide */
	sampPtr sampBuf = plk->out;	/* The sample output buffer */
	len_t M = plk->wg.upperRail.size;	/* Length of the guide rail */
	len_t i = 0, N = ksmps;
	paramT *fdbk = plk->afdbk;

	/* calculate N samples of the plucked string algorithm */
	if (plk->wg.excited)
		do {
			sampBuf[i] = guideRailAccess(&plk->wg.upperRail,
						     plk->pickupSamp)
			    + guideRailAccess(&plk->wg.lowerRail,
					      -plk->pickupSamp);
			sampBuf[i++] += *fdbk++;

			yrM = guideRailAccess(&plk->wg.upperRail, M - 1);
			/* wave into the nut */
			ylM = -yrM;	/* reflect the incoming sample at
					   the nut */

			yl0 = guideRailAccess(&plk->wg.lowerRail, 0);	/* 
									   wave into bridge */

			yr0 = -filterFIR(&plk->bridge, yl0);	/* bridge
								   reflection filter */
			yr0 = filterAllpass(&plk->wg, yr0);	/* allpass
								   tuning filter */

			guideRailUpdate(&plk->wg.upperRail, yr0);	/*
									   update the upper rail */
			guideRailUpdate(&plk->wg.lowerRail, ylM);	/*
									   update the lower rail */

		} while (--N);
}




OENTRY opcodes[] =
{
	{"wgpluck", S(WGPLUCK), 5, "a", "iikiiia", F(pluckPluck), NULL,
	 F(pluckGetSamps)},
	{NULL}
};
