/* Collection of physical modelled instruments */

#include <math.h>

#include "cs.h"
#include "entry.h"
#include "insert.h"
#include "fgens.h"
#include "auxfd.h"
#include "clarinet.h"
#include "flute.h"
#include "bowed.h"
#include "brass.h"
#include "physmod.h"

/* ************************************** */
/*  Waveguide Clarinet model ala Smith    */
/*  after McIntyre, Schumacher, Woodhouse */
/*  by Perry Cook, 1995-96                */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/* ************************************** */

/**********************************************/
/*  One break point linear reed table object  */
/*  by Perry R. Cook, 1995-96                 */
/*  Consult McIntyre, Schumacher, & Woodhouse */
/*        Smith, Hirschman, Cook, Scavone,    */
/*        more for information.               */
/**********************************************/

float
ReedTabl_LookUp(ReedTabl * r, float deltaP)
    /*   Perform "Table Lookup" by direct clipped  */
    /*   linear function calculation               */
{				/*   deltaP is differential reed pressure      
				 */
	r->lastOutput = r->offSet + (r->slope * deltaP);	/* basic
								   non-lin */
	if (r->lastOutput > 1.0f)
		r->lastOutput = 1.0f;	/* if other way, reed slams shut
					 */
	if (r->lastOutput < -1.0f)
		r->lastOutput = -1.0f;	/* if all the way open, acts like 
					   open end */
/*     info << "ReedT: Lookup=" << r->lastOutput << "\n" << endmsg;
 */
	return r->lastOutput;
}


/* *********************************************************************** */
void
clarinset(CLARIN * p)
{
	FUNC *ftp;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */

	if ((ftp = ftfind(p->ifn)) != NULL)
		p->vibr = ftp;
	else
		error << "PERF: No table for Clarinet" << endmsg;
	/* Expect sine
							   wave */
	if (*p->lowestFreq >= 0.0f) {	/* Skip initialisation */
		if (*p->lowestFreq)
			p->length = (long) (esr / *p->lowestFreq + 1.0f);
		else if (*p->frequency)
			p->length = (long) (esr / *p->frequency + 1.0f);
		else
			error << "PERF: No base frequency for clarinet" << endmsg;

		make_DLineL(&p->delayLine, p->length);
		p->reedTable.offSet = 0.7f;
		p->reedTable.slope = -0.3f;
		make_OneZero(&(p->filter));
		make_Envelope(&p->envelope);
		make_Noise(p->noise);
		/*    p->noiseGain = 0.2f; *//* Arguemnts; suggested values? */
		/*    p->vibrGain = 0.1f; */
		{
			unsigned short relestim = (unsigned short) (ekr *
								    0.1f);	/* 1/10th second decay extention */
			if (relestim > p->h.insdshead->xtratim)
				p->h.insdshead->xtratim = relestim;
		}
		p->kloop = (int) (p->h.insdshead->offtim * ekr) - (int) (ekr *
							     *p->attack);
/*     info << "offtim=" << */
/*            p->h.insdshead->offtim << "  kloop=" << p->kloop << "\n" << endmsg;
 */
		p->envelope.rate = amp / (*p->attack * esr);	/* Over
								   0.1sec */
		p->envelope.target = 0.55f + amp * 0.30f;	/* ?? values
								   here */
	}
}

void
clarin(CLARIN * p)
{
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */
	float nGain = *p->noiseGain;
	int v_len = (int) p->vibr->flen;
	float *v_data = p->vibr->ftable;
	float vibGain = *p->vibAmt;
	float vTime = p->v_time;

	p->outputGain = amp + 0.001f;
	DLineL_setDelay(&p->delayLine,	/* length - approx filter delay
					 */
			(esr / *p->frequency) * 0.5f - 1.5f);
	p->v_rate = *p->vibFreq * p->vibr->flen / esr;
	/* Check to see if into decay yet */
	if (p->kloop > 0 && p->h.insdshead->relesing)
		p->kloop = 1;
	if ((--p->kloop) == 0) {
		p->envelope.state = 1;	/* Start change */
		p->envelope.rate = p->envelope.value / (*p->dettack * esr);
		p->envelope.target = 0.0f;
/*       info << "Set off phase time = " << */
/*              (float)kcounter/ekr << " Breath v,r = " << p->envelope.value << ", " << p->envelope.rate << "\n" << endmsg;
 */
	}
	do {
		float pressureDiff;
		float breathPressure;
		long temp;
		float temp_time, alpha;
		float nextsamp;
		float v_lastOutput;
		float lastOutput;

		breathPressure = Envelope_tick(&p->envelope);
		breathPressure += breathPressure * nGain *
		    Noise_tick(&p->noise);
		/* Tick on vibrato table */
		vTime += p->v_rate;	/*  Update current time    */
		while (vTime >= v_len)	/*  Check for end of sound */
			vTime -= v_len;		/*  loop back to
						   beginning */
		while (vTime < 0.0f)	/*  Check for end of sound */
			vTime += v_len;		/*  loop back to
						   beginning */

		temp_time = vTime;

#ifdef have_phase
		if (p->v_phaseOffset != 0.0f) {
			temp_time += p->v_phaseOffset;	/*  Add phase
							   offset       */
			while (temp_time >= v_len)	/*  Check for end of
							   sound */
				temp_time -= v_len;	/*  loop back to
							   beginning */
			while (temp_time < 0.0f)	/*  Check for end of
							   sound */
				temp_time += v_len;	/*  loop back to
							   beginning */
		}
#endif
		temp = (long) temp_time;	/*  Integer part of time
						   address    */
		/*  fractional part of time address */
		alpha = temp_time - (float) temp;
		v_lastOutput = v_data[temp];	/* Do linear interpolation
						 */
		/*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
		v_lastOutput += (alpha * (v_data[temp + 1] - v_lastOutput));
		/* End of vibrato tick */
		breathPressure += breathPressure * vibGain * v_lastOutput;
		pressureDiff = OneZero_tick(&p->filter,		/*
								   differential pressure  */
					  DLineL_lastOut(&p->delayLine));
		pressureDiff = (-0.95f * pressureDiff) - breathPressure;
/* of reflected and mouth */
		nextsamp = pressureDiff * ReedTabl_LookUp(&p->reedTable,
							  pressureDiff);
		nextsamp = breathPressure + nextsamp;
		lastOutput =
		    DLineL_tick(&p->delayLine, nextsamp);	/* perform
								   scattering in economical way */
		lastOutput *= p->outputGain;
		*ar++ = lastOutput * AMP_SCALE;
	} while (--nsmps);
	p->v_time = vTime;

}

/******************************************/
/*  WaveGuide Flute ala Karjalainen,      */
/*  Smith, Waryznyk, etc.                 */
/*  with polynomial Jet ala Cook          */
/*  by Perry Cook, 1995-96                */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/******************************************/


/**********************************************/
/* Jet Table Object by Perry R. Cook, 1995-96 */
/* Consult Fletcher and Rossing, Karjalainen, */
/*       Cook, more, for information.         */
/* This, as with many other of my "tables",   */
/* is not a table, but is computed by poly-   */
/* nomial calculation.                        */
/**********************************************/

float
JetTabl_lookup(float sample)
{				/* Perform "Table Lookup"  *//* By Polynomial
				   Calculation */
	/* (x^3 - x) approximates sigmoid of jet */
	float j = sample * (sample * sample - 1.0f);
	if (j > 1.0f)
		j = 1.0f;	/* Saturation at +/- 1.0       */
	else if (j < -1.0f)
		j = -1.0f;
	return j;
}

void
fluteset(FLUTE * f)
{
	FUNC *ftp;
	long length;
	float amp = (*f->amp) * AMP_RSCALE;	/* Normalise */
	float temp;

	if ((ftp = ftfind(f->ifn)) != NULL)
		f->vibr = ftp;
	else
		error << "PERF: No table for Flute" << endmsg;
	/* Expect sine wave
							 */
	if (*f->lowestFreq >= 0.0f) {	/* Skip initialisation?? */
		if (*f->lowestFreq == 0.0)
			length = (long) (esr / *f->lowestFreq + 1.0f);
		else if (*f->frequency)
			length = (long) (esr / *f->frequency + 1.0f);
		else
			error << "PERF: No base frequency for flute" << endmsg;

		make_DLineL(&f->boreDelay, length);
		length = length >> 1;	/* ??? really */
		make_DLineL(&f->jetDelay, length);
		make_OnePole(&f->filter);
		make_DCBlock(&f->dcBlock);
		make_Noise(f->noise);
		make_ADSR(&f->adsr);
		/* Clear */
/*     OnePole_clear(&f->filter); */
/*     DCBlock_clear(&f->dcBlock); */
		/* End Clear */
/*       DLineL_setDelay(&f->boreDelay, 100.0f); */
/*       DLineL_setDelay(&f->jetDelay, 49.0f); */

		OnePole_setPole(&f->filter, 0.7f - (0.1f * RATE_NORM));
		OnePole_setGain(&f->filter, -1.0f);
		ADSR_setAll(&f->adsr, 0.02f, 0.05f, 0.8f, 0.001f);
		/* Suggested values */
		/*    f->endRefl = 0.5; */
		/*    f->jetRefl = 0.5; */
		/*    f->noiseGain = 0.15; *//* Breath pressure random
		   component   */
		/*    f->vibrGain = 0.05;  *//* breath periodic vibrato
		   component  */
		/*    f->jetRatio = 0.32;  */
		f->lastamp = amp;	/* Remember */
		ADSR_setAttackRate(&f->adsr, amp * 0.02f);	/* This
								   should be controlled by attack */
		f->maxPress = (1.1f + (amp * 0.20f)) / 0.8f;
		f->outputGain = amp + 0.001f;
		ADSR_keyOn(&f->adsr);
		f->kloop = (float) ((int) (f->h.insdshead->offtim * ekr - ekr *
					   (*f->dettack)));

		f->lastFreq = *f->frequency;
		f->lastJet = *f->jetRatio;
		/* freq = (2/3)*f->frequency as we're overblowing here */
		/* but 1/(2/3) is 1.5 so multiply for speed */
		temp = 1.5f * esr / f->lastFreq - 2.0f;		/*
								   Length - approx. filter delay */
		DLineL_setDelay(&f->boreDelay, temp);	/* Length of bore
							   tube */
		DLineL_setDelay(&f->jetDelay, temp * f->lastJet);	/* jet 
									   delay shorter */
	}
/*     info << "offtim=" << f->h.insdshead->offtim << "  kloop=" << f->kloop << "\n" << endmsg;
 */
/*     info << "Flute : NoteOn: Freq=" << *f->frequency << " Amp=" << amp << "\n" << endmsg;
 */
}

void
flute(FLUTE * f)
{
	float *ar = f->ar;
	long nsmps = ksmps;
	float amp = (*f->amp) * AMP_RSCALE;	/* Normalise */
	float temp;
	int v_len = (int) f->vibr->flen;
	float *v_data = f->vibr->ftable;
	float v_time = f->v_time;
	float vibGain = *f->vibAmt;

	if (amp != f->lastamp) {	/* If amplitude has changed */
		ADSR_setAttackRate(&f->adsr, amp * 0.02f);	/* This
								   shoudl be controlled by attack */
		f->maxPress = (1.1f + (amp * 0.20f)) / 0.8f;
		f->outputGain = amp + 0.001f;
		f->lastamp = amp;
	}
	f->v_rate = *f->vibFreq * v_len / esr;
	/* Start SetFreq */
	if (f->lastFreq != *f->frequency) {	/* It changed */
		f->lastFreq = *f->frequency;
		f->lastJet = *f->jetRatio;
		/* freq = (2/3)*f->frequency as we're overblowing here */
		/* but 1/(2/3) is 1.5 so multiply for speed */
		temp = 1.5f * esr / f->lastFreq - 2.0f;		/*
								   Length - approx. filter delay */
		DLineL_setDelay(&f->boreDelay, temp);	/* Length of bore
							   tube */
		DLineL_setDelay(&f->jetDelay, temp * f->lastJet);	/* jet 
									   delay shorter */
	} else if (*f->jetRatio != f->lastJet) {	/* Freq same but jet
							   changed */
		f->lastJet = *f->jetRatio;
		temp = 1.5f * esr / f->lastFreq - 2.0f;		/*
								   Length - approx. filter delay */
		DLineL_setDelay(&f->jetDelay, temp * f->lastJet);	/* jet 
									   delay shorter */
	}
	/* End SetFreq */

	if (f->kloop > 0.0f && f->h.insdshead->relesing)
		f->kloop = 1.0f;
	if ((--f->kloop) == 0) {
		f->adsr.releaseRate = f->adsr.value / (*f->dettack * esr);
		f->adsr.target = 0.0f;
		f->adsr.rate = f->adsr.releaseRate;
		f->adsr.state = RELEASE;
/*       info << "Set off phase time = " << (float)kcounter/ekr << "\n" << endmsg;
 */
	}
	do {
		long temp;
		float temf;
		float temp_time, alpha;
		float pressDiff;
		float randPress;
		float breathPress;
		float lastOutput;
		float v_lastOutput;

		breathPress = f->maxPress * ADSR_tick(&f->adsr);	/*
									   Breath Pressure */
		randPress = *f->noiseGain * Noise_tick(&f->noise);	/*
									   Random Deviation */
		/* Tick on vibrato table */
		v_time += f->v_rate;	/*  Update current time    */
		while (v_time >= v_len)		/*  Check for end of
						   sound */
			v_time -= v_len;	/*  loop back to beginning */
		while (v_time < 0.0f)	/*  Check for end of sound */
			v_time += v_len;	/*  loop back to beginning */

		temp_time = v_time;

#ifdef phase_offset
		if (f->v_phaseOffset != 0.0f) {
			temp_time += f->v_phaseOffset;	/*  Add phase
							   offset       */
			while (temp_time >= v_len)	/*  Check for end of
							   sound */
				temp_time -= v_len;	/*  loop back to
							   beginning */
			while (temp_time < 0.0f)	/*  Check for end of
							   sound */
				temp_time += v_len;	/*  loop back to
							   beginning */
		}
#endif

		temp = (long) temp_time;	/*  Integer part of time
						   address    */
		/*  fractional part of time address */
		alpha = temp_time - (float) temp;
		v_lastOutput = v_data[temp];	/* Do linear interpolation
						 */
		/*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
		v_lastOutput += (alpha * (v_data[temp + 1] - v_lastOutput));
/*       info << "Vibrato " << v_lastOutput << "\n" << endmsg;
 */
		/* End of vibrato tick */
		randPress += vibGain * v_lastOutput;	/* + breath vibrato 
							 */
		randPress *= breathPress;	/* All scaled by Breath
						   Pressure */
		temf = OnePole_tick(&f->filter, DLineL_lastOut(&f->boreDelay));
		temf = DCBlock_tick(&f->dcBlock, temf);		/* Block 
								   DC on reflection */
		pressDiff = breathPress + randPress	/* Breath Pressure   
							 */
		    - (*f->jetRefl * temf);	/*  - reflected      */
		pressDiff = DLineL_tick(&f->jetDelay, pressDiff);	/* Jet 
									   Delay Line */
		pressDiff = JetTabl_lookup(pressDiff)	/* Non-Lin Jet +
							   reflected */
		    +(*f->endRefl * temf);
		lastOutput = 0.3f * DLineL_tick(&f->boreDelay, pressDiff);
		/* Bore Delay and "bell" filter  */

		lastOutput *= f->outputGain;
/*       info << "sample=" << lastOutput << "\n" << endmsg;
 */
		*ar++ = lastOutput * AMP_SCALE * 1.4f;
	} while (--nsmps);

	f->v_time = v_time;
}

/******************************************/
/*  Bowed String model ala Smith          */
/*  after McIntyre, Schumacher, Woodhouse */
/*  by Perry Cook, 1995-96                */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/******************************************/


/******************************************/
/*  Simple Bow Table Object, after Smith  */
/*    by Perry R. Cook, 1995-96           */
/******************************************/

float
BowTabl_lookup(BowTabl * b, float sample)
{				/*  Perform Table Lookup    *//*  sample is
				   differential  */
	/*  string vs. bow velocity */
	float input;
	input = sample /* + b->offSet */ ;	/*  add bias to sample     
						 */
	input *= b->slope;	/*  scale it                */
	b->lastOutput = (float) fabs(input) + 0.75f;	/*  below min
							   delta, frict = 1 */
	b->lastOutput = (float) pow((double) b->lastOutput, -4.0);
	/* if (b->lastOutput < 0.0f ) b->lastOutput = 0.0f; *//* minimum frict
	   is 0.0 */
	if (b->lastOutput > 1.0f)
		b->lastOutput = 1.0f;	/*  maximum friction is 1.0 */
	return b->lastOutput;
}

void
bowedset(BOWED * b)
{
	long length;
	FUNC *ftp;
	float amp = (*b->amp) * AMP_RSCALE;	/* Normalise */

	if ((ftp = ftfind(b->ifn)) != NULL)
		b->vibr = ftp;
	else
		error << "PERF: No table for wgbow vibrato" << endmsg;
	/* Expect
								   sine wave */
	if (*b->lowestFreq >= 0.0f) {	/* If no init skip */
		if (*b->lowestFreq != 0.0f)
			length = (long) (esr / *b->lowestFreq + 1.0f);
		else if (*b->frequency != 0.0f)
			length = (long) (esr / *b->frequency + 1.0f);
		else
			error << "PERF: Unknown lowest frequency for bowed"
			      " string" << endmsg;

		/* info << "Length=" << length << "\n" << endmsg;
 */
		make_DLineL(&b->neckDelay, length);
		/*    length = length >> 1; *//* ?? Unsure about this */
		make_DLineL(&b->bridgeDelay, length);

		/*  b->bowTabl.offSet = 0.0f; *//* offset is a bias, really not 
		   needed unless */
		/* friction is different in each direction    */
		/* b->bowTabl.slope contrls width of friction pulse, related to 
		   bowForce */
		b->bowTabl.slope = 3.0f;
		make_OnePole(&b->reflFilt);
		make_BiQuad(&b->bodyFilt);
		make_ADSR(&b->adsr);

		DLineL_setDelay(&b->neckDelay, 100.0f);
		DLineL_setDelay(&b->bridgeDelay, 29.0f);

		OnePole_setPole(&b->reflFilt, 0.6f - (0.1f * RATE_NORM));
		OnePole_setGain(&b->reflFilt, 0.95f);

		BiQuad_setFreqAndReson(b->bodyFilt, 500.0f, 0.85f);
		BiQuad_setEqualGainZeroes(b->bodyFilt);
		BiQuad_setGain(b->bodyFilt, 0.2f);

		ADSR_setAll(&b->adsr, 0.002f, 0.01f, 0.9f, 0.01f);

		b->adsr.target = 1.0f;
		b->adsr.rate = b->adsr.attackRate;
		b->adsr.state = ATTACK;
		b->maxVelocity = 0.03f + (0.2f * amp);

		b->lastpress = 0.0f;	/* Set unknown state */
		b->lastfreq = 0.0f;
		b->lastbeta = 0.0f;	/* Remember states */
		b->lastamp = amp;
	}
}


void
bowed(BOWED * b)
{
	float *ar = b->ar;
	long nsmps = ksmps;
	float amp = (*b->amp) * AMP_RSCALE;	/* Normalise */
	float maxVel;

	if (amp != b->lastamp) {
		b->maxVelocity = 0.03f + (0.2f * amp);
		b->lastamp = amp;
	}
	maxVel = b->maxVelocity;
	if (b->lastpress != *b->bowPress)
		b->lastpress = *b->bowPress;	/* Bowslope not used?? */

	/* Set Frequency if changed */
	if (b->lastfreq != *b->frequency) {
		/* delay - approx. filter delay */
		b->lastfreq = *b->frequency;
		b->baseDelay = esr / b->lastfreq - 4.0f;
/* info << "Freq set to " << b->lastfreq << " with basedelay " << b->baseDelay << "\n" << endmsg;
 */
	}
	if (b->lastbeta != *b->betaRatio * b->lastfreq) {	/* Set delays
								   if changed */
		b->lastbeta = *b->betaRatio * b->lastfreq;
		DLineL_setDelay(&b->bridgeDelay, b->lastbeta);	/* bow to 
								   bridge length */
		DLineL_setDelay(&b->neckDelay,	/* bow to nut (finger)
						   length */
				b->baseDelay * (1.0f - *b->betaRatio));
	}
	b->v_rate = *b->vibFreq * b->vibr->flen / esr;
	if (b->kloop > 0 && b->h.insdshead->relesing)
		b->kloop = 1;
	if ((--b->kloop) == 0) {
		ADSR_setDecayRate(&b->adsr, (1.0f - b->adsr.value) * 0.005f);
		b->adsr.target = 0.0f;
		b->adsr.rate = b->adsr.releaseRate;
		b->adsr.state = RELEASE;
	}
	do {
		float bowVelocity;
		float bridgeRefl = 0.0f, nutRefl = 0.0f;
		float newVel = 0.0f, velDiff = 0.0f, stringVel = 0.0f;
		float lastOutput;

		bowVelocity = maxVel * ADSR_tick(&b->adsr);
/* info << "bowVelocity=" << bowVelocity << "\n" << endmsg;
 */

		bridgeRefl = -OnePole_tick(&b->reflFilt,
					   b->bridgeDelay.lastOutput);	/* Bridge Reflection      */
/* info << "bridgeRefl=" << bridgeRefl << "\n" << endmsg;
 */
		nutRefl = -b->neckDelay.lastOutput;	/* Nut Reflection 
							 */
/* info << "nutRefl=" << nutRefl << "\n" << endmsg;
 */
		stringVel = bridgeRefl + nutRefl;	/* Sum is String
							   Velocity */
/* info << "stringVel=" << stringVel  << "\n" << endmsg;
 */
		velDiff = bowVelocity - stringVel;	/* Differential
							   Velocity  */
/* info << "velDiff=" << velDiff  << "\n" << endmsg;
 */
		newVel = velDiff * BowTabl_lookup(&b->bowTabl, velDiff);
/* Non-Lin Bow Function   */
/* info << "newVel=" << newVel  << "\n" << endmsg;
 */
		DLineL_tick(&b->neckDelay, bridgeRefl + newVel);	/* Do
									   string       */
		DLineL_tick(&b->bridgeDelay, nutRefl + newVel);
		/*   propagations  */

		if (*b->vibAmt > 0.0f) {
			long temp;
			float temp_time, alpha;
			/* Tick on vibrato table */
			b->v_time += b->v_rate;		/*  Update
							   current time    */
			while (b->v_time >= b->vibr->flen)	/*  Check for 
								   end of sound */
				b->v_time -= b->vibr->flen;	/*  loop
								   back to beginning */
			while (b->v_time < 0.0f)	/*  Check for end of
							   sound */
				b->v_time += b->vibr->flen;	/*  loop
								   back to beginning */

			temp_time = b->v_time;

#ifdef phase_offset
			if (b->v_phaseOffset != 0.0f) {
				temp_time += b->v_phaseOffset;	/*  Add
								   phase offset       */
				while (temp_time >= b->vibr->flen)	/* 
									   Check for end of sound */
					temp_time -= b->vibr->flen;	/* 
									   loop back to beginning */
				while (temp_time < 0.0f)	/*  Check for
								   end of sound */
					temp_time += b->vibr->flen;	/* 
									   loop back to beginning */
			}
#endif
			temp = (long) temp_time;	/*  Integer part of
							   time address    */
			/*  fractional part of time address */
			alpha = temp_time - (float) temp;
			b->v_lastOutput = b->vibr->ftable[temp];	/* Do
									   linear interpolation */
			/*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
			b->v_lastOutput = b->v_lastOutput +
			    (alpha * (b->vibr->ftable[temp + 1] -
				      b->v_lastOutput));
			/* End of vibrato tick */

			DLineL_setDelay(&b->neckDelay,
				  (b->baseDelay * (1.0f - b->lastbeta)) +
			  (b->baseDelay * *b->vibAmt * b->v_lastOutput));
		} else
			DLineL_setDelay(&b->neckDelay,
				  (b->baseDelay * (1.0f - b->lastbeta)));

		lastOutput = BiQuad_tick(&b->bodyFilt,
					 b->bridgeDelay.lastOutput);
/* info << "lastOutput=" << lastOutput  << "\n" << endmsg;
 */

		*ar++ = lastOutput * AMP_SCALE * 1.8f;
	} while (--nsmps);
}

/******************************************/
/*  Waveguide Brass Instrument Model ala  */
/*  Cook (TBone, HosePlayer)              */
/*  by Perry R. Cook, 1995-96             */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/******************************************/


/****************************************************************************/
/*                                                                          */
/*  AllPass Interpolating Delay Line Object by Perry R. Cook 1995-96        */
/*  This one uses a delay line of maximum length specified on creation,     */
/*  and interpolates fractional length using an all-pass filter.  This      */
/*  version is more efficient for computing static length delay lines       */
/*  (alpha and coeff are computed only when the length is set, there        */
/*  probably is a more efficient computational form if alpha is changed     */
/*  often (each sample)).                                                   */
/****************************************************************************/

void
make_DLineA(DLineA * p, long max_length)
{
	long i;
	p->length = max_length;
	auxalloc(max_length * sizeof (float), &p->inputs);
	for (i = 0; i < max_length; i++)
		((float *) p->inputs.auxp)[i] = 0.0f;
	p->lastIn = 0.0f;
	p->lastOutput = 0.0f;
	p->inPoint = 0;
	p->outPoint = max_length >> 1;
}

void
DLineA_clear(DLineA * p)
{
	long i;
	for (i = 0; i < p->length; i++)
		((float *) p->inputs.auxp)[i] = 0.0f;
	p->lastIn = 0.0f;
	p->lastOutput = 0.0f;
}

void
DLineA_setDelay(DLineA * p, float lag)
{
	float outputPointer;
	outputPointer = p->inPoint - lag + 2;	/* outPoint chases inpoint 
						 */
	/*   + 2 for interp and other     */
	if (p->length <= 0)
		error << "PERF: DlineA not initialised" << endmsg;

	while (outputPointer < 0)
		outputPointer += p->length;	/* modulo table length       
						 */
	p->outPoint = (long) outputPointer;	/* Integer part of delay     
						 */
	p->alpha = 1.0f + p->outPoint - outputPointer;	/* fractional
							   part of delay */
	if (p->alpha < 0.1) {
		outputPointer += 1.0f;	/*  Hack to avoid pole/zero       
					 */
		p->outPoint += 1;	/*  cancellation.  Keeps allpass  */
		p->alpha += 1.0f;	/*  delay in range of .1 to 1.1   */
	}
	p->coeff = (1.0f - p->alpha) / (1.0f + p->alpha);	/* coefficient 
								   for all pass */
}

float
DLineA_tick(DLineA * p, float sample)
{				/*   Take sample, yield sample */
	float temp;
	((float *) p->inputs.auxp)[p->inPoint++] = sample;	/* Write
								   input sample  */
	if (p->inPoint == p->length)	/* Increment input pointer */
		p->inPoint -= p->length;	/* modulo length           */
	temp = ((float *) p->inputs.auxp)[p->outPoint++];	/* filter
								   input         */
	if (p->outPoint == p->length)	/* Increment output pointer */
		p->outPoint -= p->length;	/* modulo length           */
	p->lastOutput = -p->coeff * p->lastOutput;	/* delayed output     
							 */
	p->lastOutput += p->lastIn + (p->coeff * temp);		/* input 
								   + delayed Input */
	p->lastIn = temp;
	return p->lastOutput;	/* save output and return  */
}

/* ====================================================================== */

/****************************************************************************/
/*  Lip Filter Object by Perry R. Cook, 1995-96                             */
/*  The lip of the brass player has dynamics which are controlled by the    */
/*  mass, spring constant, and damping of the lip.  This filter simulates   */
/*  that behavior and the transmission/reflection properties as well.       */
/*  See Cook TBone and HosePlayer instruments and articles.                 */
/****************************************************************************/

void
make_LipFilt(LipFilt * p)
{
	make_BiQuad(p);
}

void
LipFilt_setFreq(LipFilt * p, float frequency)
{
	float coeffs[2];
	coeffs[0] = 2.0f * 0.997f *
	    (float) cos((double) (twopi * frequency / esr));	/* damping
								   should  */
	coeffs[1] = -0.997f * 0.997f;	/* change with lip */
	BiQuad_setPoleCoeffs(p, coeffs);	/* parameters, but */
	BiQuad_setGain(*p, 0.03f);	/* not yet.        */
}

/*  NOTE:  Here we should add lip tension                 */
/*              settings based on Mass/Spring/Damping     */
/*              Maybe in TookKit97                        */

float
LipFilt_tick(LipFilt * p, float mouthSample, float boreSample)
		/*   Perform "Table Lookup" By Polynomial Calculation */
{
	float temp;
	float output;
	temp = mouthSample - boreSample;	/* Differential pressure        
						 */
	temp = BiQuad_tick(p, temp);	/* Force -> position            */
	temp = temp * temp;	/* Simple position to area mapping */
	if (temp > 1.0f)
		temp = 1.0f;	/* Saturation at + 1.0          */
	output = temp * mouthSample;	/* Assume mouth input = area    */
	output += (1.0f - temp) * boreSample;	/* and Bore reflection is
						   compliment */
	return output;
}

/* ====================================================================== */

void
brassset(BRASS * p)
{
	FUNC *ftp;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */

	if ((ftp = ftfind(p->ifn)) != NULL)
		p->vibr = ftp;
	else
		error << "PERF: No table for Brass" << endmsg;
	/* Expect sine wave
							 */
	p->frq = *p->frequency;	/* Remember */
	if (*p->lowestFreq >= 0.0f) {
		if (*p->lowestFreq != 0.0f)
			p->length = (long) (esr / *p->lowestFreq + 1.0f);
		else if (p->frq != 0.0f)
			p->length = (long) (esr / p->frq + 1.0f);
		else
			error << "PERF: No base frequency for brass" << endmsg;

		make_DLineA(&p->delayLine, p->length);
		make_LipFilt(&p->lipFilter);
		make_DCBlock(&p->dcBlock);
		make_ADSR(&p->adsr);
		ADSR_setAll(&p->adsr, 0.02f, 0.05f, 1.0f, 0.001f);

		ADSR_setAttackRate(&p->adsr, amp * 0.001f);

		p->maxPressure = amp;
		ADSR_keyOn(&p->adsr);

		/* Set frequency */
		/*     p->slideTarget = (esr / p->frq * 2.0f) + 3.0f; */
		/* fudge correction for filter delays */
		DLineA_setDelay(&p->delayLine, p->slideTarget);
		/*  we'll play a harmonic  */
		p->lipTarget = 0.0f;
		/*     LipFilt_setFreq(&p->lipFilter, p->frq); */
		/* End of set frequency */
		p->lipT = 0.0f;
		/*     LipFilt_setFreq(&p->lipFilter, */
		/*                     p->lipTarget * (float)pow(4.0,(2.0*
		   p->lipT) -1.0)); */
		{
			unsigned short relestim = (unsigned short) (ekr *
								    0.1f);	/* 1/10th second decay extention */
			if (relestim > p->h.insdshead->xtratim)
				p->h.insdshead->xtratim = relestim;
		}
		p->kloop = (int) (p->h.insdshead->offtim * ekr) - (int) (ekr *
							    *p->dettack);
		/*     info << "offtim=" << */
		/*            p->h.insdshead->offtim << "  kloop=" << p->kloop << "\n" << endmsg;
 */
	}
}

void
brass(BRASS * p)
{
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */
	float maxPressure = p->maxPressure = amp;
	int v_len = (int) p->vibr->flen;
	float *v_data = p->vibr->ftable;
	float vibGain = *p->vibAmt;
	float vTime = p->v_time;

	p->v_rate = *p->vibFreq * v_len / esr;
	/*   vibr->setFreq(6.137); */
	/* vibrGain = 0.05; *//* breath periodic vibrato component  */
	if (p->kloop > 0 && p->h.insdshead->relesing)
		p->kloop = 1;
	if ((--p->kloop) == 0) {
		ADSR_setReleaseRate(&p->adsr, amp * 0.005f);
		ADSR_keyOff(&p->adsr);
	}
	if (p->frq != *p->frequency) {	/* Set frequency if changed */
		p->frq = *p->frequency;
		p->slideTarget = (esr / p->frq * 2.0f) + 3.0f;
		/* fudge correction for filter delays */
		DLineA_setDelay(&p->delayLine, p->slideTarget);
		/*  we'll play a harmonic */
		p->lipTarget = p->frq;
	}			/* End of set frequency */
	if (*p->liptension != p->lipT) {
		p->lipT = *p->liptension;
		LipFilt_setFreq(&p->lipFilter,
		 p->lipTarget * (float) pow(4.0, (2.0 * p->lipT) - 1.0));
	}
	do {
		float breathPressure;
		float lastOutput;
		int temp;
		float temp_time, alpha;
		float v_lastOutput;

		breathPressure = maxPressure * ADSR_tick(&p->adsr);
		/* Tick on vibrato table */
		vTime += p->v_rate;	/*  Update current time    */
		while (vTime >= v_len)	/*  Check for end of sound */
			vTime -= v_len;		/*  loop back to
						   beginning */
		while (vTime < 0.0f)	/*  Check for end of sound */
			vTime += v_len;		/*  loop back to
						   beginning */

		temp_time = vTime;

#ifdef phase_offset
		if (p->v_phaseOffset != 0.0f) {
			temp_time += p->v_phaseOffset;	/*  Add phase
							   offset       */
			while (temp_time >= v_len)	/*  Check for end of
							   sound */
				temp_time -= v_len;	/*  loop back to
							   beginning */
			while (temp_time < 0.0f)	/*  Check for end of
							   sound */
				temp_time += v_len;	/*  loop back to
							   beginning */
		}
#endif

		temp = (int) temp_time;		/*  Integer part of time 
						   address    */
		/*  fractional part of time address */
		alpha = temp_time - (float) temp;
		v_lastOutput = v_data[temp];	/* Do linear interpolation, 
						   same as */
		v_lastOutput +=	/*alpha*data[temp+1]+(1-alpha)data[temp] 
				 */
		    (alpha * (v_data[temp + 1] - v_lastOutput));
		/* End of vibrato tick */
		breathPressure += vibGain * v_lastOutput;
		lastOutput =
		    DLineA_tick(&p->delayLine,	/* bore delay  */
				DCBlock_tick(&p->dcBlock,	/* block DC    
								 */
					     LipFilt_tick(&p->lipFilter,
						   0.3f * breathPressure,
		/* mouth input */
				      0.85f * p->delayLine.lastOutput)));
		/* and bore reflection */
		*ar++ = lastOutput * AMP_SCALE * 3.5f;
	} while (--nsmps);

	p->v_time = vTime;
}




static OENTRY opcodes[] =
{
	{"wgclar", S(CLARIN), 5, "a", "kkkiikkkio", F(clarinset), NULL,
	 F(clarin)},
	{"wgflute", S(FLUTE), 5, "a", "kkkiikkkiovv", F(fluteset), NULL,
	 F(flute)},
    {"wgbow", S(BOWED), 5, "a", "kkkkkkio", F(bowedset), NULL, F(bowed)},
  {"wgbrass", S(BRASS), 5, "a", "kkkikkio", F(brassset), NULL, F(brass)},
	{NULL}
};
