#include "cs.h"
#include "entry.h"
#include "moog1.h"
#include "fgens.h"


/********************************************/
/*  Sweepable Formant (2-pole)              */
/*  Filter Class, by Perry R. Cook, 1995-96 */
/*  See books on filters to understand      */
/*  more about how this works.  This drives */
/*  to a target at speed set by rate.       */
/********************************************/

void
make_FormSwep(FormSwep * p)
{
	p->poleCoeffs[0] = p->poleCoeffs[1] = 0.0f;
	p->gain = 1.0f;
	p->freq = p->reson = 0.0f;
	p->currentGain = 1.0f;
	p->currentFreq = p->currentReson = 0.0f;
	p->targetGain = 1.0f;
	p->targetFreq = p->targetReson = 0.0f;
	p->deltaGain = 0.0f;
	p->deltaFreq = p->deltaReson = 0.0f;
	p->sweepState = 0.0f;
	p->sweepRate = 0.002f;
	p->dirty = 0;
	p->outputs[0] = p->outputs[1] = 0.0f;
}

/* void FormSwep_setFreqAndReson(FormSwep *p, float aFreq, float aReson) */
/* { */
/*     p->dirty = 0; */
/*     p->reson = p->currentReson = aReson; */
/*     p->freq = p->currentFreq = aFreq; */
/*     p->poleCoeffs[1] = - (aReson * aReson); */
/*     p->poleCoeffs[0] = 2.0*aReson*(float)cos((double)(twopi*aFreq/esr)); */
/* } */

void
FormSwep_setStates(FormSwep * p, float aFreq, float aReson, float aGain)
{
	p->dirty = 0;
	p->freq = p->targetFreq = p->currentFreq = aFreq;
	p->reson = p->targetReson = p->currentReson = aReson;
	p->gain = p->targetGain = p->currentGain = aGain;
}

void
FormSwep_setTargets(FormSwep * p, float aFreq, float aReson, float aGain)
{
	p->dirty = 1;
	p->targetFreq = aFreq;
	p->targetReson = aReson;
	p->targetGain = aGain;
	p->deltaFreq = aFreq - p->currentFreq;
	p->deltaReson = aReson - p->currentReson;
	p->deltaGain = aGain - p->currentGain;
	p->sweepState = 0.0f;
}

float
FormSwep_tick(FormSwep * p, float sample)
{				/* Perform Filter Operation */
	float temp;

	if (p->dirty) {
		p->sweepState += p->sweepRate;
		if (p->sweepState >= 1.0f) {
			p->sweepState = 1.0f;
			p->dirty = 0;
			p->currentReson = p->targetReson;
			p->reson = p->targetReson;
			p->currentFreq = p->targetFreq;
			p->freq = p->targetFreq;
			p->currentGain = p->targetGain;
			p->gain = p->targetGain;
		} else {
			p->currentReson = p->reson + (p->deltaReson *
						      p->sweepState);
			p->currentFreq = p->freq + (p->deltaFreq *
						    p->sweepState);
			p->currentGain = p->gain + (p->deltaGain *
						    p->sweepState);
		}
		p->poleCoeffs[1] = -(p->currentReson * p->currentReson);
		p->poleCoeffs[0] = 2.0f * p->currentReson *
		    (float) cos((double) (twopi * p->currentFreq / esr));
	}
	temp = p->currentGain * sample;
	temp += p->poleCoeffs[0] * p->outputs[0];
	temp += p->poleCoeffs[1] * p->outputs[1];
	p->outputs[1] = p->outputs[0];
	p->outputs[0] = temp;
	return temp;
}

float
Samp_tick(Wave * p)
{
	long temp, temp1;
	float temp_time, alpha;
	float lastOutput;

	p->time += p->rate;	/*  Update current time    */
	while (p->time >= p->wave->flen)	/*  Check for end of sound */
		p->time -= p->wave->flen;	/*  loop back to beginning */
	while (p->time < 0.0)	/*  Check for end of sound */
		p->time += p->wave->flen;	/*  loop back to beginning */

	temp_time = p->time;

	if (p->phase != 0.0) {
		temp_time += p->phase;	/*  Add phase offset       */
		while (temp_time >= p->wave->flen)	/*  Check for end of
							   sound */
			temp_time -= p->wave->flen;	/*  loop back to
							   beginning */
		while (temp_time < 0.0)		/*  Check for end of
						   sound */
			temp_time += p->wave->flen;	/*  loop back to
							   beginning */
	}
	temp = (long) temp_time;	/*  Integer part of time address    */
	temp1 = temp + 1;
	if (temp1 == p->wave->flen)
		temp1 = 0;	/* Wrap!! */
#ifdef DEBUG
	info << "Samp_tick: in (" << temp << "," << temp1 << ")\n" << endmsg;

#endif
	/*  fractional part of time address */
	alpha = temp_time - (float) temp;
	lastOutput = p->wave->ftable[temp];	/* Do linear interpolation
						 */
#ifdef DEBUG
	info << "Part 1=" << lastOutput << "\n" << endmsg;

#endif
	/* same as alpha*data[temp+1] + (1-alpha)data[temp] */
	lastOutput += (alpha * (p->wave->ftable[temp + 1] - lastOutput));
	/* End of vibrato tick */
	return lastOutput;
}


void
Moog1set(MOOG1 * p)
{
	FUNC *ftp;
	float tempCoeffs[2] =
	{0.0f, -1.0f};

	make_ADSR(&p->adsr);
	make_OnePole(&p->filter);
	make_TwoZero(&p->twozeroes[0]);
	TwoZero_setZeroCoeffs(&p->twozeroes[0], tempCoeffs);
	make_TwoZero(&p->twozeroes[1]);
	TwoZero_setZeroCoeffs(&p->twozeroes[1], tempCoeffs);
	make_FormSwep(&p->filters[0]);
	make_FormSwep(&p->filters[1]);

	if ((ftp = ftfind(p->iatt)) != NULL)
		p->attk.wave = ftp;	/* mandpluk */
	if ((ftp = ftfind(p->ifn)) != NULL)
		p->loop.wave = ftp;	/* impuls20 */
	if ((ftp = ftfind(p->ivfn)) != NULL)
		p->vibr.wave = ftp;	/* sinewave */
	p->attk.time = p->attk.phase = 0.0f;
	p->loop.time = p->loop.phase = 0.0f;
	p->vibr.time = p->vibr.phase = 0.0f;
	p->oldfilterQ = p->oldfilterRate = 0.0f;
	ADSR_setAll(&p->adsr, 0.05f, 0.00003f, 0.6f, 0.0002f);
	ADSR_keyOn(&p->adsr);
}

static float last = 0.0f;

void
Moog1(MOOG1 * p)
{
	float amp = *p->amp * AMP_RSCALE;	/* Normalised */
	float *ar = p->ar;
	long nsmps = ksmps;
	float temp;
	float vib = *p->vibAmt;

	p->baseFreq = *p->frequency;
	p->attk.rate = p->baseFreq * 0.01f * p->attk.wave->flen / esr;
	p->loop.rate = p->baseFreq * p->loop.wave->flen / esr;
	p->attackGain = amp * 0.5f;
	p->loopGain = amp;
	if (*p->filterQ != p->oldfilterQ) {
		p->oldfilterQ = *p->filterQ;
		temp = p->oldfilterQ + 0.05f;
		FormSwep_setStates(&p->filters[0], 2000.0f, temp, 2.0f * (1.0f
								- temp));
		FormSwep_setStates(&p->filters[1], 2000.0f, temp, 2.0f * (1.0f
								- temp));
		temp = p->oldfilterQ + 0.099f;
		FormSwep_setTargets(&p->filters[0], 0.0f, temp, 2.0f * (1.0f -
								  temp));
		FormSwep_setTargets(&p->filters[1], 0.0f, temp, 2.0f * (1.0f -
								  temp));
	}
	if (*p->filterRate != p->oldfilterRate) {
		p->oldfilterRate = *p->filterRate;
		p->filters[0].sweepRate = p->oldfilterRate * RATE_NORM;
		p->filters[1].sweepRate = p->oldfilterRate * RATE_NORM;
	}
	p->vibr.rate = *p->vibf * p->vibr.wave->flen / esr;
/*     info << "   (" << */
/*             last << "): modDepth=" << vib << "\tfilterQ=" << *p->filterQ << "\tfilterRate=" << *p->filterRate << "\n" << endmsg;
 */
/*     info << "FormSwep: poleCoeffs=" << */
/*            p->filters[0].poleCoeffs[0] << ", " << p->filters[0].poleCoeffs[1] << "\tfreq=" << */
/*            p->filters[0].freq << "\treson=" << p->filters[0].reson << "\n" << endmsg;
 */
/*     info << "        : dirty=" << */
/*            p->filters[0].dirty << "\ttargetFreq=" << p->filters[0].targetFreq << "\ttargetReson=" << */
/*            p->filters[0].targetReson << "\n" << endmsg;
 */
/*     info << "        : targetGain=" << */
/*            p->filters[0].targetGain << "\tcurrentFreq=" << p->filters[0].currentFreq << "\tcurrentReson=" << */
/*            p->filters[0].currentReson << "\n" << endmsg;
 */
/*     info << "        : currentGain=" << */
/*            p->filters[0].currentGain << "\tdeltaFreq=" << p->filters[0].deltaFreq << "\tdeltaReson=" << */
/*            p->filters[0].deltaReson << "\n" << endmsg;
 */
/*     info << "        : deltaGain=" << */
/*            p->filters[0].deltaGain << "\tsweepState=" << p->filters[0].sweepState << "\tsweepRate=" << */
/*            p->filters[0].sweepRate << "\n" << endmsg;
 */
/*     info << "FormSwep: poleCoeffs=" << */
/*            p->filters[1].poleCoeffs[0] << ", " << p->filters[1].poleCoeffs[1] << "\tfreq=" << */
/*            p->filters[1].freq << "\treson=" << p->filters[1].reson << "\n" << endmsg;
 */
/*     info << "        : dirty=" << */
/*            p->filters[1].dirty << "\ttargetFreq=" << p->filters[1].targetFreq << "\ttargetReson=" << */
/*            p->filters[1].targetReson << "\n" << endmsg;
 */
/*     info << "        : targetGain=" << */
/*            p->filters[1].targetGain << "\tcurrentFreq=" << p->filters[1].currentFreq << "\tcurrentReson=" << */
/*            p->filters[1].currentReson << "\n" << endmsg;
 */
/*     info << "        : currentGain=" << */
/*            p->filters[1].currentGain << "\tdeltaFreq=" << p->filters[1].deltaFreq << "\tdeltaReson=" << */
/*            p->filters[1].deltaReson << "\n" << endmsg;
 */
/*     info << "        : deltaGain=" << */
/*            p->filters[1].deltaGain << "\tsweepState=" << p->filters[1].sweepState << "\tsweepRate=" << */
/*            p->filters[1].sweepRate << "\n" << endmsg;
 */
/*     info << "TwoZero: " << */
/*            p->twozeroes[0].zeroCoeffs[0] << " " << p->twozeroes[0].zeroCoeffs[1] << "\n" << endmsg;
 */
/*     info << "TwoZero: " << */
/*            p->twozeroes[1].zeroCoeffs[0] << " " << p->twozeroes[1].zeroCoeffs[1] << "\n" << endmsg;
 */

	do {
		float temp;
		float output;
		long itemp;
		float temp_time, alpha;

		if (vib != 0.0f) {
			temp = vib * Samp_tick(&p->vibr);
			p->loop.rate = p->baseFreq * (1.0f + temp) *
			    (float) (p->loop.wave->flen) / esr;
		}
		p->attk.time += p->attk.rate;	/*  Update current time    
						 */
#ifdef DEBUG
		info << "Attack_time=" << p->attk.time << "\tAttack_rate=" << p->attk.rate << "\n" << endmsg;

#endif
		temp_time = p->attk.time;
		if (p->attk.time >= (float) p->attk.wave->flen)
			output = 0.0f;	/* One shot */
		else {
			itemp = (long) temp_time;	/*  Integer part of
							   time address    */
			/*  fractional part of time address */
			alpha = temp_time - (float) itemp;
#ifdef DEBUG
			info << "Attack: (" << itemp << ", " << itemp + 1 << "), alpha=" << alpha << "\t" << endmsg;

#endif
			output = p->attk.wave->ftable[itemp];	/* Do
								   linear interpolation */
			/*  same as alpha*data[itemp+1] + (1-alpha)data[itemp]
			 */
#ifdef DEBUG
			info << "->" << output << "+\n" << endmsg;

#endif
			output += (alpha * (p->attk.wave->ftable[itemp + 1] -
					    output));
			output *= p->attackGain;
			/* End of attack tick */
		}
#ifdef DEBUG
		info << "After Attack: " << output << "\n" << endmsg;

#endif
		output += p->loopGain * Samp_tick(&p->loop);
#ifdef DEBUG
		info << "Before OnePole: " << output << "\n" << endmsg;

#endif
		output = OnePole_tick(&p->filter, output);
#ifdef DEBUG
		info << "After OnePole: " << output << "\n" << endmsg;

#endif
		output *= ADSR_tick(&p->adsr);
#ifdef DEBUG
		info << "Sampler_tick: " << output << "\n" << endmsg;

#endif
		output = TwoZero_tick(&p->twozeroes[0], output);
#ifdef DEBUG
		info << "TwoZero0_tick: " << output << "\n" << endmsg;

#endif
		output = FormSwep_tick(&p->filters[0], output);
#ifdef DEBUG
		info << "Filters0_tick: " << output << "\n" << endmsg;

#endif
		output = TwoZero_tick(&p->twozeroes[1], output);
#ifdef DEBUG
		info << "TwoZero1_tick: " << output << "\n" << endmsg;

#endif
		output = FormSwep_tick(&p->filters[1], output);
#ifdef DEBUG
		info << "Filter2_tick: " << output << "\n" << endmsg;

#endif
		last = output;
		*ar++ = output * AMP_SCALE * 8.0f;
	} while (--nsmps);
}




static OENTRY opcodes[] =
{
    {"moog", S(MOOG1), 5, "a", "kkkkkkiii", F(Moog1set), NULL, F(Moog1)},
	{NULL}
};
