/*******************************************/
/*  4 Resonance Modal Synthesis Instrument */
/*  by Perry R. Cook, 1995-96              */
/*  This instrument contains an excitation */
/*  wavetable, an envelope, and four reso- */
/*  nances (Non-Sweeping BiQuad Filters).  */
/*******************************************/

#include <math.h>
#include <stdlib.h>

#include "cs.h"
#include "entry.h"
#include "fgens.h"
#include "insert.h"
#include "modal4.h"
#include "marimba.h"
#include "vibraphn.h"

void
make_Modal4(Modal4 * m, float *ifn, float vgain, float vrate)
{
	FUNC *ftp;

	if ((ftp = ftfind(ifn)) != NULL)
		m->vibr = ftp;
	else
		error << "PERF: No table for Modal4 case" << endmsg;
	/* Expect sine
								   wave */
	make_Envelope(&m->envelope);
	/*  We don't make the excitation wave here yet,   */
	/*  because we don't know what it's going to be.  */
	make_BiQuad(&m->filters[0]);
	make_BiQuad(&m->filters[1]);
	make_BiQuad(&m->filters[2]);
	make_BiQuad(&m->filters[3]);
	make_OnePole(&m->onepole);

	m->v_rate = vrate;	/* 6.0; */
	m->vibrGain = vgain;	/* 0.05; */

/*     m->directGain = 0.0; */
	m->masterGain = 1.0f;
/*     m->baseFreq = 440.0; */
	/*     Modal4_setRatioAndReson(m, 0, 1.00, 0.9997); *//*  Set some      
	 */
	/*     Modal4_setRatioAndReson(m, 1, 1.30, 0.9997); *//*  silly         
	 */
	/*     Modal4_setRatioAndReson(m, 2, 1.77, 0.9997); *//*  default       
	 */
	/*     Modal4_setRatioAndReson(m, 3, 2.37, 0.9997); *//*  values here   
	 */
/*     Modal4_setFiltGain(m, 0, 0.01); */
/*     Modal4_setFiltGain(m, 1, 0.01); */
/*     Modal4_setFiltGain(m, 2, 0.01); */
/*     Modal4_setFiltGain(m, 3, 0.01); */
/*     OnePole_clear(&m->onepole); */
	BiQuad_clear(&m->filters[0]);
	BiQuad_clear(&m->filters[1]);
	BiQuad_clear(&m->filters[2]);
	BiQuad_clear(&m->filters[3]);
	BiQuad_setEqualGainZeroes(m->filters[0]);
	BiQuad_setEqualGainZeroes(m->filters[1]);
	BiQuad_setEqualGainZeroes(m->filters[2]);
	BiQuad_setEqualGainZeroes(m->filters[3]);
/*     stickHardness = 0.5; */
/*     strikePosition = 0.561; */
}

/* void Modal4_clear(Modal4 *m) */
/* {     */
/*     OnePole_clear(&m->onepole); */
/*     BiQuad_clear(&m->filters[0]); */
/*     BiQuad_clear(&m->filters[1]); */
/*     BiQuad_clear(&m->filters[2]); */
/*     BiQuad_clear(&m->filters[3]); */
/* } */

void
Modal4_setFreq(Modal4 * m, float frequency)
{
	m->baseFreq = frequency;
	Modal4_setRatioAndReson(m, 0, m->ratios[0], m->resons[0]);
	Modal4_setRatioAndReson(m, 1, m->ratios[1], m->resons[1]);
	Modal4_setRatioAndReson(m, 2, m->ratios[2], m->resons[2]);
	Modal4_setRatioAndReson(m, 3, m->ratios[3], m->resons[3]);
}

#include <stdio.h>

void
Modal4_setRatioAndReson(Modal4 * m, int whichOne, float ratio, float reson)
{
	float temp;
	if (ratio * m->baseFreq < esr * 0.5f) {
		m->ratios[whichOne] = ratio;
	} else {
		temp = ratio;
		while (temp * m->baseFreq > esr / 2.0f)
			temp *= 0.5f;
		m->ratios[whichOne] = temp;
/*     info << "Modal4 : Aliasing would occur here, correcting.\n" << endmsg;
 */
	}
	m->resons[whichOne] = reson;
	if (ratio < 0)
		temp = -ratio;
	else
		temp = ratio * m->baseFreq;
	BiQuad_setFreqAndReson(m->filters[whichOne], temp, reson);
}

void
Modal4_strike(Modal4 * m, float amplitude)
{
	int i;
	float temp;
	Envelope_setRate(&m->envelope, 1.0f);
	Envelope_setTarget(&m->envelope, amplitude);
	OnePole_setPole(&m->onepole, 1.0f - amplitude);
	Envelope_tick(&m->envelope);
	m->w_time = 0.0f;
	m->w_lastOutput = 0.0f;
	m->w_allDone = 0;
	/*     wave->reset(); */
	for (i = 0; i < 4; i++) {
		if (m->ratios[i] < 0)
			temp = -m->ratios[i];
		else
			temp = m->ratios[i] * m->baseFreq;
		BiQuad_setFreqAndReson(m->filters[i], temp, m->resons[i]);
	}
}


void
Modal4_damp(Modal4 * m, float amplitude)
{
	int i;
	float temp;
	for (i = 0; i < 4; i++) {
		if (m->ratios[i] < 0)
			temp = -m->ratios[i];
		else
			temp = m->ratios[i] * m->baseFreq;
		BiQuad_setFreqAndReson(m->filters[i], temp, m->resons[i] *
				       amplitude);
	}
}

float
Modal4_tick(Modal4 * m)
{
	float temp, temp2;
	long itemp;
	float temp_time, alpha;
	int length = (int) m->wave->flen;

	m->w_time += m->w_rate;	/*  Update current time  */
/* info << "Modal4 tick: time=" << m->w_time << " rate=" << m->w_rate << "\n" << endmsg;
   */
	if (m->w_time >= length) {	/*  Check for end of sound       */
		m->w_time = (float) (length - 1);	/*  stick at end       
							 */
		m->w_allDone = 1;	/*  Information for one-shot use */
	} else if (m->w_time < 0.0f)	/*  Check for end of sound       */
		m->w_time = 0.0f;	/*  stick at beg                 */

	temp_time = m->w_time;

#ifdef phase_offset
	if (m->w_phaseOffset != 0.0f) {
		temp_time += m->w_phaseOffset;	/*  Add phase offset      
						 */
		if (temp_time >= length)	/*  Check for end of sound      
						 */
			temp_time = length - 1;		/*  stick at end 
							 */
		else if (temp_time < 0.0f)	/*  check for end of sound    
						 */
			temp_time = 0.0f;	/*  stick at beg               
						 */
	}
#endif

	itemp = (long) temp_time;	/*  Integer part of time address    */
/* info << "temp_time=" << temp_time << "\t" << endmsg;
 */
	alpha = temp_time - (float) itemp;	/*  fractional part of time
						   address */
	m->w_lastOutput = m->wave->ftable[itemp];	/*  Do linear
							   interpolation         */
/* info << "w_last1=" << m->w_lastOutput << "\t" << endmsg;
 */
	m->w_lastOutput = m->w_lastOutput +	/*  same as
						   alpha*data[temp+1]      */
	    (alpha * (m->wave->ftable[itemp + 1] -
		      m->w_lastOutput));	/*  + (1-alpha)data[temp]       
						 */
/* info << "w_last2=" << m->w_lastOutput << "\n" << endmsg;
 */

	temp = m->masterGain *
	    OnePole_tick(&m->onepole, m->w_lastOutput *
			 Envelope_tick(&m->envelope));
/* info << "onepole->" << temp << "\n" << endmsg;
 */
	temp2 = BiQuad_tick(&m->filters[0], temp);
/* info << "BiQ_tick0: " << temp2 << "\n" << endmsg;
 */
	temp2 += BiQuad_tick(&m->filters[1], temp);
/* info << "BiQ_tick1: " << temp2 << "\n" << endmsg;
 */
	temp2 += BiQuad_tick(&m->filters[2], temp);
/* info << "BiQ_tick2: " << temp2 << "\n" << endmsg;
 */
	temp2 += BiQuad_tick(&m->filters[3], temp);
/* info << "BiQ_tick3: " << temp2 << "\n" << endmsg;
 */
	temp2 = temp2 - (temp2 * m->directGain);
	temp2 += m->directGain * temp;
/* info << "Temp2: " << temp2 << "\n" << endmsg;
 */

	if (m->vibrGain != 0.0) {
/* info << "Vibrato gain=" << m->vibrGain << "\n" << endmsg;
 */
		/* Tick on vibrato table */
		m->v_time += m->v_rate;		/*  Update current time  
						 */
		while (m->v_time >= m->vibr->flen)	/*  Check for end of
							   sound */
			m->v_time -= m->vibr->flen;	/*  loop back to
							   beginning */
		while (m->v_time < 0.0f)	/*  Check for end of sound */
			m->v_time += m->vibr->flen;	/*  loop back to
							   beginning */

		temp_time = m->v_time;

#ifdef phase_offset
		if (m->v_phaseOffset != 0.0f) {
			temp_time += m->v_phaseOffset;	/*  Add phase
							   offset       */
			while (temp_time >= m->vibr->flen)	/*  Check for 
								   end of sound */
				temp_time -= m->vibr->flen;	/*  loop
								   back to beginning */
			while (temp_time < 0.0f)	/*  Check for end of
							   sound */
				temp_time += m->vibr->flen;	/*  loop
								   back to beginning */
		}
#endif

		itemp = (long) temp_time;	/*  Integer part of time
						   address    */
		/*  fractional part of time address */
		alpha = temp_time - (float) itemp;
		m->v_lastOutput = m->vibr->ftable[itemp];	/* Do linear
								   interpolation */
		/*  same as alpha*data[itemp+1] + (1-alpha)data[temp] */
		m->v_lastOutput = m->v_lastOutput +
		    (alpha * (m->vibr->ftable[itemp + 1] - m->v_lastOutput));
		/* End of vibrato tick */
		temp = 1.0f + (m->v_lastOutput * m->vibrGain);	/* 
								   Calculate AM           */
		temp2 = temp * temp2;	/*  and apply to master out */
	}
	return (temp2 + temp2);
}

/*******************************************/
/*  Marimba SubClass of Modal4 Instrument, */
/*  by Perry R. Cook, 1995-96              */
/*                                         */
/*   Controls:    stickHardness            */
/*                strikePosition           */
/*                vibFreq                  */
/*                vibAmt                   */
/*******************************************/

void
marimbaset(MARIMBA * p)
{
	Modal4 *m = &(p->m4);
	float temp, temp2;
	int itemp;
	FUNC *ftp;

	if ((ftp = ftfind(p->ifn)) != NULL)
		p->m4.wave = ftp;
	else
		error << "PERF: No table for Marimba strike" << endmsg;
	/* Expect an 
								   impulslything */

/*     { int i; */
/*      for (i=0; i<256; i++) info << "" << i << ":" << p->m4.wave->ftable[i] << "\t" << endmsg;
 */
/*     } */
	make_Modal4(m, p->ivfn, *p->vibAmt, *p->vibFreq);
	p->m4.w_phaseOffset = 0.0f;
/*     p->m4.w_rate = 0.5; */
	Modal4_setRatioAndReson(m, 0, 1.00f, 0.9996f);	/*  Set all       
							   132.0  */
	Modal4_setRatioAndReson(m, 1, 3.99f, 0.9994f);	/*  of our        
							   523.0  */
	Modal4_setRatioAndReson(m, 2, 10.65f, 0.9994f);		/* 
								   default       1405.0 */
	Modal4_setRatioAndReson(m, 3, -18.50f, 0.999f);		/* 
								   resonances    2443.0 */
	Modal4_setFiltGain(m, 0, 0.08f);	/*  and        */
	Modal4_setFiltGain(m, 1, 0.02f);	/*  gains      */
	Modal4_setFiltGain(m, 2, 0.02f);	/*  for each   */
	Modal4_setFiltGain(m, 3, 0.015f);	/*  resonance  */
	p->m4.directGain = 0.1f;
	p->multiStrike = 0;
	p->strikePosition = *p->spos;
	/* Set Stick hardness stuff */
	p->stickHardness = *p->hardness;
/*     info << "Stickhardness=" << p->stickHardness << "\n" << endmsg;
 */
	p->m4.w_rate = (0.25f * (float) pow(4.0, (double) p->stickHardness));
	p->m4.masterGain = (0.1f + (1.8f * p->stickHardness));
/* info << "Hardness=" << */
/*        *p->hardness << ", w_rate=" << p->m4.w_rate << " masterGain=" << p->m4.masterGain << " [" << */
/*        pow(4.0 << "]\n" << (double)p->stickHardness) << endmsg;
 */
	/* Set Strike position */
	temp2 = p->strikePosition * pi;
	temp = (float) sin((double) temp2);
	BiQuad_setGain(p->m4.filters[0], 0.12f * temp);		/* 1st
								   mode function of pos. */
	temp = (float) sin(0.05 + (3.9 * (double) temp2));
	BiQuad_setGain(p->m4.filters[1], -0.03f * temp);	/* 2nd mode
								   function of pos. */
	temp = (float) sin(-0.05 + (11.0 * (double) temp2));
	BiQuad_setGain(p->m4.filters[2], 0.11f * temp);		/* 3rd
								   mode function of pos. */
	/* Strike */
	itemp = rand() % 5;
	if (itemp < 2) {
		p->multiStrike = 1;
		info << "striking twice here!!\n" << endmsg;

	} else if (itemp < 1) {
		p->multiStrike = 2;
		info << "striking three times here!!!\n" << endmsg;

	} else
		p->multiStrike = 0;
	Modal4_strike(m, *p->amplitude * AMP_RSCALE);
	Modal4_setFreq(m, *p->frequency);
/*     { int i; */
/*      for (i=0; i<256; i++) info << "" << i << ":" << p->m4.wave->ftable[i] << "\t" << endmsg;
 */
/*     } */
	{
		unsigned short relestim =
		(unsigned short) (ekr * *p->dettack);	/* 1/10th second
							   decay extention */
		if (relestim > p->h.insdshead->xtratim)
			p->h.insdshead->xtratim = relestim;
	}
	p->kloop = (int) ((p->h.insdshead->offtim * ekr) - (int) (ekr *
							   *p->dettack));
}


void
marimba(MARIMBA * p)
{
	Modal4 *m = &(p->m4);
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amplitude) * AMP_RSCALE;	/* Normalise */

	if (p->kloop > 0 && p->h.insdshead->relesing)
		p->kloop = 1;
	if ((--p->kloop) == 0) {
/*       info << "Start damp\n" << endmsg;
 */
		Modal4_damp(m, 1.0f - (amp * 0.03f));
	}
	p->m4.v_rate = *p->vibFreq;	/* 6.0; */
	p->m4.vibrGain = *p->vibAmt;	/* 0.05; */

	do {
		float lastOutput;
		if (p->multiStrike > 0)
			if (p->m4.w_allDone) {
				p->m4.w_time = 0.0f;
				p->m4.w_lastOutput = 0.0f;
				p->m4.w_allDone = 0;
				p->multiStrike -= 1;
			}
		lastOutput = Modal4_tick(m);
/*       info << "Sample=" << lastOutput << "\n" << endmsg;
 */
		*ar++ = lastOutput * AMP_SCALE * 0.6f;
	} while (--nsmps);
}

/*******************************************/
/*  Vibraphone SubClass of Modal4          */
/*  Instrument, by Perry R. Cook, 1995-96  */
/*                                         */
/*   Controls:    CONTROL1 = stickHardness */
/*                CONTROL2 = strikePosition */
/*                CONTROL3 = vibFreq       */
/*                MOD_WHEEL= vibAmt        */
/*******************************************/

void
vibraphnset(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	float temp;
	FUNC *ftp;

	if ((ftp = ftfind(p->ifn)) != NULL)
		p->m4.wave = ftp;
	else
		error << "PERF: No table for Vibraphone strike" << endmsg;
	/* Expect 
								   an impulslything */

	make_Modal4(m, p->ivfn, *p->vibAmt, *p->vibFreq);

	p->m4.w_phaseOffset = 0.0f;
/*     p->m4.w_rate = 13.33; */
	OnePole_setPole(&p->m4.onepole, 0.2f);
	Modal4_setRatioAndReson(m, 0, 1.0f, 0.99995f);	/*  Set        
							 */
	Modal4_setRatioAndReson(m, 1, 2.01f, 0.99991f);		/*  our  
								 */
	Modal4_setRatioAndReson(m, 2, 3.9f, 0.99992f);	/*  resonance  
							 */
	Modal4_setRatioAndReson(m, 3, 14.37f, 0.99990f);	/*  values here 
								 */
	Modal4_setFiltGain(m, 0, 0.025f);
	Modal4_setFiltGain(m, 1, 0.015f);
	Modal4_setFiltGain(m, 2, 0.015f);
	Modal4_setFiltGain(m, 3, 0.015f);
	p->m4.directGain = 0.0f;
/*     vibrGain = 0.2; */
	p->m4.w_rate = 2.0f + (22.66f * *p->hardness);
	p->m4.masterGain = 0.2f + (*p->hardness * 1.6f);
/* info << "Hardness=" << */
/*        *p->hardness << ", w_rate=" << p->m4.w_rate << " masterGain=" << p->m4.masterGain << "\n" << endmsg;
 */
	/* Set Strike position */
	temp = p->strikePosition * pi;
	BiQuad_setGain(p->m4.filters[0], 0.025f * (float) sin((double) temp));
	BiQuad_setGain(p->m4.filters[1], 0.015f * (float) sin(0.1 + (2.01 *
							(double) temp)));
	BiQuad_setGain(p->m4.filters[2], 0.015f * (float) sin(3.95 * (double)
							      temp));
	/* Strike */
	Modal4_strike(m, *p->amplitude * AMP_RSCALE);
	Modal4_setFreq(m, *p->frequency);
}

void
vibraphn(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amplitude) * AMP_RSCALE;	/* Normalise */

	if (p->kloop > 0 && p->h.insdshead->relesing)
		p->kloop = 1;
	if ((--p->kloop) == 0) {
/*       info << "Start damp\n" << endmsg;
 */
		Modal4_damp(m, 1.0f - (amp * 0.03f));
	}
	Modal4_setFreq(m, *p->frequency);
	p->m4.v_rate = *p->vibFreq;
	p->m4.vibrGain = *p->vibAmt;
	do {
		float lastOutput;
		lastOutput = Modal4_tick(m);
/*       info << "Sample=" << lastOutput << "\n" << endmsg;
 */
		*ar++ = lastOutput * 8.0f * AMP_SCALE;	/* Times 8 as
							   seems too quiet */
	} while (--nsmps);
}


/*******************************************/
/*  AgogoBell SubClass of Modal4 Instrument */
/*  by Perry R. Cook, 1995-96              */
/*                                         */
/*   Controls:    CONTROL1 = stickHardness */
/*                CONTROL2 = strikePosition */
/*                CONTROL3 = vibFreq       */
/*                MOD_WHEEL= vibAmt        */
/*******************************************/


/*   Modes measured from my Agogo Bell by FFT:  */
/*   360, 1470, 2401, 4600                      */


void
agogobelset(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	FUNC *ftp;

	if ((ftp = ftfind(p->ifn)) != NULL)
		p->m4.wave = ftp;
	else
		error << "PERF: No table for Agogobell strike" << endmsg;
	/* Expect
								   an impulslything */

	make_Modal4(m, p->ivfn, *p->vibAmt, *p->vibFreq);

	p->m4.w_phaseOffset = 0.0f;
/*     p->m4.w_rate = 7.0; */
	OnePole_setPole(&p->m4.onepole, 0.2f);
	Modal4_setRatioAndReson(m, 0, 1.00f, 0.999f);	/*  Set         */
	Modal4_setRatioAndReson(m, 1, 4.08f, 0.999f);	/*  our         */
	Modal4_setRatioAndReson(m, 2, 6.669f, 0.999f);	/*  resonance  
							 */
	Modal4_setRatioAndReson(m, 3, -3725.0f, 0.999f);	/*  values here 
								 */
	Modal4_setFiltGain(m, 0, 0.07f);
	Modal4_setFiltGain(m, 1, 0.06f);
	Modal4_setFiltGain(m, 2, 0.04f);
	Modal4_setFiltGain(m, 3, 0.02f);
	p->m4.directGain = 0.3f;
/*     vibrGain = 0.2; */
	p->m4.w_rate = 3.0f + (8.0f * *p->hardness);
	p->m4.masterGain = 1.0f;
/* info << "Hardness=" << */
/*        *p->hardness << ", w_rate=" << p->m4.w_rate << " masterGain=" << p->m4.masterGain << "\n" << endmsg;
 */
	/* Set Strike position */
/*     temp = p->strikePosition * pi; */
/*     info << "gain0=" << 0.08 * sin(0.7 * temp) << "\n" << endmsg;
 */
/*     BiQuad_setGain(p->m4.filters[0], 0.08 * sin(0.7 * temp)); */
/*     info << "gain1=" << 0.07 * sin(0.1 + (5.0 * temp)) << "\n"" << endmsg;
 */
/*     BiQuad_setGain(p->m4.filters[1], 0.07 * sin(0.1 + (5.0 * temp))); */
/*     info << "gain2=" << 0.04 * sin(0.2 + (7.0 * temp)) << "\n" << endmsg;
 */
/*     BiQuad_setGain(p->m4.filters[2], 0.04 * sin(0.2 + (7.0 * temp))); */
	/* Strike */
	Modal4_strike(m, *p->amplitude * AMP_RSCALE);
	Modal4_setFreq(m, *p->frequency);
}

void
agogobel(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	float *ar = p->ar;
	long nsmps = ksmps;

	p->m4.v_rate = *p->vibFreq;
	p->m4.vibrGain = *p->vibAmt;
	do {
		float lastOutput;
		lastOutput = Modal4_tick(m);
/*       info << "Sample=" << lastOutput << "\n" << endmsg;
 */
		*ar++ = lastOutput * AMP_SCALE;
	} while (--nsmps);
}



static OENTRY opcodes[] =
{
	{"marimba", S(MARIMBA), 5, "a", "kkiiikkii", F(marimbaset), NULL,
	 F(marimba)},
	{"vibes", S(VIBRAPHN), 5, "a", "kkiiikkii", F(vibraphnset), NULL,
	 F(vibraphn)},
      {"agogobel", S(VIBRAPHN), 5, "a", "kkiiikki", F(agogobelset), NULL,
       F(agogobel)},
	{NULL}
};
