/* sliders and other MIDI opcodes by Gabriel Maldonado */

#include <math.h>
#include "cs.h"
#include "entry.h"
#include "midi_cs.h"
#include "midirecv.h"
#include "midiops.h"
#include "midiops3.h"
#include "fgens.h"
#include "insert.h"

#define f7bit  127.0f
#define oneTOf7bit (float)(1./127.)
#define f14bit  16383.0f
#define oneTOf14bit  (float)(1/16383.)
#define f21bit 2097151.0f
#define oneTOf21bit	 (float)(1/2097151.)
void dieu_gab(char *s, char *title);	/* gab */
void midi_in_error(char *op);

void
midi_in_error(char *op)
{
	fatal << "An error occurred while allocating '" << op << "' "
	      "opcode: MIDI IN PORT not enabled.\n"
	      "For using '" << op << "' opcode you must set '-K' "
	      "flag in command line!" << endmsg;

}

/*--------------------------------------------------------*/

void
slider_init8(SLIDER8 * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider8");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLD *sld = p->s;
		register char *slnum = p->slnum;
		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 8) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;

			value = (*(sld++)->initvalue - *min) / (*max++ -
								*min++);
			chanblock[*slnum++] = (float) ((int) (value * f7bit +
							      .5));
		}
	}
}

void
slider8(SLIDER8 * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;

	while (j++ < 8) {
		value = (float) (chanblock[*slnum++] * oneTOf7bit);
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));	/* no interpolation */
		**result++ = value * (*max++ - *min) + *min++;	/* scales 
								   the output */
	}
}


void
slider_init16(SLIDER16 * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider16");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLD *sld = p->s;
		register char *slnum = p->slnum;
		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 16) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				error << "INIT: illegal control number" << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;

			value = (*(sld++)->initvalue - *min) / (*max++ -
								*min++);
			chanblock[*slnum++] = (float) ((int) (value * f7bit +
							      .5));
		}
	}
}


void
slider16(SLIDER16 * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;

	while (j++ < 16) {
		value = (float) (chanblock[*slnum++] * oneTOf7bit);
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));	/* no interpolation */
		**result++ = value * (*max++ - *min) + *min++;	/* scales 
								   the output */
	}
}




void
slider_init32(SLIDER32 * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider32");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLD *sld = p->s;
		register char *slnum = p->slnum;
		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 32) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;

			value = (*(sld++)->initvalue - *min) / (*max++ -
								*min++);
			chanblock[*slnum++] = (float) ((int) (value * f7bit +
							      .5));
		}
	}
}


void
slider32(SLIDER32 * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;

	while (j++ < 32) {
		value = (float) (chanblock[*slnum++] * oneTOf7bit);
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));	/* no interpolation */
		**result++ = value * (*max++ - *min) + *min++;	/* scales 
								   the output */
	}
}



void
slider_init64(SLIDER64 * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider64");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLD *sld = p->s;
		register char *slnum = p->slnum;
		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 64) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;

			value = (*(sld++)->initvalue - *min) / (*max++ -
								*min++);
			chanblock[*slnum++] = (float) (int) (value * f7bit +
							     .5f);
		}
	}
}


void
slider64(SLIDER64 * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;

	while (j++ < 64) {
		value = chanblock[*slnum++] * oneTOf7bit;
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));	/* no interpolation */
		**result++ = value * (*max++ - *min) + *min++;	/* scales 
								   the output */
	}
}

/*==============================*/
void
slider_init8f(SLIDER8f * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider8f");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLDf *sld = p->s;
		register char *slnum = p->slnum;
		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float b;
		register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;
		while (j++ < 8) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;

			value = (*sld->initvalue - *min) / (*max++ - *min++);
			chanblock[*slnum++] = (float) (int) (value * f7bit +
							     .5);

/*----- init filtering coeffs*/
			*yt1++ = 0;
			b = (float) (2. - cos((double) (*(sld++)->ihp * tpidsr
							* ksmps)));
			*c2 = (float) (b - sqrt((double) (b * b - 1.)));
			*c1++ = 1.0f - *c2++;

		}
	}
}

void
slider8f(SLIDER8f * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;
	register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

	while (j++ < 8) {
		value = chanblock[*slnum++] * oneTOf7bit;
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));
		value = value * (*max++ - *min) + *min++;	/* scales the
								   output */
		**result++ = *yt1++ = *c1++ * value + *c2++ * *yt1;	/*
									   filters the output */

		/* yt1++; c1++;  c2++; */
	}
}


void
slider_init16f(SLIDER16f * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider16f");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLDf *sld = p->s;
		register char *slnum = p->slnum;


		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float b;
		register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 16) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;
			value = (*sld->initvalue - *min) / (*max++ - *min++);

			chanblock[*slnum++] = (float) (int) (value * f7bit +
							     .5);

/*----- init filtering coeffs*/
			*yt1++ = 0;
			b = (float) (2. - cos((double) (*(sld++)->ihp * tpidsr
							* ksmps)));
			*c2 = (float) (b - sqrt((double) (b * b - 1.)));
			*c1++ = 1.0f - *c2++;
		}
	}
}

void
slider16f(SLIDER16f * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;
	register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

	while (j++ < 16) {
		value = chanblock[*slnum++] * oneTOf7bit;
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));
		value = value * (*max++ - *min) + *min++;	/* scales the
								   output */
		**result++ = *yt1++ = *c1++ * value + *c2++ * *yt1;	/*
									   filters the output */
	}
}


void
slider_init32f(SLIDER32f * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider32f");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLDf *sld = p->s;
		register char *slnum = p->slnum;


		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float b;
		register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 32) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;
			value = (*sld->initvalue - *min) / (*max++ - *min++);

			chanblock[*slnum++] = (float) (int) (value * f7bit +
							     .5);

/*----- init filtering coeffs*/
			*yt1++ = 0;
			b = (float) (2. - cos((double) (*(sld++)->ihp * tpidsr
							* ksmps)));
			*c2 = (float) (b - sqrt((double) (b * b - 1.)));
			*c1++ = 1.0f - *c2++;
		}
	}
}

void
slider32f(SLIDER32f * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;
	register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

	while (j++ < 32) {
		value = chanblock[*slnum++] * oneTOf7bit;
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));
		value = value * (*max++ - *min) + *min++;	/* scales the
								   output */
		**result++ = *yt1++ = *c1++ * value + *c2++ * *yt1;	/*
									   filters the output */
	}
}


void
slider_init64f(SLIDER64f * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider64f");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register SLDf *sld = p->s;
		register char *slnum = p->slnum;


		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;

		register float b;
		register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 64) {
			if ((*slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;
			value = (*sld->initvalue - *min) / (*max++ - *min++);

			chanblock[*slnum++] = (float) (int) (value * f7bit +
							     .5);

/*----- init filtering coeffs*/
			*yt1++ = 0;
			b = (float) (2. - cos((double) (*(sld++)->ihp * tpidsr
							* ksmps)));
			*c2 = (float) (b - sqrt((double) (b * b - 1.)));
			*c1++ = 1.0f - *c2++;
		}
	}
}

void
slider64f(SLIDER64f * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum = p->slnum;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;
	register float *yt1 = p->yt1, *c1 = p->c1, *c2 = p->c2;

	while (j++ < 64) {
		value = chanblock[*slnum++] * oneTOf7bit;
		if (*(++ftp))	/* if valid ftable,use value as index   */
			value = *((*ftp)->ftable + (long) (value *
							   (*ftp)->flen));
		value = value * (*max++ - *min) + *min++;	/* scales the
								   output */
		**result++ = *yt1++ = *c1++ * value + *c2++ * *yt1;	/*
									   filters the output */
	}
}



/*===================================*/

void
islider8(ISLIDER8 * p)
{
	char chan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("islider8");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register ISLD *sld = p->s;
		register char slnum;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;
		register FUNC *ftp;
		register float **result = p->r;

		while (j++ < 8) {
			if ((slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			value = chanblock[slnum] * oneTOf7bit;

			if (*sld->ifn > 0) {
				ftp = ftfind(sld->ifn);
				value = *(ftp->ftable + (long) (value *
							     ftp->flen));	/* no interpolation */
			}
			**result++ = value * (*sld->imax - *sld->imin) +
			    *sld->imin;		/* scales the output */
			sld++;
		}
	}
}


void
islider16(ISLIDER16 * p)
{
	char chan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("islider16");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register ISLD *sld = p->s;
		register char slnum;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;
		register FUNC *ftp;
		register float **result = p->r;

		while (j++ < 16) {
			if ((slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			value = chanblock[slnum] * oneTOf7bit;

			if (*sld->ifn > 0) {
				ftp = ftfind(sld->ifn);
				value = *(ftp->ftable + (long) (value *
							     ftp->flen));	/* no interpolation */
			}
			**result++ = value * (*sld->imax - *sld->imin) +
			    *sld->imin;		/* scales the output */
			sld++;
		}
	}
}


void
islider32(ISLIDER32 * p)
{
	char chan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("islider32");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register ISLD *sld = p->s;
		register char slnum;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;
		register FUNC *ftp;
		register float **result = p->r;

		while (j++ < 32) {
			if ((slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			value = chanblock[slnum] * oneTOf7bit;

			if (*sld->ifn > 0) {
				ftp = ftfind(sld->ifn);
				value = *(ftp->ftable + (long) (value *
							     ftp->flen));	/* no interpolation */
			}
			**result++ = value * (*sld->imax - *sld->imin) +
			    *sld->imin;		/* scales the output */
			sld++;
		}
	}
}


void
islider64(ISLIDER64 * p)
{
	char chan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("islider64");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register ISLD *sld = p->s;
		register char slnum;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;
		register FUNC *ftp;
		register float **result = p->r;

		while (j++ < 64) {
			if ((slnum = (char) *sld->ictlno) < 0) {
				error << "INIT: illegal control number at position "
				      "n." << j << endmsg;

				break;
			}
			value = chanblock[slnum] * oneTOf7bit;

			if (*sld->ifn > 0) {
				ftp = ftfind(sld->ifn);
				value = *(ftp->ftable + (long) (value *
							     ftp->flen));	/* no interpolation */
			}
			**result++ = value * (*sld->imax - *sld->imin) +
			    *sld->imin;		/* scales the output */
			sld++;
		}
	}
}


/*-------------------------------*/

void
slider_init16bit14(SLIDER16BIT14 * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider16");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int intvalue, j = 0;
		register SLD14 *sld = p->s;
		register char *slnum_msb = p->slnum_msb;
		register char *slnum_lsb = p->slnum_lsb;
		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 16) {
			if ((*slnum_msb = (char) *sld->ictlno_msb) < 0) {
				error << "INIT: illegal msb control number at"
				      " position n." << j << endmsg;

				break;
			}
			if ((*slnum_lsb = (char) *sld->ictlno_lsb) < 0) {
				error << "INIT: illegal lsb control number at"
				      " position n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;

			intvalue = (int) (((*(sld++)->initvalue - *min) /
				       (*max++ - *min++)) * f14bit + .5);
			chanblock[*slnum_msb++] = (float) (intvalue >> 7);
			chanblock[*slnum_lsb++] = (float) (intvalue & 0x7f);
		}
	}
}


void
slider16bit14(SLIDER16BIT14 * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum_msb = p->slnum_msb;
	register char *slnum_lsb = p->slnum_lsb;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;

	while (j++ < 16) {
		value = (float) ((chanblock[*slnum_msb++] * 128
			       + chanblock[*slnum_lsb++]) * oneTOf14bit);
		if (*(++ftp)) {	/* if valid ftable,use value as index  
				 */
			register float phase = value * (*ftp)->flen;
			register float *base = (*ftp)->ftable + (long)
			(phase);;
			value = *base + (*(base + 1) - *base) * (phase - (long)
								 phase);
		}
		**result++ = value * (*max++ - *min) + *min++;	/* scales 
								   the output */
	}
}


void
slider_init32bit14(SLIDER32BIT14 * p)
{
	char chan = p->slchan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("slider32");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int intvalue, j = 0;
		register SLD14 *sld = p->s;
		register char *slnum_msb = p->slnum_msb;
		register char *slnum_lsb = p->slnum_lsb;
		register float *min = p->min, *max = p->max;
		register FUNC **ftp = p->ftp;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;

		while (j++ < 32) {
			if ((*slnum_msb = (char) *sld->ictlno_msb) < 0) {
				error << "INIT: illegal msb control number at"
				      " position n." << j << endmsg;

				break;
			}
			if ((*slnum_lsb = (char) *sld->ictlno_lsb) < 0) {
				error << "INIT: illegal lsb control number at"
				      " position n." << j << endmsg;

				break;
			}
			if ((value = *sld->initvalue) < (*min = *sld->imin) ||
			    value > (*max = *sld->imax)) {
				error << "INIT: illegal initvalue at position"
				      " n." << j << endmsg;

				break;
			}
			if (*sld->ifn > 0)
				*ftp++ = ftfind(sld->ifn);
			else
				*ftp++ = NULL;

			intvalue = (int) (((*(sld++)->initvalue - *min) /
				       (*max++ - *min++)) * f14bit + .5);
			chanblock[*slnum_msb++] = (float) (intvalue >> 7);
			chanblock[*slnum_lsb++] = (float) (intvalue & 0x7f);
		}
	}
}


void
slider32bit14(SLIDER32BIT14 * p)
{
	register float value;
	register int j = 0;
	register FUNC **ftp = p->ftp - 1;
	register float *chanblock = (float *) m_chnbp[p->slchan]->ctl_val;
	register char *slnum_msb = p->slnum_msb;
	register char *slnum_lsb = p->slnum_lsb;
	register float *min = p->min, *max = p->max;
	register float **result = p->r;

	while (j++ < 32) {
		value = (chanblock[*slnum_msb++] * 128
			 + chanblock[*slnum_lsb++]) * oneTOf14bit;
		if (*(++ftp)) {	/* linear interpolation routine */
			register float phase = value * (*ftp)->flen;
			register float *base = (*ftp)->ftable + (long)
			(phase);;
			value = *base + (*(base + 1) - *base) * (phase - (long)
								 phase);
		}
		/* if valid ftable,use value as index   */

		**result++ = value * (*max++ - *min) + *min++;	/* scales 
								   the output */
	}
}

/*--------------------------------*/
void
islider16bit14(ISLIDER16BIT14 * p)
{
	char chan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("islider16");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register ISLD14 *sld = p->s;
		register char slnum_msb;
		register char slnum_lsb;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;
		register float **result = p->r;

		while (j++ < 16) {
			if ((slnum_msb = (char) *sld->ictlno_msb) < 0) {
				error << "INIT: illegal msb control number at"
				      " position n." << j << endmsg;

				break;
			}
			if ((slnum_lsb = (char) *sld->ictlno_lsb) < 0) {
				error << "INIT: illegal lsb control number at"
				      " position n." << j << endmsg;

				break;
			}
			value = (float) ((chanblock[slnum_msb] * 128
				  + chanblock[slnum_lsb]) * oneTOf14bit);
			if (*sld->ifn > 0) {	/* linear interpolation
						   routine */
				register FUNC *ftp = ftfind(sld->ifn);
				register float phase = value * ftp->flen;
				register float *base = ftp->ftable + (long)
				(phase);
				value = *base + (*(base + 1) - *base) * (phase
							 - (long) phase);
			}
			**result++ = value * (*sld->imax - *sld->imin) +
			    *sld->imin;		/* scales the output */
			sld++;
		}
	}
}

void
islider32bit14(ISLIDER32BIT14 * p)
{
	char chan = (char) ((*p->ichan) - 1);
	if (midi_in_p_num < 0) {
		midi_in_error("islider32");
		return;
	}
	if (chan < 0 || chan > 15) {
		error << "INIT: illegal channel" << endmsg;

		return;
	} {
		register float value;
		register int j = 0;
		register ISLD14 *sld = p->s;
		register char slnum_msb;
		register char slnum_lsb;
		register float *chanblock = (float *) m_chnbp[chan]->ctl_val;
		register float **result = p->r;

		while (j++ < 32) {
			if ((slnum_msb = (char) *sld->ictlno_msb) < 0) {
				error << "INIT: illegal msb control number at"
				      " position n." << j << endmsg;

				break;
			}
			if ((slnum_lsb = (char) *sld->ictlno_lsb) < 0) {
				error << "INIT: illegal lsb control number at"
				      " position n." << j << endmsg;

				break;
			}
			value = (chanblock[slnum_msb] * 128
				 + chanblock[slnum_lsb]) * oneTOf14bit;
			if (*sld->ifn > 0) {
				/* linear interpolation routine */
				register FUNC *ftp = ftfind(sld->ifn);
				register float phase = value * ftp->flen;
				register float *base = ftp->ftable + (long)
				(phase);
				value = *base + (*(base + 1) - *base) * (phase
							 - (long) phase);
			}
			**result++ = value * (*sld->imax - *sld->imin) +
			    *sld->imin;		/* scales the output */
			sld++;
		}
	}
}
OENTRY opcodes[] =
{
	{"slider8", S(SLIDER8), 3, "kkkkkkkk",
	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(slider_init8), F(slider8), NULL},
	{"slider8f", S(SLIDER8f), 3, "kkkkkkkk",
	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(slider_init8f), F(slider8f), NULL},
	{"islider8", S(SLIDER8), 1, "iiiiiiii",
	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(islider8), NULL, NULL},
	{"slider16", S(SLIDER16), 3, "kkkkkkkkkkkkkkkk",
	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(slider_init16), F(slider16), NULL},
	{"slider16f", S(SLIDER16f), 3, "kkkkkkkkkkkkkkkk",
	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(slider_init16f), F(slider16f), NULL},
	{"islider16", S(SLIDER16), 1, "iiiiiiiiiiiiiiii",
	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(islider16), NULL, NULL},
	{"slider32", S(SLIDER32), 3, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",

	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(slider_init32), F(slider32), NULL},
	{"slider32f", S(SLIDER32f), 3, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",

	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(slider_init32f), F(slider32f), NULL},
	{"islider32", S(SLIDER32), 1, "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",

	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(islider32), NULL, NULL},
	{"slider64", S(SLIDER64), 3,
      "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",
	 "iiiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiii", F(slider_init64), F(slider64), NULL},
	{"slider64f", S(SLIDER64f), 3, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk"
	 "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",
	 "iiiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiiiiiiiiiii"
      "iiiiiiiiiiiiiiiiiiiiiiii", F(slider_init64f), F(slider64f), NULL},
	{"islider64", S(SLIDER64), 1,
      "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",

	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"

	 "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
	 ,F(islider64), NULL, NULL},
	{"s16b14", S(SLIDER16BIT14), 3, "kkkkkkkkkkkkkkkk",
	 "iiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii"
      "iiiiiiiiiiiiiiii", F(slider_init16bit14), F(slider16bit14), NULL},
      {"s32b14", S(SLIDER32BIT14), 3, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",
       "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
       "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
       "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
       "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
       "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
       "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(slider_init32bit14),
       F(slider32bit14), NULL},
	{"is16b14", S(ISLIDER16BIT14), 1, "iiiiiiiiiiiiiiii",
	 "iiiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii"
	 "iiiiiiiiiiiiiiii", F(islider16bit14), NULL, NULL},
    {"is32b14", S(ISLIDER32BIT14), 1, "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
     "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
     "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
     "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
     "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
     "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", F(islider32bit14), NULL, NULL},
	{NULL}
};
