/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: ugens5.cc,v 1.1 1999/11/01 04:29:07 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include "ugens5.h"

void porset(PORT *p)
{
	p->c2 = (float)pow(0.5, (double)onedkr / *p->ihtim);
	p->c1 = 1.0f - p->c2;
	if (*p->isig >= 0.0f)
	    p->yt1 = *p->isig;
}

void port(PORT *p)
{
	*p->kr = p->yt1 = p->c1 * *p->ksig + p->c2 * p->yt1;
}

void tonset(TONE *p)
{
	p->c1 = p->prvhp = 0.0f;
	p->c2 = 1.0f;
	if (!(*p->istor))
	    p->yt1 = 0.0f;
}

void tone(TONE *p)
{
	float	*ar, *asig;
	int		nsmps = ksmps;
	float       c1 = p->c1, c2 = p->c2;
	float       yt1 = p->yt1;

	if (*p->khp != p->prvhp) {
		float b;
		p->prvhp = *p->khp;
		b = 2.0f - (float)cos((double)(p->prvhp * tpidsr));
		p->c2 = c2 = b - (float)sqrt((double)(b * b - 1.0));
		p->c1 = c1 = 1.0f - c2;
	}
	ar = p->ar;
	asig = p->asig;
	do  {
		*ar++ = yt1 = c1 * *asig++ + c2 * yt1;
	} while (--nsmps);
	p->yt1 = yt1;
}

/* From Gabriel Maldonado, modified for arbitrary order */

void tonsetx(TONEX *p) 
{
	p->c1 = p->prvhp = 0.0f;
	p->c2 = 1.0f;

	if ((p->loop = (int) (*p->ord + .5)) < 1) {
		p->loop = 4;
	}

	if (!*p->istor && (p->aux.auxp == NULL ||
			   (size_t) (p->loop*sizeof(float)) > p->aux.size)) {
		p->aux.alloc ((size_t)(p->loop*sizeof(float)));
	}

	p->yt1 = (float*)p->aux.auxp;

	if (!(*p->istor)) {
		int j;
		for (j=0; j< p->loop; j++) p->yt1[j] = 0.0f;
	}
}

void tonex(TONEX *p)	 /* From Gabriel Maldonado, modifies */
{
	float b;
	int j;
	int nsmps;
	float *asig, *ar, c1, c2, *yt1;
	if (*p->khp != p->prvhp) {	  
		p->prvhp = *p->khp;
		b = 2.0f - (float)cos((double)(*p->khp * tpidsr));
		p->c2 = b - (float)sqrt((double)(b * b - 1.));
		p->c1 = 1.0f - p->c2;
	}							
	c1 = p->c1;
	c2 = p->c2;
	yt1= p->yt1;
	asig = p->asig;
	for (j=0; j< p->loop; j++) {
		nsmps = ksmps;
		ar = p->ar;
		do *ar++ = *yt1 = c1 * *asig++ + c2 * *yt1;			
		while (--nsmps);
		yt1++;
		asig = p->ar;
	}
}
 
void atone(TONE *p)
{
	float	*ar, *asig;
	int	nsmps = ksmps;
	float	c1 = p->c1;  /* Not used */
	float   c2 = p->c2, yt1 = p->yt1;

	if (*p->khp != p->prvhp) {
		float b;
		p->prvhp = *p->khp;
		b = 2.0f - (float)cos((double)(*p->khp * tpidsr));
		p->c2 = c2 = b - (float)sqrt((double)(b * b - 1.0));
		p->c1 = c1 = 1.0f - c2;
	}
	ar = p->ar;
	asig = p->asig;
	do {
		*ar++ = yt1 = c2 * (yt1 + *asig);
		yt1 -= *asig++;		/* yt1 contains yt1-xt1 */
	} while (--nsmps);
	p->yt1 = yt1;
}

void atonex(TONEX *p)	  /* Gavriel Maldonado, modified */
{
	float	*ar, *asig;
	float c2, *yt1;
	int	nsmps, j;

	if (*p->khp != p->prvhp) {
		float b;
		p->prvhp = *p->khp;
		b = 2.0f - (float)cos((double)(*p->khp * tpidsr));
		p->c2 = b - (float)sqrt((double)(b * b - 1.0));
		/*p->c1 = 1. - p->c2;*/
	}

	c2 = p->c2;	
	yt1=p->yt1;	
	asig = p->asig;
	for (j=0; j< p->loop; j++) {
		nsmps = ksmps;
		ar = p->ar;
		do {
			*ar++ = *yt1 = c2 * (*yt1 + *asig);
			*yt1 -= *asig++; /* yt1 contains yt1-xt1 */
		} while (--nsmps);
		yt1++;
		asig= p->ar;
	}

}

void rsnset(RESON *p)
{
	int scale;
	p->scale = scale = (int)*p->iscl;
	if (scale && scale != 1 && scale != 2) {
		p->error ("illegal reson iscl value, %f",
			       *p->iscl);
	}
	p->prvcf = p->prvbw = -100.0f;
	if (!(*p->istor))
	    p->yt1 = p->yt2 = 0.0f;
}

void rsnsetx(RESONX *p) /* Gabriel Maldonado, modifies for arb order */
{
	int scale;
	p->scale = scale = (int) *p->iscl;
	if((p->loop = (int) (*p->ord + .5)) < 1) p->loop = 4; /*default value*/
	if (!*p->istor && (p->aux.auxp == NULL ||
			   (size_t)(p->loop*2*sizeof(float)) > p->aux.size)) {
		p->aux.alloc ((size_t)(p->loop*2*sizeof(float)));
	}
	p->yt1 = (float*)p->aux.auxp; p->yt2 = (float*)p->aux.auxp + p->loop;
	if (scale && scale != 1 && scale != 2) {
	        p->error ("illegal reson iscl value, %f",
			       *p->iscl);
	}
	p->prvcf = p->prvbw = -100.0f;

	if (!(*p->istor)) {
		int j;
		for (j=0; j< p->loop; j++) p->yt1[j] = p->yt2[j] = 0.0f;
	}

}

void reson(RESON *p)
{
	int	flag = 0, nsmps = ksmps;
	float	*ar, *asig;
	float	c3p1, c3t4, omc3, c2sqr;
	float	yt1, yt2, c1 = p->c1, c2 = p->c2, c3 = p->c3;

	if (*p->kcf != p->prvcf) {
		p->prvcf = *p->kcf;
		p->cosf = (float)cos((double)(p->prvcf * tpidsr));
		flag = 1;
	}
	if (*p->kbw != p->prvbw) {
		p->prvbw = *p->kbw;
		c3 = p->c3 = (float)exp((double)(p->prvbw * mtpdsr));
		flag = 1;
	}
	if (flag) {
		c3p1 = c3 + 1.0f;
		c3t4 = c3 * 4.0f;
		omc3 = 1.0f - c3;
		c2 = p->c2 = c3t4 * p->cosf / c3p1;		/* -B, so + below */
		c2sqr = c2 * c2;
		if (p->scale == 1) {
		    c1 = p->c1 = omc3 *
			    (float)sqrt(1.0 - (double)c2sqr / (double)c3t4);
		} else if (p->scale == 2) {
		    c1 = p->c1 = (float)sqrt((double)((c3p1*c3p1-c2sqr) * 
						      omc3/c3p1));
		} else {
			c1 = p->c1 = 1.0f;
		}
	}
	asig = p->asig;
	ar = p->ar;
	yt1 = p->yt1; yt2 = p->yt2;
	do {
		float yt0 = c1 * *asig++ + c2 * yt1 - c3 * yt2;
		*ar++ = yt0;
		yt2 = yt1;
		yt1 = yt0;
	} while (--nsmps);
	p->yt1 = yt1; p->yt2 = yt2;	/* Write back for next cycle */
}

void resonx(RESONX *p) /* Gabriel Maldonado, modified  */
{
	int	flag = 0, nsmps, j;
	float	*ar, *asig;
	float	c3p1, c3t4, omc3, c2sqr;
	float *yt1, *yt2, c1,c2,c3; 

	if (*p->kcf != p->prvcf) {
		p->prvcf = *p->kcf;
		p->cosf = (float) cos((double)(*p->kcf * tpidsr));
		flag = 1;
	}
	if (*p->kbw != p->prvbw) {
		p->prvbw = *p->kbw;
		p->c3 = (float) exp((double)(*p->kbw * mtpdsr));
		flag = 1;
	}
	if (flag) {
		c3p1 = p->c3 + 1.0f;
		c3t4 = p->c3 * 4.0f;
		omc3 = 1.0f - p->c3;
		p->c2 = c3t4 * p->cosf / c3p1;		/* -B, so + below */
		c2sqr = p->c2 * p->c2;
		if (p->scale == 1)
		    p->c1 = omc3 * (float)sqrt(1.0 - (double)(c2sqr / c3t4));
		else if (p->scale == 2)
		    p->c1 = (float)sqrt((double)((c3p1*c3p1-c2sqr) * omc3/c3p1));
		else p->c1 = 1.0f;
	}
	
	ar = p->ar;
	c1=p->c1; 
	c2=p->c2;
	c3=p->c3;
	yt1= p->yt1;
	yt2= p->yt2;
	asig = p->asig;
	for (j=0; j< p->loop; j++) {
		nsmps = ksmps;
		ar = p->ar;
		do {
			*ar = c1 * *asig++ + c2 * *yt1 - c3 * *yt2;
			*yt2 = *yt1;
			*yt1 = *ar++;
		} while (--nsmps);
		yt1++;
		yt2++;
		asig= p->ar;
	}
}

void areson(RESON *p)
{
	int	flag = 0, nsmps = ksmps;
	float	*ar, *asig;
	float	c3p1, c3t4, omc3, c2sqr, D = 2.0f; /* 1/RMS = root2 (rand) */
	/*      or 1/.5  (sine) */
	float	yt1, yt2, c1, c2, c3;

	if (*p->kcf != p->prvcf) {
		p->prvcf = *p->kcf;
		p->cosf = (float)cos((double)(*p->kcf * tpidsr));
		flag = 1;
	}
	if (*p->kbw != p->prvbw) {
		p->prvbw = *p->kbw;
		p->c3 = (float)exp((double)(*p->kbw * mtpdsr));
		flag = 1;
	}
	if (flag) {
		c3p1 = p->c3 + 1.0f;
		c3t4 = p->c3 * 4.0f;
		omc3 = 1.0f - p->c3;
		p->c2 = c3t4 * p->cosf / c3p1;
		c2sqr = p->c2 * p->c2;

		if (p->scale == 1) { /* i.e. 1 - A(reson) */
		    p->c1 = 1.0f - omc3 *
			    (float)sqrt((double)1. - c2sqr / c3t4);
		} else if (p->scale == 2) { /* i.e. D - A(reson) */
			p->c1 = D - (float)sqrt((double)((c3p1*c3p1-c2sqr) *
							 omc3/c3p1));
		} else {
			p->c1 = 0.0f; /* can't tell        */
		}
	}
	asig = p->asig;
	ar = p->ar;
	c1 = p->c1; c2 = p->c2; c3 = p->c3; yt1 = p->yt1; yt2 = p->yt2;
	if (p->scale == 1 || p->scale == 0) {
		do {
			*ar = c1 * *asig + c2 * yt1 - c3 * yt2;
			yt2 = yt1;
			yt1 = *ar++ - *asig++;	/* yt1 contains yt1-xt1 */
		} while (--nsmps);
	}
	else if (p->scale == 2) {
		do {
			*ar = c1 * *asig + c2 * yt1 - c3 * yt2;
			yt2 = yt1;
			yt1 = *ar++ - D * *asig++;	/* yt1 contains yt1-D*xt1 */
		} while (--nsmps);
	}
	p->yt1 = yt1; p->yt2 = yt2;
}


void rmsset(RMS *p)
{
        float	b;

	b = 2.0f - (float)cos((double)(*p->ihp * tpidsr));
	p->c2 = b - (float)sqrt((double)(b * b - 1.0));
	p->c1 = 1.0f - p->c2;
	if (!*p->istor)
	    p->prvq = 0.0f;
}

void gainset(GAIN *p)
{
        float	b;

	b = 2.0f - (float)cos((double)(*p->ihp * tpidsr));
	p->c2 = b - (float)sqrt((double)(b * b - 1.0));
	p->c1 = 1.0f - p->c2;
	if (!*p->istor)
	    p->prvq = p->prva = 0.0f;
}

void balnset(BALANCE *p)
{
        float	b;

	b = 2.0f - (float)cos((double)(*p->ihp * tpidsr));
	p->c2 = b - (float)sqrt((double)(b * b - 1.0));
	p->c1 = 1.0f - p->c2;
	if (!*p->istor)
	    p->prvq = p->prvr = p->prva = 0.0f;
}

void rms(RMS *p)
{
        int	nsmps = ksmps;
	float	*asig;
	float	q;
	float	c1 = p->c1, c2 = p->c2;

	q = p->prvq;
	asig = p->asig;
	do {
		float as = *asig++;
		q = c1 * as * as + c2 * q;
	} while (--nsmps);
	p->prvq = q;
	*p->kr = (float) sqrt((double)q);
}

void gain(GAIN *p)
{
        int	nsmps = ksmps;
	float	*ar, *asig;
	float	q, a, m, diff, inc;
	float   c1 = p->c1, c2 = p->c2;

	q = p->prvq;
	asig = p->asig;
	do {
		float as = *asig++;
		q = c1 * as * as + c2 * q;
	} while (--nsmps);
	p->prvq = q;
	if ((q = (float)sqrt(q))) {
	    a = *p->krms / q;
	} else	{
		a = *p->krms;
	}
	asig = p->asig;
	ar = p->ar;
	nsmps = ksmps;
	if ((diff = a - p->prva) != 0) {
		m = p->prva;
		inc = diff/ksmps;
		do {	*ar++ = *asig++ * m;
		m += inc;
		} while (--nsmps);
		p->prva = a;
	}
	else {	do *ar++ = *asig++ * a;
		while (--nsmps);
	}
}

void balance(BALANCE *p)
{
        int	nsmps = ksmps;
	float	*ar, *asig, *csig;
	float	q, r, a, m, diff, inc;
	float	c1 = p->c1, c2 = p->c2;

	q = p->prvq;
	r = p->prvr;
	asig = p->asig;
	csig = p->csig;
	do {
		float as = *asig++;
		float cs = *csig++;
		q = c1 * as * as + c2 * q;
		r = c1 * cs * cs + c2 * r;
	} while (--nsmps);
	p->prvq = q;
	p->prvr = r;
	if (q)
	    a = (float)sqrt(r/q);
	else	a = (float)sqrt(r);
	asig = p->asig;
	ar = p->ar;
	nsmps = ksmps;
	if ((diff = a - p->prva) != 0) {
		m = p->prva;
		inc = diff/ksmps;
		do {	*ar++ = *asig++ * m;
		m += inc;
		} while (--nsmps);
		p->prva = a;
	}
	else {	do *ar++ = *asig++ * a;
		while (--nsmps);
	}
}

Opcode opcodes[] = {

	UGENS5_OPCODE_LIST,
	{ NULL }
};
