/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: ugens4.cc,v 1.1 1999/11/01 04:29:07 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/function_tables.h>
#include "ugens4.h"

void
bzzset(BUZZ * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->ifn)) != 0) {
		p->ftp = ftp;
		if (*p->iphs >= 0)
			p->lphs = (long) (*p->iphs * 0.5f * fmaxlen);
		p->ampcod = p->nth_arg_is_audio (0);
		p->cpscod = p->nth_arg_is_audio (1);
	} else {
		if (is_string_ptr (p->ifn)) {
			p->error ("buzz: no such function "
				       "table \"%s\"", get_string (p->ifn));
		} else {
			p->error ("buzz: no such function "
				       "table %d", (int) p->ifn);
		}
	}
}

void
buzz(BUZZ * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, *ampp, *cpsp, *ftbl;
	long phs, inc, lobits, dwnphs, tnp1, lenmask, nn;
	Number sicvt2, over2n, scal, num, denom;
	int n;

	ftp = p->ftp;
	ftbl = ftp->ftable;
	sicvt2 = sicvt * 0.5f;	/* for theta/2  */
	lobits = ftp->lobits;
	lenmask = ftp->lenmask;
	ampp = p->xamp;
	cpsp = p->xcps;
	if ((n = (int) *p->knh) <= 0) { /* fix n = knh */
		n = 1;
	}
	tnp1 = (n << 1) + 1;	/* calc 2n + 1 */
	over2n = 0.5f / n;
	scal = *ampp * over2n;
	inc = (long) (*cpsp * sicvt2);
	ar = p->ar;
	phs = p->lphs;
	nn = ksmps;
	do {
		dwnphs = phs >> lobits;
		denom = *(ftbl + dwnphs);
		if (denom > .0002 || denom < -.0002f) {
			num = *(ftbl + (dwnphs * tnp1 & lenmask));
			*ar++ = (num / denom - 1.0f) * scal;
		} else
			*ar++ = *ampp;
		phs += inc;
		phs &= PMASK;
		if (p->ampcod)
			scal = *(++ampp) * over2n;
		if (p->cpscod)
			inc = (long) (*(++cpsp) * sicvt2);
	}
	while (--nn);
	p->lphs = phs;
}

void
gbzset(GBUZZ * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->ifn)) != 0) {
		p->ftp = ftp;
		if (*p->iphs >= 0) {
			p->lphs = (long) (*p->iphs * fmaxlen);
			p->prvr = 0.0f;
		}
		p->ampcod = p->nth_arg_is_audio (0);
		p->cpscod = p->nth_arg_is_audio (1);
	}
}

void
gbuzz(GBUZZ * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, *ampp, *cpsp, *ftbl;
	long phs, inc, lobits, lenmask, k, km1, kpn, kpnm1, nn;
	long n;
	Number r, absr, num, denom, scal;

	ftp = p->ftp;
	if (ftp == NULL) {
		p->error ("gbuzz: not initialized");
	}
	ftbl = ftp->ftable;
	lobits = ftp->lobits;
	lenmask = ftp->lenmask;
	ampp = p->xamp;
	cpsp = p->xcps;
	k = (long) *p->kk;	/* fix k and n  */
	if ((n = (long) *p->kn) <= 0) {		/* n must be > 0 */
		error << "PERF: gbuzz knh <= 0" << endmsg;

		return;
	}
	km1 = k - 1;
	kpn = k + n;
	kpnm1 = kpn - 1;
	if ((r = *p->kr) != p->prvr || n != p->prvn) {
		p->twor = r * 2.0f;
		p->rsqp1 = r * r + 1.0f;
		p->rtn = (Number) pow((double) r, (double) n);
		p->rtnp1 = p->rtn * r;
		if ((absr = (Number) fabs(r)) > .999f && absr < 1.001f)
			p->rsumr = 1.0f / n;
		else
			p->rsumr = (1.0f - absr) / (1.0f - (Number)
						    fabs(p->rtn));
		p->prvr = r;
		p->prvn = (short) n;
	}
	scal = *ampp * p->rsumr;
	inc = (long) (*cpsp * sicvt);
	ar = p->ar;
	nn = ksmps;
	do {
		phs = p->lphs >> lobits;
		denom = p->rsqp1 - p->twor * *(ftbl + phs);
		if (denom > .0002f || denom < -.0002f) {
			num = *(ftbl + (phs * k & lenmask))
			    - r * *(ftbl + (phs * km1 & lenmask))
			    - p->rtn * *(ftbl + (phs * kpn & lenmask))
			    + p->rtnp1 * *(ftbl + (phs * kpnm1 & lenmask));
			*ar++ = num / denom * scal;
		} else
			*ar++ = *ampp;
		if (p->ampcod)
			scal = p->rsumr * *(++ampp);
		p->lphs += inc;
		p->lphs &= PMASK;
		if (p->cpscod)
			inc = (long) (*(++cpsp) * sicvt);
	}
	while (--nn);
}

#define PLUKMIN 64
static short rand15(void);
static short rand16(void);

void
plukset(PLUCK * p)
{
	int n;
	long npts, iphs;
	char *auxp;
	RCPointer<FunctionTable> ftp;
	Number *ap, *fp;
	Number phs, phsinc;

	if ((npts = (long) (esr / *p->icps)) < PLUKMIN)		/* npts
								   is wavelen in sampls */
		npts = PLUKMIN;	/*  (but at least min size)  */
	if ((auxp = p->auxch.auxp) == NULL ||
	    npts > p->maxpts) {	/* get newspace    */
		p->auxch.alloc ((npts + 1) * sizeof (Number));
		auxp = p->auxch.auxp;
		p->maxpts = npts;	/*      if reqd    */
	}
	ap = (Number *) auxp;	/* as Number array   */
	if (*p->ifn == 0.0)
		for (n = npts; n--;)	/* f0: fill w. rands */
			*ap++ = (Number) rand16() * dv32768;
	else if ((ftp = ftfind(p->ifn)) != 0) {
		fp = ftp->ftable;	/* else from ftable  */
		phs = 0.0f;
		phsinc = (Number) (ftp->flen / npts);
		for (n = npts; n--; phs += phsinc) {
			iphs = (long) phs;
			*ap++ = *(fp + iphs);
		}
	}
	*ap = *(Number *) auxp;	/* last= copy of 1st */
	p->npts = npts;
	p->sicps = (npts * 256.0f + 128.0f) / esr;	/* tuned pitch convt
							 */
	p->phs256 = 0;
	p->method = (short) *p->imeth;
	p->param1 = *p->ipar1;
	p->param2 = *p->ipar2;
	switch (p->method) {
	case 1:		/* ignore any given parameters */
		break;
	case 2:		/* stretch factor: param1 >= 1 */
		if (p->param1 < 1.0f)
			p->error ("illegal stretch factor(param1) value");
		else
			p->thresh1 = (short) (32768.0f / p->param1);
		break;
	case 3:		/* roughness factor: 0 <= param1 <= 1 */
		if (p->param1 < 0.0f || p->param1 > 1.0f)
			p->error ("illegal roughness factor(param1) value");
		p->thresh1 = (short) (32768.0f * p->param1);
		break;
	case 4:		/* rough and stretch factor: 0 <= param1 <= 1,
				   param2 >= 1 */
		if (p->param1 < 0.0f || p->param1 > 1.0f)
			p->error ("illegal roughness factor(param1) value");
		else
			p->thresh1 = (short) (32768.0f * p->param1);
		if (p->param2 < 1.0f)
			p->error ("illegal stretch factor(param2) value");
		else
			p->thresh2 = (short) (32768.0f / p->param2);
		break;
	case 5:		/* weighting coeff's: param1 + param2 <= 1 */
		if (p->param1 + p->param2 > 1)
			p->error ("coefficients too large(param1 + param2)");
		break;
	case 6:		/* ignore any given parameters */
		break;

	default:
		p->error ("unknown method code");
	}
}

void
pluck(PLUCK * p)
{
	Number *ar, *fp;
	long phs256, phsinc, ltwopi, offset;
	int nsmps;
	Number frac, diff;

	if (p->auxch.auxp == NULL) {	/* RWD FIX */
		p->error ("pluck: not initialized");
	}

	ar = p->ar;
	phsinc = (long) (*p->kcps * p->sicps);
	phs256 = p->phs256;
	ltwopi = p->npts << 8;
	nsmps = ksmps;
	do {
		offset = phs256 >> 8;
		fp = (Number *) p->auxch.auxp + offset;	/* lookup
							   position  */
		diff = *(fp + 1) - *fp;
		frac = (Number) (phs256 & 255) / 256.0f;		/*  w.
								   interpolation */
		*ar++ = (*fp + diff * frac) * *p->kamp;		/* 
								   gives output val */
		if ((phs256 += phsinc) >= ltwopi) {
			int nn;
			Number newval, preval;
			phs256 -= ltwopi;	/* at phase wrap,    */
			fp = (Number *) p->auxch.auxp;
			preval = *fp;	/*   copy last pnt   */
			*fp = *(fp + p->npts);	/*     to first,     */
			fp++;	/*   apply smoothing */
			nn = p->npts;	/*     up to npts+1  */
			switch (p->method) {
			case 1:
				do {	/* simple averaging */
					newval = (*fp + preval) / 2.0f;
					preval = *fp;
					*fp++ = newval;
				} while (--nn);
				break;
			case 2:
				do {	/* stretched avrging */
					if (rand15() < p->thresh1) {
						newval = (*fp + preval) / 2.0f;
						preval = *fp;
						*fp++ = newval;
					} else
						preval = *fp++;
				} while (--nn);
				break;
			case 3:
				do {	/* simple drum */
					if (rand15() < p->thresh1)
						newval = -(*fp + preval) /
						    2.0f;
					else
						newval = (*fp + preval) / 2.0f;
					preval = *fp;
					*fp++ = newval;
				} while (--nn);
				break;
			case 4:
				do {	/* stretched drum */
					if (rand15() < p->thresh2) {
						if (rand15() < p->thresh1)
							newval = -(*fp +
							  preval) / 2.0f;
						else
							newval = (*fp + preval)
							    / 2.0f;
						preval = *fp;
						*fp++ = newval;
					} else
						preval = *fp++;
				} while (--nn);
				break;
			case 5:
				do {	/* weighted avraging */
					newval = p->param1 * *fp + p->param2 *
					    preval;
					preval = *fp;
					*fp++ = newval;
				} while (--nn);
				break;
			case 6:
				do {	/* 1st order recursive filter */
					preval = (*fp + preval) / 2.0f;
					*fp++ = preval;
				} while (--nn);
				break;
			}
		}
	}
	while (--nsmps);
	p->phs256 = phs256;
}

#define	RNDMUL	15625L
#define MASK16   0xFFFFL
#define MASK15   0x7FFFL

static short
rand16(void)
{				/* quick generate a random short between
				   -32768 and 32767 */
	static long rand = 1000;
	rand *= RNDMUL;
	rand += 1L;
	rand &= MASK16;
	return ((short) rand);
}

static short
rand15(void)
{				/* quick generate a random short between 0 and 
				   32767 */
	static long rand = 1000;
	rand *= RNDMUL;
	rand += 1L;
	rand &= MASK15;
	return ((short) rand);
}


void
rndset(RAND * p)
{
	if (*p->iseed >= 0)
		p->rand = (short) (*p->iseed * 32768.0f);
	p->ampcod = p->nth_arg_is_audio (0);
}

void
krand(RAND * p)
{
	p->rand *= RNDMUL;
	p->rand += 1;
	*p->ar = (Number) p->rand * *p->xamp * dv32768;
}

void
arand(RAND * p)
{
	Number *ar;
	short rand, rndmul = RNDMUL, n = ksmps;
	Number ampscl;

	rand = p->rand;
	ar = p->ar;
	if (!(p->ampcod)) {
		ampscl = *p->xamp * dv32768;
		do {
			rand *= rndmul;
			rand += 1;
			*ar++ = (Number) rand *ampscl;
		} while (--n);
	} else {
		Number *xamp = p->xamp;
		do {
			rand *= rndmul;
			rand += 1;
			*ar++ = (Number) rand **xamp++ * dv32768;
		} while (--n);
	}
	p->rand = rand;		/* save current rand */
}

void
rhset(RANDH * p)
{
	if (*p->iseed >= 0) {	/* new seed:            */
		p->rand = (short) (*p->iseed * 32768L);		/*  
								   init rand integ    */
		p->phs = 0;	/*      & phs           */
		p->num1 = *p->iseed;	/*      store fnum      */
	}

	p->ampcod = p->nth_arg_is_audio (0);
	p->cpscod = p->nth_arg_is_audio (1);
}

void
krandh(RANDH * p)
{
	*p->ar = p->num1 * *p->xamp;	/* rslt = num * amp     */
	p->phs += (long) (*p->xcps * kicvt);	/* phs += inc           */
	if (p->phs >= MAXLEN) {	/* when phs overflows,  */
		p->phs &= PMASK;	/*      mod the phs     */
		p->rand *= RNDMUL;	/*      & recalc number */
		p->rand += 1;
		p->num1 = (Number) p->rand * dv32768;
	}
}

void
randh(RANDH * p)
{
	long phs = p->phs, inc;
	int n = ksmps;
	Number *ar, *ampp, *cpsp;

	cpsp = p->xcps;
	ampp = p->xamp;
	ar = p->ar;
	inc = (long) (*cpsp++ * sicvt);
	do {
		*ar++ = p->num1 * *ampp;	/* rslt = num * amp */
		if (p->ampcod)
			ampp++;
		phs += inc;	/* phs += inc       */
		if (p->cpscod)
			inc = (long) (*cpsp++ * sicvt);
		if (phs >= MAXLEN) {	/* when phs o'flows, */
			phs &= PMASK;
			p->rand *= RNDMUL;	/*   calc new number */
			p->rand += 1;
			p->num1 = (Number) p->rand * dv32768;
		}
	} while (--n);
	p->phs = phs;
}

void
riset(RANDI * p)
{
	if (*p->iseed >= 0) {	/* new seed:            */
		p->rand = (short) (*p->iseed * 32768.0f);	/*       init
								   rand integ */
		p->rand *= RNDMUL;	/*      to 2nd value    */
		p->rand += 1;
		p->phs = 0;	/*      & clear phs     */
		p->num1 = *p->iseed;	/*      store num1,2    */
		p->num2 = (Number) p->rand * dv32768;
		p->dfdmax = (p->num2 - p->num1) / fmaxlen;	/* & diff     
								 */
	}

	p->ampcod = p->nth_arg_is_audio (0);
	p->cpscod = p->nth_arg_is_audio (1);
}

void
krandi(RANDI * p)
{				/* rslt = (num1 + diff*phs) * amp */
	*p->ar = (p->num1 + (Number) p->phs * p->dfdmax) * *p->xamp;
	p->phs += (long) (*p->xcps * kicvt);	/* phs += inc           */
	if (p->phs >= MAXLEN) {	/* when phs overflows,  */
		p->phs &= PMASK;	/*      mod the phs     */
		p->rand *= RNDMUL;	/*      recalc random   */
		p->rand += 1;
		p->num1 = p->num2;	/*      & new num vals  */
		p->num2 = (Number) p->rand * dv32768;
		p->dfdmax = (p->num2 - p->num1) / fmaxlen;
	}
}

void
randi(RANDI * p)
{
	long phs = p->phs, inc;
	int n = ksmps;
	Number *ar, *ampp, *cpsp;

	cpsp = p->xcps;
	ampp = p->xamp;
	ar = p->ar;
	inc = (long) (*cpsp++ * sicvt);
	do {
		*ar++ = (p->num1 + (Number) phs * p->dfdmax) * *ampp;
		if (p->ampcod)
			ampp++;
		phs += inc;	/* phs += inc       */
		if (p->cpscod)
			inc = (long) (*cpsp++ * sicvt);		/*  
								   (nxt inc)      */
		if (phs >= MAXLEN) {	/* when phs o'flows, */
			phs &= PMASK;
			p->rand *= RNDMUL;	/*   calc new numbers */
			p->rand += 1;
			p->num1 = p->num2;
			p->num2 = (Number) p->rand * dv32768;
			p->dfdmax = (p->num2 - p->num1) / fmaxlen;
		}
	} while (--n);
	p->phs = phs;
}

Opcode opcodes[] =
{
	UGENS4_OPCODE_LIST,
	{NULL}
};
