/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: ugens3.cc,v 1.1 1999/11/01 04:29:07 pbd Exp $
*/

#include <math.h>
#include <stdio.h> /* for sprintf */

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/process.h>
#include <quasimodo/function_tables.h>

#include "ugens3.h"

void
foscset(FOSC * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->ifn)) != 0) {
		p->ftp = ftp;
		if (*p->iphs >= 0)
			p->cphs = p->mphs = (long) (*p->iphs * fmaxlen);
	}
}

void
foscil(FOSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, *ampp, car, fmod, cfreq, mod, ndx, *ftab;
	int32 mphs, cphs, minc, cinc, lobits;
	int16 nsmps = ksmps;

	ar = p->rslt;
	ftp = p->ftp;
	if (ftp == NULL) {
		p->error ("foscil: not initialized");
	}
	ftab = ftp->ftable;
	lobits = ftp->lobits;
	mphs = p->mphs;
	cphs = p->cphs;
	car = *p->kcps * *p->kcar;
	mod = *p->kcps * *p->kmod;
	ndx = *p->kndx * mod;
	ampp = p->xamp;
	minc = (long) (mod * sicvt);
	if (p->has_audio_arguments()) {
		do {
			mphs &= PMASK;
			fmod = *(ftab + (mphs >> lobits)) * ndx;
			mphs += minc;
			cfreq = car + fmod;
			cinc = (long) (cfreq * sicvt);
			cphs &= PMASK;
			*ar++ = *(ftab + (cphs >> lobits)) * *ampp++;
			cphs += cinc;
		}
		while (--nsmps);
	} else {
		Number amp;
		amp = *ampp;
		do {
			mphs &= PMASK;
			fmod = *(ftab + (mphs >> lobits)) * ndx;
			mphs += minc;
			cfreq = car + fmod;
			cinc = (long) (cfreq * sicvt);
			cphs &= PMASK;
			*ar++ = *(ftab + (cphs >> lobits)) * amp;
			cphs += cinc;
		}
		while (--nsmps);
	}
	p->mphs = mphs;
	p->cphs = cphs;
}

void
foscili(FOSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, *ampp, fract, v1, car, fmod, cfreq, mod;
	Number ndx, *ftab;
	int32 mphs, cphs, minc, cinc, lobits;
	int16 nsmps = ksmps;

	ar = p->rslt;
	ftp = p->ftp;
	if (ftp == NULL) {	/* RWD fix */
		p->error ("foscili: not initialized");
	}
	lobits = ftp->lobits;
	mphs = p->mphs;
	cphs = p->cphs;
	car = *p->kcps * *p->kcar;
	mod = *p->kcps * *p->kmod;
	ndx = *p->kndx * mod;
	ampp = p->xamp;
	minc = (long) (mod * sicvt);
	if (p->has_audio_arguments()) {
		do {
			mphs &= PMASK;
			fract = PFRAC(mphs);
			ftab = ftp->ftable + (mphs >> lobits);
			v1 = *ftab++;
			fmod = (v1 + (*ftab - v1) * fract) * ndx;
			mphs += minc;
			cfreq = car + fmod;
			cinc = (long) (cfreq * sicvt);
			cphs &= PMASK;
			fract = PFRAC(cphs);
			ftab = ftp->ftable + (cphs >> lobits);
			v1 = *ftab++;
			*ar++ = (v1 + (*ftab - v1) * fract) * *ampp++;
			cphs += cinc;
		}
		while (--nsmps);
	} else {
		Number amp;
		amp = *ampp;
		do {
			mphs &= PMASK;
			fract = PFRAC(mphs);
			ftab = ftp->ftable + (mphs >> lobits);
			v1 = *ftab++;
			fmod = (v1 + (*ftab - v1) * fract) * ndx;
			mphs += minc;
			cfreq = car + fmod;
			cinc = (long) (cfreq * sicvt);
			cphs &= PMASK;
			fract = PFRAC(cphs);
			ftab = ftp->ftable + (cphs >> lobits);
			v1 = *ftab++;
			*ar++ = (v1 + (*ftab - v1) * fract) * amp;
			cphs += cinc;
		}
		while (--nsmps);
	}
	p->mphs = mphs;
	p->cphs = cphs;
}

void
losset(LOSC * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind (p->ifn, true, true)) != 0) {
		p->ftp = ftp;
		if (*p->ibas != FZERO)
			p->cpscvt = ftp->cvtbas / *p->ibas;
		else if ((p->cpscvt = ftp->cpscvt) == FZERO)
			goto lerr1;
		if ((p->mod1 = (short) *p->imod1) < 0) {
			if ((p->mod1 = ftp->loopmode1) == 0) {
				warning << "locscil: sustain defers "
					 "to non-looping source" << endmsg;

			}
			p->beg1 = ftp->begin1;
			p->end1 = ftp->end1;
		} else {
			p->beg1 = (long) *p->ibeg1;
			p->end1 = (long) *p->iend1;
			if (p->mod1 < 0 || p->mod1 > 3
			    || p->beg1 < 0 || p->end1 > ftp->flenfrms
			    || p->beg1 >= p->end1)
				goto lerr2;
		}
		if ((p->mod2 = (short) *p->imod2) < 0) {
			p->mod2 = ftp->loopmode2;
			p->beg2 = ftp->begin2;
			p->end2 = ftp->end2;
		} else {
			p->beg2 = (long) *p->ibeg2;
			p->end2 = (long) *p->iend2;
			if (p->mod2 < 0 || p->mod2 > 3
			    || p->beg2 < 0 || p->end2 > ftp->flenfrms
			    || p->beg2 >= p->end2)
				goto lerr3;
		}
		if (!p->mod2 && !p->end2)	/* if no release looping */
			p->end2 = ftp->soundend;	/*   set a reading
							   limit */
		p->beg1 <<= LOBITS;
		p->end1 <<= LOBITS;
		p->beg2 <<= LOBITS;
		p->end2 <<= LOBITS;
		p->lphs = 0;
		p->seg1 = 1;
		if ((p->curmod = p->mod1))
			p->looping = 1;
		else
			p->looping = 0;
		if (p->OUTOCOUNT == 1) {
			p->stereo = 0;
			if (ftp->nchnls != 1)
				error << "INIT: mono loscil cannot read from"
				      " stereo ftable" << endmsg;

		} else {
			p->stereo = 1;
			if (ftp->nchnls != 2)
				error << "INIT: stereo loscil cannot read from"
				      " mono ftable" << endmsg;

		}
	}
	return;
      lerr1:
	p->error ("loscili: no legal base frequency");
	return;
      lerr2:
	p->error ("loscili: illegal sustain loop data");
	return;
      lerr3:
	p->error ("loscili: illegal release loop data");
	return;
}

void
loscil(LOSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar1, *ftbl, *ftab, *xamp;
	int32 phs, inc, beg, end;
	int16 nsmps = ksmps, aamp;
	Number fract, v1, v2, *ar2;

	ftp = p->ftp;
	ftbl = ftp->ftable;
	if ((inc = (long) (*p->kcps * p->cpscvt)) < 0)
		inc = -inc;
	xamp = p->xamp;
	aamp = (p->has_audio_arguments()) ? 1 : 0;
	if (p->seg1) {		/* if still segment 1  */
		beg = p->beg1;
		end = p->end1;
		if (p->PROCESS->releasing())	/*    sense note_off   */
			p->looping = 0;
	} else {
		beg = p->beg2;
		end = p->end2;
	}
	phs = p->lphs;
	ar1 = p->ar1;
	if (p->stereo) {
		ar2 = p->ar2;
		goto phsck2;
	}
      phschk:if (phs >= end)
		goto put0;
	switch (p->curmod) {
	case 0:
		do {
			fract = (phs & LOMASK) * loscal;	/* NO LOOPING 
								 */
			ftab = ftbl + (phs >> LOBITS);
			v1 = *ftab++;
			*ar1++ = (v1 + (*ftab - v1) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs += inc) >= end)
				goto nxtseg;
		} while (--nsmps);
		break;
	case 1:
		do {
			fract = (phs & LOMASK) * loscal;	/* NORMAL
								   LOOPING */
			ftab = ftbl + (phs >> LOBITS);
			v1 = *ftab++;
			*ar1++ = (v1 + (*ftab - v1) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs += inc) >= end) {
				if (!(p->looping))
					goto nxtseg;
				phs -= end - beg;
			}
		} while (--nsmps);
		break;
	case 2:
	      case2:do {
			fract = (phs & LOMASK) * loscal;	/* BIDIR FORW,
								   EVEN */
			ftab = ftbl + (phs >> LOBITS);
			v1 = *ftab++;
			*ar1++ = (v1 + (*ftab - v1) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs += inc) >= end) {
				if (!(p->looping))
					goto nxtseg;
				phs -= (phs - end) * 2;
				p->curmod = 3;
				if (--nsmps)
					goto case3;
				else
					break;
			}
		} while (--nsmps);
		break;
	case 3:
	      case3:do {
			fract = (phs & LOMASK) * loscal;	/* BIDIR BACK,
								   EVEN */
			ftab = ftbl + (phs >> LOBITS);
			v1 = *ftab++;
			*ar1++ = (v1 + (*ftab - v1) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs -= inc) < beg) {
				phs += (beg - phs) * 2;
				p->curmod = 2;
				if (--nsmps)
					goto case2;
				else
					break;
			}
		} while (--nsmps);
		break;

	      nxtseg:if (p->seg1) {
			p->seg1 = 0;
			if ((p->curmod = p->mod2) != 0)
				p->looping = 1;
			if (--nsmps) {
				beg = p->beg2;
				end = p->end2;
				p->lphs = phs;
				goto phschk;
			}
			break;
		}
		if (--nsmps)
			goto phsout;
		break;
	}
	p->lphs = phs;
	return;

      phsout:p->lphs = phs;
      put0:do
		*ar1++ = FZERO;
	while (--nsmps);
	return;

      phsck2:if (phs >= end)
		goto put0s;	/* for STEREO:  */
	switch (p->curmod) {
	case 0:
		do {
			fract = (phs & LOMASK) * loscal;	/* NO LOOPING 
								 */
			ftab = ftbl + ((phs >> LOBITS) << 1);
			v1 = *ftab++;
			v2 = *ftab++;
			*ar1++ = (v1 + (*ftab++ - v1) * fract) * *xamp;
			*ar2++ = (v2 + (*ftab - v2) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs += inc) >= end)
				goto nxtseg2;
		} while (--nsmps);
		break;
	case 1:
		do {
			fract = (phs & LOMASK) * loscal;	/* NORMAL
								   LOOPING */
			ftab = ftbl + ((phs >> LOBITS) << 1);
			v1 = *ftab++;
			v2 = *ftab++;
			*ar1++ = (v1 + (*ftab++ - v1) * fract) * *xamp;
			*ar2++ = (v2 + (*ftab - v2) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs += inc) >= end) {
				if (!(p->looping))
					goto nxtseg2;
				phs -= end - beg;
			}
		} while (--nsmps);
		break;
	case 2:
	      case2s:do {
			fract = (phs & LOMASK) * loscal;	/* BIDIR FORW,
								   EVEN */
			ftab = ftbl + ((phs >> LOBITS) << 1);
			v1 = *ftab++;
			v2 = *ftab++;
			*ar1++ = (v1 + (*ftab++ - v1) * fract) * *xamp;
			*ar2++ = (v2 + (*ftab - v2) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs += inc) >= end) {
				if (!(p->looping))
					goto nxtseg2;
				phs -= (phs - end) * 2;
				p->curmod = 3;
				if (--nsmps)
					goto case3s;
				else
					break;
			}
		} while (--nsmps);
		break;
	case 3:
	      case3s:do {
			fract = (phs & LOMASK) * loscal;	/* BIDIR BACK,
								   EVEN */
			ftab = ftbl + ((phs >> LOBITS) << 1);
			v1 = *ftab++;
			v2 = *ftab++;
			*ar1++ = (v1 + (*ftab++ - v1) * fract) * *xamp;
			*ar2++ = (v2 + (*ftab - v2) * fract) * *xamp;
			if (aamp)
				xamp++;
			if ((phs -= inc) < beg) {
				phs += (beg - phs) * 2;
				p->curmod = 2;
				if (--nsmps)
					goto case2s;
				else
					break;
			}
		} while (--nsmps);
		break;

	      nxtseg2:if (p->seg1) {
			p->seg1 = 0;
			if ((p->curmod = p->mod2) != 0)
				p->looping = 1;
			if (--nsmps) {
				beg = p->beg2;
				end = p->end2;
				p->lphs = phs;
				goto phsck2;
			}
			break;
		}
		if (--nsmps)
			goto phsout2;
		break;
	}
	p->lphs = phs;
	return;

      phsout2:p->lphs = phs;
      put0s:do {
		*ar1++ = FZERO;
		*ar2++ = FZERO;
	} while (--nsmps);
}

#define ISINSIZ 32768L
#define ADMASK  32767L

static int16 *isintab;
static Number sinsizdsr;		/* ISINSIZ / esr */
static Number mkdkr;		        /* 1024000 / ekr */

void
adset(ADSYN * p)
{
	int32 n;
	char filename[PATH_MAX+1];
	RCPointer<MemoryMappedFile> mfp;
	int16 *adp, *endata, val;
	PTLPTR *ptlap, *ptlfp, *ptlim;
	int16 size;

	if (isintab == NULL) {	/* if no sin table yet, make one */
		int16 *ip;
		isintab = ip = (short *) new short[ISINSIZ];

		for (n = 0; n < ISINSIZ; n++) {
			*ip++ = (short) (sin(twopi * n / ISINSIZ) * 32767.);
		}
		sinsizdsr = ISINSIZ * onedsr;
		mkdkr = 1024000.0f * onedkr;	/*  & set local consts */
	}

	if (is_string_ptr (p->ifilcod)) {
		strcpy (filename, get_string (p->ifilcod));
	} else {
		sprintf(filename, "adsyn.%d", (int) *p->ifilcod);
	}

	mfp = memoryfiles.open (filename);
	
	if (mfp == 0) {
		p->error ("adsyn: cannot open file");
	}
	
	p->mfp = mfp;

	adp = (short *) mfp->beginp;	/* align on file data */
	endata = (short *) mfp->endp;

	size = 1 + (*adp == -1 ? MAXPTLS : *adp++);	/* Old no header ->
							   MAXPIL */
	if (p->aux.auxp == NULL || 
	    p->aux.size < sizeof (PTLPTR) * size) {
		p->aux.alloc (sizeof (PTLPTR) * size);
	}

	ptlap = ptlfp = (PTLPTR *) p->aux.auxp;		/* find base ptl 
							   blk */
	ptlim = ptlap + size;

	do {
		if ((val = *adp++) < 0) {	/* then for each brkpt set,  
						 */
			switch (val) {
			case -1:
				ptlap->nxtp = ptlap + 1;	/* chain the
								   ptl blks */
				if ((ptlap = ptlap->nxtp) >= ptlim)
					goto adsful;
				ptlap->ap = (DUPLE *) adp;	/*  record
								   start amp  */
				ptlap->amp = ptlap->ap->val;
				break;
			case -2:
				if ((ptlfp += 1) >= ptlim)
					goto adsful;
				ptlfp->fp = (DUPLE *) adp;	/*  record
								   start frq  */
				ptlfp->frq = ptlfp->fp->val;
				ptlfp->phs = 0;		/*  and clr the
							   phase */
				break;
			default:
				p->error (
					       "adsyn: illegal code %d "
					       "encountered", val);
			}
		}
	} while (adp < endata);

	if (ptlap != ptlfp) {
		p->error ("adsyn: "
			       "%d amp tracks, %d freq tracks",
			       ptlap - (PTLPTR *) p->aux.auxp - 1,
			       ptlfp - (PTLPTR *) p->aux.auxp - 1);
	}
	ptlap->nxtp = NULL;	/* terminate the chain */
	p->mksecs = 0;

	return;

      adsful:
	p->error ("adsyn: partial count exceeds MAXPTLS");
	return;
}

void
adsyn(ADSYN * p)
{
	PTLPTR *curp, *prvp;
	DUPLE *ap, *fp;
	int16 curtim, diff, ktogo;
	int32 phs, sinc, *sp, amp;
	int16 nsmps;
	Number *ar;
	Number ampscale, frqscale;
	int32 timkincr, nxtim;

	ampscale = *p->kamod * dv32768;		/* (since 15-bit sine
						   table) */
	frqscale = *p->kfmod * sinsizdsr;
	timkincr = (int32) (*p->ksmod * mkdkr);	/* 1024 * msecs of
						   analysis  */
	sp = (int32 *) p->rslt;	/* use out array for sums */
	nsmps = ksmps;

	do {
		*sp++ = 0;	/* cleared first to zero */
	} while (--nsmps);

	curtim = (short) (p->mksecs >> 10);	/* cvt mksecs to msecs */
	curp = (PTLPTR *) p->aux.auxp;	/* now for each partial:    */

	while ((prvp = curp) && (curp = curp->nxtp) != NULL) {
		ap = curp->ap;
		fp = curp->fp;

		while (curtim >= (ap + 1)->tim) { /* timealign ap, fp */
			curp->ap = ap += 1;
		}

		while (curtim >= (fp + 1)->tim) {
			curp->fp = fp += 1;
		}

		if ((amp = curp->amp)) {	/* for non-zero amp   */
			sinc = (long) (curp->frq * frqscale);
			phs = curp->phs;
			sp = (int32 *) p->rslt;
			nsmps = ksmps;	/*   addin a sinusoid */
			do {
				*sp++ += *(isintab + phs) * amp;
				phs += sinc;
				phs &= ADMASK;
			}
			while (--nsmps);
			curp->phs = phs;
		}

		if ((nxtim = (ap + 1)->tim) == 32767) { /* if last amp
							   this partial */
			prvp->nxtp = curp->nxtp;	/* remov from activ
							   chain */
			curp = prvp;

		} else {	/* else interp towds nxt amp */

			if ((diff = (short) ((ap + 1)->val - amp))) {
				ktogo = (short) (((nxtim << 10) - p->mksecs +
					       timkincr - 1) / timkincr);
				curp->amp += diff / ktogo;
			}
			if ((nxtim = (fp + 1)->tim) != 32767	/* & nxt frq */
			    && (diff = (fp + 1)->val - curp->frq)) {
				ktogo = (short) (((nxtim << 10) - p->mksecs +
					       timkincr - 1) / timkincr);
				curp->frq += diff / ktogo;
			}
		}
	}

	p->mksecs += timkincr;	/* advance the time */
	ar = p->rslt;
	sp = (int32 *) ar;
	nsmps = ksmps;

	do {
		*ar++ = *sp++ * ampscale;
	} while (--nsmps);
}

Opcode opcodes[] =
{
	UGENS3_OPCODE_LIST,
	{ NULL }
};
