/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: ugens2.cc,v 1.2 1999/12/15 19:43:27 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dsp.h>
#include "ugens2.h"

void
phsset(PHSOR * p)
{
	Number phs;
	int32 int32phs;
	if ((phs = *p->iphs) >= FZERO) {
		if ((int32phs = (int32) phs))
			warning << "init phase truncation" << endmsg;

		p->curphs = phs - (Number) int32phs;
	}
}

void
kphsor(PHSOR * p)
{
	Number phs;
	*p->sr = phs = p->curphs;
	if ((phs += *p->xcps * onedkr) >= FONE)
		phs -= FONE;
	else if (phs < FZERO)
		phs += FONE;
	p->curphs = phs;
}

void
phsor(PHSOR * p)
{
	int nsmps = ksmps;
	Number *rs, phase, incr;

	rs = p->sr;
	phase = p->curphs;
	if (p->has_audio_arguments()) {
		Number *cps = p->xcps;
		do {
			incr = *cps++ / esr;
			*rs++ = phase;
			phase += incr;
			if (phase >= FONE)
				phase -= FONE;
			else if (phase < FZERO)
				phase += FONE;
		} while (--nsmps);
	} else {
		incr = *p->xcps / esr;
		do {
			*rs++ = phase;
			phase += incr;
			if (phase >= FONE)
				phase -= FONE;
			else if (phase < FZERO)
				phase += FONE;
		} while (--nsmps);
	}
	p->curphs = phase;
}


/*****************************************************************************/
/*****************************************************************************/

/* Table read code - see TABLE data structure in ugens2.h.  */


/*************************************/

/* itblchk()

 * This is called at init time by tblset() to set up the TABLE data
 * structure for subsequent k and a rate operations.
 *
 * It is also called at init time by itable() and itablei() prior to
 * them calling ktable() and ktabli() respectively to produce a single
 * result at init time.
 *
 * A similar function - ptblchk() does the same job, but reports
 * errors in a way suitable for performance time.  */

/* If the specified table number can be found, then the purpose is to
 * read the three i rate input variables and the function table number
 * input variable - (which we can assume here is also i rate) to set
 * up the TABLE data structure ready for the k and a rate functions.  */

int
itblchk(TABLE * p)
{
	if ((p->ftp = ftfind(p->xfn)) == NULL) {
		p->error ("tblset: no such table %s", 
			  FunctionTable::name_as_string (p->xfn));
		return (0);
	}

	/* Although TABLE has an integer variable for the table number
	 * (p->pfn) we do not need to write it.  We know that the k
	 * and a rate functions which will follow will not be
	 * expecting a changed table number.
	 *
	 * p->pfn exists only for checking table number changes for
	 * functions which are expecting a k rate table number.  */

	/* Set denormalisation factor to 1 or table length, depending
	 * on the state of ixmode. */
	if (*p->ixmode)
		p->xbmul = p->ftp->flen;

	else
		p->xbmul = 1L;

	/* Multiply the ixoff value by the xbmul denormalisation
	 * factor and then check it is between 0 and the table length.
	 *
	 * Bug fix: was p->offset * *p->ixoff */

	if ((p->offset = p->xbmul * *p->ixoff) < 0.0 ||
	    p->offset > p->ftp->flen) {
		p->error ("Offset %f < 0 or > tablelength", p->offset);
		return (0);
	}
	p->wrap = (int32) *p->iwrap;
	return (1);
}

/* ptblchk()

 * This is called at init time by tblsetkt() to set up the TABLE data
 * structure for subsequent k and a rate operations which are
 * expecting the table number to change at k rate.
 *
 * tblsetkt() does very little - just setting up the wrap variable in
 * TABLE. All the other variables depend on the table number. This is
 * not available at init time, so the following 4 functions must look
 * for the changed table number and set up the variables accordingly -
 * generating error messages in a way which works at performance time.
 *
 * k rate   a rate
 *
 * ktablekt tablekt   Non interpolated
 * ktablikt tablikt   Interpolated
 *  */
void
ptblchk(TABLE * p)
{
	/* TABLE has an integer variable for the previous table number
	 * (p->pfn).
	 *
	 * Now (at init time) we do not know the function table number
	 * which will be provided at perf time, so set p->pfn to 0, so
	 * that the k or a rate code will recognise that the first table
	 * number is different from the "previous" one.  */
	p->pfn = 0;

	/* The only other thing to do is write the wrap value into the
	 * immediate copy of it in TABLE.  */
	p->wrap = (int32) *p->iwrap;
}


/*---------------------------------------------------------------------------*/

/* tblset() */

void
tblset(TABLE * p)
{
	itblchk(p);
}


/* tblsetkt() */

void
tblsetkt(TABLE * p)
{
	ptblchk(p);
}

/*************************************/

/* Special functions to use when the output value is an init time
 * variable.
 *
 * These are called by the opodlst lines for itable and itablei ugens.
 *
 * They call itblchk() and if the table was found, they call the k
 * rate function just once.
 *
 */
void ktable(TABLE *);
void ktabli(TABLE *);

void
itable(TABLE * p)
{
	if (itblchk(p))
		ktable(p);
}

void
itabli(TABLE * p)
{
	if (itblchk(p))
		ktabli(p);
}

/*---------------------------------------------------------------------------*/

/* Functions which read the table.

 * First we have the four basic functions for a and k rate, non
 * interpolated and interpolated reading.  These all assume that the
 * TABLE data structure has been correctly set up - they are not
 * expecting the table number to change at k rate.
 *
 * These are:
 * k rate  a rate
 *
 * ktable  table   Non interpolated
 * ktabli  tabli   Interpolated
 *
 * Then we have four more functions which are expecting the table
 * number to change at k rate.  They deal with this, and then call one
 * of the above functions to do the reading.
 *
 * These are:
 * k rate   a rate
 *
 * ktablekt tablekt   Non interpolated
 * ktablikt tablikt   Interpolated
 *  */

/* ktable() and ktabli()
 * ---------------------
 *
 * These both read a single value from the table. ktabli() does it
 * with interpolation.
 *
 * This is typically used for k rate reading - where they are called
 * as a result of being listed in a line in opcodlst.  They are also
 * called by two functions which after they have coped with any change
 * in the k rate function table number.
 *
 * ktablekt() and ktablikt().
 *
 * In addition, they can be called by the init time functions:
 * itable() and itabli().
 *
 *
 * table() and tabli()
 * -------------------
 *
 * These do the reading at a rate with an a rate index.
 *
 * They are called directly via their entries in opcodlst, and also by
 * two functions which call them after they have coped with any change
 * in the k rate function table number.
 *
 * tablekt() and tablikt().
 *
 * */

/*************************************/

/* ktable() */

void
ktable(TABLE * p)
{
	RCPointer<FunctionTable> ftp;
	int32 indx, length;
	Number ndx;

	ftp = p->ftp;
	if (ftp == NULL) {	/* RWD fix */
		p->error ("table(krate): not initialized");
		return;
	}
	ndx = *p->xndx;
	length = ftp->flen;
	/* Multiply ndx by denormalisation factor, and add in the offset
	 * - already denormalised - by tblchk().
	 * xbmul = 1 or table length depending on state of ixmode.  */

	ndx = (ndx * p->xbmul) + p->offset;

	/* ndx now includes the offset and is ready to address the table.

	 * The original code was:
	 *  indx = (int32) (ndx + p->offset);
	 *
	 * This is a problem, causes problems with negative numbers.
	 *
	 */
	indx = (int32) floor(ndx);

	/* Now for "limit mode" - the "non-wrap" function, depending on
	 * iwrap.
	 *
	 * The following section of code limits the final index to 0 and
	 * the last location in the table.
	 *
	 * It is only used when wrap is OFF.  The wrapping is achieved by
	 * code after this - when this code is not run.  */
	if (!p->wrap) {
		/* Previously this code limited the upper range of the indx to
		 * the table length - for instance 8.  Then the result was
		 ANDed
		 * with a mask (for instance 7).
		 *
		 * This meant that when the input index was 8 or above, it got
		 * reduced to 0.  What we want is for it to stick at the index
		 * which reads the last value from the table - in this example
		 * from location 7.
		 *
		 * So instead of limiting to the table length, we limit to
		 * (table length - 1).  */
		if (indx > length - 1)
			indx = length - 1;

		/* Now limit negative values to zero.  */
		else if (indx < 0L)
			indx = 0L;
	}
	/* The following code uses an AND with an integer like 0000 0111
	 * to wrap the current index within the range of the table.  In
	 * the original version, this code always ran, but with the new
	 * (length - 1) code above, it would have no effect, so it is now
	 * an else operation - running only when iwrap = 1.  This may save
	 * half a usec or so.  */
	else
		indx &= ftp->lenmask;

	/* Now find the address of the start of the table, add it to the
	 * index, read the value from there and write it to the
	 * destination.  */
	*p->rslt = *(ftp->ftable + indx);
}


/* table()  */

/* table() is similar to ktable() above, except that it processes an
 * array of input indexes, to send results to another array.  These
 * arrays are ksmps int32.  */
void
table(TABLE * p)
{
	RCPointer<FunctionTable> ftp;
	Number *rslt, *pxndx, *tab;
	int32 indx, mask, length;
	int nsmps = ksmps;
	Number ndx, xbmul, offset;

	ftp = p->ftp;
	if (ftp == NULL) {	/* RWD fix */
		p->error ("table: not initialized");
		return;
	}
	rslt = p->rslt;
	length = ftp->flen;
	pxndx = p->xndx;
	xbmul = (Number) p->xbmul;
	offset = p->offset;
	mask = ftp->lenmask;
	tab = ftp->ftable;
	do {
		/* Read in the next raw index and increment the pointer ready
		 * for the next cycle.
		 *
		 * Then multiply the ndx by the denormalising factor and add in
		 * the offset.  */

		ndx = (*pxndx++ * xbmul) + offset;
		indx = (int32) floor(ndx);

		/* Limit = non-wrap.  Limits to 0 and (length - 1), or does the
		 * wrap code.  See notes above in ktable().  */
		if (!p->wrap) {
			if (indx > length - 1)
				indx = length - 1;
			else if (indx < 0L)
				indx = 0L;
		}
		/* do the wrap code only if we are not doing the non-wrap code. 
		 */
		else
			indx &= mask;
		*rslt++ = *(tab + indx);
	}
	while (--nsmps);
}


/* ktabli() */

/* ktabli() is similar to ktable() above, except that it uses the
 * fractional part of the final index to interpolate between one value
 * in the table and the next.
 *
 * This means that it may read the guard point.  In a table of
 * "length" 8, the guardpoint is at locaton 8. The normal part of the
 * table is locations 0 to 7.
 *
 * In non-wrap mode, when the final index is negative, the output
 * should be the value in location 0.
 *
 * In non-wrap mode, when the final index is >= length, then the
 * output should be the value in the guard point location.  */
void
ktabli(TABLE * p)
{
	RCPointer<FunctionTable> ftp;
	int32 indx, length;
	Number v1, v2, fract, ndx;

	ftp = p->ftp;
	if (ftp == NULL) {
		p->error ("tablei(krate): not initialized");
		return;
	}
	ndx = *p->xndx;
	length = ftp->flen;
	/* Multiply ndx by denormalisation factor.
	 * xbmul is 1 or table length depending on state of ixmode.
	 * Add in the offset, which has already been denormalised by
	 * tblchk().  */

	ndx = (ndx * p->xbmul) + p->offset;
	indx = (int32) floor(ndx);

	/* We need to generate a fraction - How much above indx is ndx?
	 * It will be between 0 and just below 1.0.  */
	fract = ndx - indx;

	/* Start of changes to fix non- wrap bug.

	 * There are two changes here:
	 *
	 * 1 - We only run the wrap code if iwrap = 1. Previously it was
	 * always run.
	 *
	 * 2 - The other change is similar in concept to limiting the
	 * index to (length - 1) when in non-wrap mode.
	 *
	 * This would be fine - the fractional code would enable us to
	 * interpolate using an index value which is almost as high as the
	 * length of the table.  This would be good for 7.99999 etc.
	 * However, to be a little pedantic, what we want is for any index
	 * of 8 or more to produce a result exactly equal to the value at
	 * the guard point.
	 *
	 * We will let all (non negative) values which are less than
	 * length pass through. This deals with all cases 0 to 7.9999
	 * . . .
	 *
	 * However we will look for final indexes of length (8) and above
	 * and take the following steps:
	 *
	 * fract = 1
	 * indx = length - 1
	 *
	 * We then continue with the rest of code.  This causes the result
	 * to be the value read from the guard point - which is what we
	 * want.
	 *
	 * Likewise, if the final index is negative, set both fract and
	 * indx to 0.  */
	if (!p->wrap) {
		if (ndx > length) {
			indx = length - 1;
			fract = 1.0f;
		} else if (ndx < 0) {
			indx = 0L;
			fract = 0.0f;
		}
	}
	/* We are in wrap mode, so do the wrap function.  */
	else
		indx &= ftp->lenmask;

	/* Now read the value at indx and the one beyond */
	v1 = *(ftp->ftable + indx);
	v2 = *(ftp->ftable + indx + 1);
	*p->rslt = v1 + (v2 - v1) * fract;
}


/* tabli() */

/* tabli() is similar to ktabli() above, except that it processes an
 * array of input indexes, to send results to another array. */

void
tabli(TABLE * p)
{
	RCPointer<FunctionTable> ftp;
	int32 indx, mask, length;
	int nsmps = ksmps;
	Number *rslt, *pxndx, *tab;
	Number fract, v1, v2, ndx, xbmul, offset;

	ftp = p->ftp;
	if (ftp == NULL) {
		p->error ("tablei: not initialized");
		return;
	}
	rslt = p->rslt;
	length = ftp->flen;
	pxndx = p->xndx;
	xbmul = (Number) p->xbmul;
	offset = p->offset;
	mask = ftp->lenmask;
	tab = ftp->ftable;
	do {
		/* Read in the next raw index and increment the pointer ready
		 * for the next cycle.
		 * Then multiply the ndx by the denormalising factor and add in
		 * the offset.  */

		ndx = (*pxndx++ * xbmul) + offset;
		indx = (int32) floor(ndx);

		/* We need to generate a fraction - How much above indx is ndx?
		 * It will be between 0 and just below 1.0.  */
		fract = ndx - indx;
		/* As for ktabli() code to handle non wrap mode, and wrap mode. 
		 */
		if (!p->wrap) {
			if (ndx > length) {
				indx = length - 1;
				fract = 1.0f;
			} else if (ndx < 0) {
				indx = 0L;
				fract = 0.0f;
			}
		} else
			indx &= mask;
		/* As for ktabli(), read two values and interpolate between
		 * them.  */
		v1 = *(tab + indx);
		v2 = *(tab + indx + 1);
		*rslt++ = v1 + (v2 - v1) * fract;
	}
	while (--nsmps);
}

/*************************************/

/* Four functions to call the above four, after handling the k rate
 * table number variable.
 *
 * tblsetkt() does very little - just setting up the wrap variable in
 * TABLE. All the other variables depend on the table number. This is
 * not available at init time, so the following 4 functions must look
 * for the changed table number and set up the variables accordingly -
 * generating error messages in a way which works at performance time.
 * * k rate   a rate
 *
 * ktablekt tablekt   Non interpolated
 * ktablikt tablikt   Interpolated
 *
 * Since these perform identical operations, apart from the function
 * they call, create a common function to do this work:
 *
 * ftkrchk() */

int
ftkrchk(TABLE * p)
{
	/* Check the table number is >= 1.  Print error and deactivate if
	 * it is not.  Return 0 to tell calling function not to proceed
	 * with a or k rate operations.
	 *
	 * We must do this to catch the situation where the first call has
	 * a table number of 0, and since that equals pfn, we would
	 * otherwise proceed without checking the table number - and none
	 * of the pointers would have been set up.  */
	if (*p->xfn < 1) {
		p->error ("k rate function table no. %f < 1", *p->xfn);
		return (0);
	}
	/* Check to see if table number has changed from previous value.
	 * On the first run through, the previous value will be 0.  */

	if (p->pfn != (int32) *p->xfn) {
		/* If it is different, check to see if the table exists.
                 * 
                 *
		 * Return 0 to tell calling function not to proceed with a or
		 * k rate operations. */
		
		if ((p->ftp = ftfind (p->xfn)) == NULL) {
			p->error ("ftkrchk: no such table %s", 
				  FunctionTable::name_as_string (p->xfn));
			return (0);
		}
		/* p->ftp now points to the FunctionTable data structure of the newly
		 * selected table.
		 *
		 * Now we set up some variables in TABLE ready for the k or a
		 * rate functions which follow.  */

		/* Write the integer version of the table number into pfn so
		 * we can later decide whether subsequent calls to the k and a
		 * rate functions occur with a table number value which points
		 * to a different table. */
		p->pfn = (int32) *p->xfn;

		/* Set denormalisation factor to 1 or table length, depending
		 * on the state of ixmode. */
		if (*p->ixmode)
			p->xbmul = p->ftp->flen;
		else
			p->xbmul = 1L;

		/* Multiply the ixoff value by the xbmul denormalisation
		 * factor and then check it is between 0 and the table length.  
		 */

		if ((p->offset = p->xbmul * *p->ixoff) < 0.0 ||
		    p->offset > p->ftp->flen) {
			p->error ("Offset %f < 0 or > tablelength",
				    p->offset);
			return (0);
		}
	}
	return (1);
}

/* Now for the four functions, which are called as a result of being
 * listed in opcodlst in entry.c */

void
ktablekt(TABLE * p)
{
	if (ftkrchk(p))
		ktable(p);
}

void
tablekt(TABLE * p)
{
	if (ftkrchk(p))
		table(p);
}

void
ktablikt(TABLE * p)
{
	if (ftkrchk(p))
		ktabli(p);
}

void
tablikt(TABLE * p)
{
	if (ftkrchk(p))
		tabli(p);
}

void
ko1set(OSCIL1 * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->ifn)) == NULL) {
		p->error ("oscil1: no such table %s", 
			  FunctionTable::name_as_string (p->ifn));
		return;
	}
	if (*p->idur <= FZERO) {
		warning << "oscil1: duration < zero" << endmsg;

	}
	p->ftp = ftp;
	p->phs = 0;
	p->dcnt = (int32) (*p->idel * ekr);
	p->kinc = (int32) (kicvt / *p->idur);
}

void
kosc1(OSCIL1 * p)
{
	RCPointer<FunctionTable> ftp;
	int32 phs, dcnt;

	ftp = p->ftp;
	phs = p->phs;
	*p->rslt = *(ftp->ftable + (phs >> ftp->lobits)) * *p->kamp;
	if ((dcnt = p->dcnt) > 0L)
		dcnt--;
	else if (dcnt == 0L) {
		phs += p->kinc;
		if (phs >= MAXLEN) {
			phs = MAXLEN;
			dcnt--;
		}
		p->phs = phs;
	}
	p->dcnt = dcnt;
}

void
kosc1i(OSCIL1 * p)
{
	RCPointer<FunctionTable> ftp;
	Number fract, v1, *ftab;
	int32 phs, dcnt;

	ftp = p->ftp;
	phs = p->phs;
	fract = PFRAC(phs);
	ftab = ftp->ftable + (phs >> ftp->lobits);
	v1 = *ftab++;
	*p->rslt = (v1 + (*ftab - v1) * fract) * *p->kamp;
	if ((dcnt = p->dcnt) > 0L) {
		dcnt--;
		p->dcnt = dcnt;
	} else if (dcnt == 0L) {
		phs += p->kinc;
		if (phs >= MAXLEN) {
			phs = MAXLEN;
			dcnt--;
			p->dcnt = dcnt;
		}
		p->phs = phs;
	}
}

void
oscnset(OSCILN * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->ifn)) == NULL) {
		p->error ("osciln: no such table %s", 
			  FunctionTable::name_as_string (p->ifn));
		return;
	}
	p->ftp = ftp;
	p->inc = ftp->flen * *p->ifrq * onedsr;
	p->index = 0.0f;
	p->maxndx = ftp->flen - 1.0f;
	p->ntimes = (int32) *p->itimes;
}

void
osciln(OSCILN * p)
{
	Number *rs = p->rslt;
	int32 nsmps = ksmps;

	if (p->ntimes) {
		Number *ftbl = p->ftp->ftable;
		Number amp = *p->kamp;
		Number ndx = p->index;
		Number inc = p->inc;
		Number maxndx = p->maxndx;
		do {
			*rs++ = *(ftbl + (int32) ndx) * amp;
			if ((ndx += inc) > maxndx) {
				if (--p->ntimes)
					ndx -= maxndx;
				else if (--nsmps)
					goto putz;
				else
					return;
			}
		} while (--nsmps);
		p->index = ndx;
	} else {
	      putz:
		do
			*rs++ = 0.0f;
		while (--nsmps);
	}
}

void
oscset(OSC * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->ifn)) == NULL) {
		p->error ("koscil: no such table %s", 
			  FunctionTable::name_as_string (p->ifn));
		return;
	}

	p->ftp = ftp;
	if (*p->iphs >= 0) {
		p->lphs = ((int32) (*p->iphs * fmaxlen)) & PMASK;
	}
}

void
koscil(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	int32 phs, inc;

	ftp = p->ftp;
	phs = p->lphs;
	inc = (int32) (*p->xcps * kicvt);
	*p->sr = *(ftp->ftable + (phs >> ftp->lobits)) * *p->xamp;
	phs += inc;
	phs &= PMASK;
	p->lphs = phs;
}

void
osckk(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number amp, *ar, *ftbl;
	int32 phs, inc, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	ftbl = ftp->ftable;
	phs = p->lphs;
	inc = (int32) (*p->xcps * sicvt);
	lobits = ftp->lobits;
	amp = *p->xamp;
	ar = p->sr;
	do {
		*ar++ = *(ftbl + (phs >> lobits)) * amp;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}

void
oscka(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, amp, *cpsp, *ftbl;
	int32 phs, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	ftbl = ftp->ftable;
	lobits = ftp->lobits;
	amp = *p->xamp;
	cpsp = p->xcps;
	phs = p->lphs;
	ar = p->sr;
	do {
		int32 inc;
		inc = (int32) (*cpsp++ * sicvt);
		*ar++ = *(ftbl + (phs >> lobits)) * amp;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}

void
oscak(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, *ampp, *ftbl;
	int32 phs, inc, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	ftbl = ftp->ftable;
	lobits = ftp->lobits;
	phs = p->lphs;
	inc = (int32) (*p->xcps * sicvt);
	ampp = p->xamp;
	ar = p->sr;
	do {
		*ar++ = *(ftbl + (phs >> lobits)) * *ampp++;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}

void
oscaa(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, *ampp, *cpsp, *ftbl;
	int32 phs, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	ftbl = ftp->ftable;
	lobits = ftp->lobits;
	phs = p->lphs;
	ampp = p->xamp;
	cpsp = p->xcps;
	ar = p->sr;
	do {
		int32 inc;
		inc = (int32) (*cpsp++ * sicvt);
		*ar++ = *(ftbl + (phs >> lobits)) * *ampp++;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}

void
koscli(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	int32 phs, inc;
	Number *ftab, fract, v1;

	phs = p->lphs;
	ftp = p->ftp;
	fract = PFRAC(phs);
	ftab = ftp->ftable + (phs >> ftp->lobits);
	v1 = *ftab++;
	*p->sr = (v1 + (*ftab - v1) * fract) * *p->xamp;
	inc = (int32) (*p->xcps * kicvt);
	phs += inc;
	phs &= PMASK;
	p->lphs = phs;
}

void
osckki(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number fract, v1, amp, *ar, *ftab;
	int32 phs, inc, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	lobits = ftp->lobits;
	phs = p->lphs;
	inc = (int32) (*p->xcps * sicvt);
	amp = *p->xamp;
	ar = p->sr;
	do {
		fract = PFRAC(phs);
		ftab = ftp->ftable + (phs >> lobits);
		v1 = *ftab++;
		*ar++ = (v1 + (*ftab - v1) * fract) * amp;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}

void
osckai(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, amp, *cpsp, fract, v1, *ftab;
	int32 phs, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	lobits = ftp->lobits;
	amp = *p->xamp;
	cpsp = p->xcps;
	phs = p->lphs;
	ar = p->sr;
	do {
		int32 inc;
		inc = (int32) (*cpsp++ * sicvt);
		fract = PFRAC(phs);
		ftab = ftp->ftable + (phs >> lobits);
		v1 = *ftab++;
		*ar++ = (v1 + (*ftab - v1) * fract) * amp;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}

void
oscaki(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number v1, fract, *ar, *ampp, *ftab;
	int32 phs, inc, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	ftab = ftp->ftable;
	lobits = ftp->lobits;
	phs = p->lphs;
	inc = (int32) (*p->xcps * sicvt);
	ampp = p->xamp;
	ar = p->sr;
	do {
		fract = (Number) PFRAC(phs);
		ftab = ftp->ftable + (phs >> lobits);
		v1 = *ftab++;
		*ar++ = (v1 + (*ftab - v1) * fract) * *ampp++;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}

void
oscaai(OSC * p)
{
	RCPointer<FunctionTable> ftp;
	Number v1, fract, *ar, *ampp, *cpsp, *ftab;
	int32 phs, lobits;
	int nsmps = ksmps;

	ftp = p->ftp;
	ftab = ftp->ftable;
	lobits = ftp->lobits;
	phs = p->lphs;
	ampp = p->xamp;
	cpsp = p->xcps;
	ar = p->sr;
	do {
		int32 inc;
		inc = (int32) (*cpsp++ * sicvt);
		fract = (Number) PFRAC(phs);
		ftab = ftp->ftable + (phs >> lobits);
		v1 = *ftab++;
		*ar++ = (v1 + (*ftab - v1) * fract) * *ampp++;
		phs += inc;
		phs &= PMASK;
	}
	while (--nsmps);
	p->lphs = phs;
}


Opcode opcodes[] =
{
	UGENS2_OPCODE_LIST,
	{NULL}
};
