/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: mathops.cc,v 1.1 1999/11/01 04:29:06 pbd Exp $
*/


#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/opcode_enums.h>

#include <time.h>


#include "random.h"
#include "mathops.h"

void
ipow(POW * p)
{				/*      Power for i-rate */
	*p->sr = (Number) pow(*p->in, *p->pow);	
}

void
apow(POW * p)
{				/*      Power routine for a-rate  */
	int32 n = ksmps;
	Number *in = p->in, *out = p->sr;

	do {
		*out++ = (Number) pow(*in++, *p->pow) / *p->norm;
	} while (--n);
}

void
kpow(POW * p)
{				/*      Power routine for k-rate        */
	*p->sr = (Number) pow(*p->in, *p->pow) / *p->norm;
}

void 
seedrand(PRAND *p)
{
	/* Make sure that the time of day function is returning
	 * non-zero.
	 */
	if ( (*p->out == 0) && (time(NULL) != 0) ) {
		info << "Setting random seed from current time and date" << endmsg;

		
		/* Since the time changes rather slowly, and the 
		 * first generated randint31() result depends
		 * on it and therefore changes rather slowly
		 * call randint31() twice the mash the seed
		 * well before an opcode can use it.
		 */
		
		seed31 = (int32) time(NULL);
		seed31 = randint31();
		seed31 = randint31();
	} else {
		seed31 = (int32) floor( fmod( (double)(1000 * fabs(*p->out)), (double)2147483647 ) );
		info << "Random seed set with " << *p->out << "   " << endmsg;

	}
	
	if (seed31 == 0) seed31 = 1;

	info << "Random seed = " << seed31 << endmsg;
}

/*=========================================================================
 *
 * Funcions for xbunirand opcodes in entry.c.
 * Based on auinform() and ikuniform() below.
 * unirand was 0 to +1.  bunirand is -1 to +1
 */

void
abuniform(PRAND * p)
{				/* Uniform bipolar distribution */
	int32 n = ksmps;
	Number *out = p->out;
	Number arg1 = *p->arg1;

	do
		*out++ = arg1 * (unifrand(2) - 1);
	while (--n);
}

void
ikbuniform(PRAND * p)
{
	Number *out = p->out;

	*out = *p->arg1 * (unifrand(2) - 1);
}

/* Back to Paris' code.  - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
auniform(PRAND * p)
{				/* Uniform distribution */
	int32 n = ksmps;
	Number *out = p->out;
	Number arg1 = *p->arg1;

	do
		*out++ = unifrand(arg1);
	while (--n);
}

void
ikuniform(PRAND * p)
{
	Number *out = p->out;

	*out = unifrand(*p->arg1);
}

void
alinear(PRAND * p)
{				/* Linear random functions      */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = linrand(*p->arg1);
	} while (--n);
}

void
iklinear(PRAND * p)
{
	Number *out = p->out;
	*out++ = linrand(*p->arg1);
}

void
atrian(PRAND * p)
{				/*      Triangle random functions       */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = trirand(*p->arg1);
	} while (--n);
}

void
iktrian(PRAND * p)
{
	Number *out = p->out;
	*out++ = trirand(*p->arg1);
}

void
aexp(PRAND * p)
{				/*      Exponential random functions    */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = exprand(*p->arg1);
	} while (--n);
}

void
ikexp(PRAND * p)
{
	Number *out = p->out;
	*out++ = exprand(*p->arg1);
}

void
abiexp(PRAND * p)
{				/*      Bilateral exponential rand. functions
				 */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = biexprand(*p->arg1);
	} while (--n);
}

void
ikbiexp(PRAND * p)
{
	Number *out = p->out;
	*out++ = biexprand(*p->arg1);
}

void
agaus(PRAND * p)
{				/*      Gaussian random functions       */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = gaussrand(*p->arg1);
	} while (--n);
}

void
ikgaus(PRAND * p)
{
	Number *out = p->out;
	*out++ = gaussrand(*p->arg1);
}

void
acauchy(PRAND * p)
{				/*      Cauchy random functions */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = cauchrand(*p->arg1);
	} while (--n);
}

void
ikcauchy(PRAND * p)
{
	Number *out = p->out;
	*out++ = cauchrand(*p->arg1);
}

void
apcauchy(PRAND * p)
{				/*      Positive Cauchy random functions */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = pcauchrand(*p->arg1);
	} while (--n);
}

void
ikpcauchy(PRAND * p)
{
	Number *out = p->out;
	*out++ = pcauchrand(*p->arg1);
}

void
abeta(PRAND * p)
{				/*      Beta random functions   */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = betarand(*p->arg1, *p->arg2, *p->arg3);
	} while (--n);
}

void
ikbeta(PRAND * p)
{
	Number *out = p->out;
	*out++ = betarand(*p->arg1, *p->arg2, *p->arg3);
}

void
aweib(PRAND * p)
{				/*      Weibull randon functions        */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = weibrand(*p->arg1, *p->arg2);
	} while (--n);
}

void
ikweib(PRAND * p)
{
	Number *out = p->out;
	*out++ = weibrand(*p->arg1, *p->arg2);
}

void
apoiss(PRAND * p)
{				/*      Poisson random funcions */
	int32 n = ksmps;
	Number *out = p->out;

	do {
		*out++ = poissrand(*p->arg1);
	} while (--n);
}

void
ikpoiss(PRAND * p)
{
	Number *out = p->out;
	*out++ = poissrand(*p->arg1);
}

Opcode opcodes[] =
{
	MATHOPS_OPCODE_LIST,
       	{NULL}
};

