/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: lfo.cc,v 1.1 1999/11/01 04:29:06 pbd Exp $
*/

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>

#include "lfo.h"

#define MAXPHASE 0x1000000
#define MAXMASK  0x0ffffff

void lfoset(LFO *p)
{
	/* Types: 0:	sine
	   1:  triangles
	   2:  square (biplar)
	   3:  square (unipolar)
	   4:  saw-tooth
	   5:  saw-tooth(down)
	*/
	int32 type = (int32)*p->type;
	if (type == 0) {		/* Sine wave so need to create */
		int32 i;
		if (p->auxd.auxp==NULL) {
			p->auxd.alloc (sizeof(Number)*4097L);
			p->sine = (Number*)p->auxd.auxp;
		}
		for (i=0; i<4096; i++) {
		    p->sine[i] = (Number)sin(TWOPI*(double)i/4096.0);
		}
	}
	else if (type>5 || type<0) {
		p->error ("LFO: unknown oscilator type %d",
			       type);
	}
	p->lasttype = type;
	p->phs = 0;
}

void lfok(LFO *p)
{
	int32	phs;
	Number	fract;
	Number	res = 0;
	int32	iphs;

	phs = p->phs;
	switch (p->lasttype) {
	case 0:
		iphs = phs >> 12;
		fract = (Number)(phs & 0xfff)/4096.0f;
		res = p->sine[iphs];
		res = res + (p->sine[iphs+1]-res)*fract;
		break;      
	case 1:			/* Trangular */
		res = (Number)((phs<<2)&MAXMASK)/(Number)MAXPHASE;
		if (phs < MAXPHASE/4) {}
		else if (phs < MAXPHASE/2)
		    res = 1.0f - res;
		else if (phs < 3*MAXPHASE/4)
		    res = - res;
		else
		    res = res - 1.0f;
		break;
	case 2:			/* Bipole square wave */
		if (phs<MAXPHASE/2) res = 1.0f;
		else res = -1.0f;
		break;
	case 3:			/* Unipolar square wave */
		if (phs<MAXPHASE/2) res = 1.0f;
		else res = 0.0f;
		break;
	case 4:			/* Saw Tooth */
		res = (Number)phs/(Number)MAXPHASE;
		break;
	case 5:			/* Reverse Saw Tooth */
		res = 1.0f - (Number)phs/(Number)MAXPHASE;
		break;
	}
	phs += (int32)(*p->xcps * MAXPHASE / ekr);
	phs &= MAXMASK;
	p->phs = phs;
	*p->res = *p->kamp * res;
}

void lfoa(LFO *p)
{
	int32	nsmps = ksmps;
	int32	phs;
	Number	fract;
	Number	res = 0;
	int32	iphs, inc;
	Number	*ar, amp;

	phs = p->phs;
	if (p->lasttype) inc = (int32)(*p->xcps * 4096.0f /esr);
	else inc = (int32)(*p->xcps * (Number)MAXPHASE /esr);
    
	amp = *p->kamp;
	ar = p->res;
	do {
		iphs = phs >> 12;
		switch (p->lasttype) {
		case 0:
			fract = (Number)(phs & 0xfff)/4096.0f;
			res = p->sine[iphs];
			res = res + (p->sine[iphs+1]-res)*fract;
			break;      
		case 1:			/* Trangular */
			res = (Number)((phs<<2)&MAXMASK)/(Number)MAXPHASE;
			if (phs < MAXPHASE/4) {}
			else if (phs < MAXPHASE/2)
			    res = 1.0f - res;
			else if (phs < 3*MAXPHASE/4)
			    res = - res;
			else
			    res = res - 1.0f;
			break;
		case 2:			/* Bipole square wave */
			if (phs<MAXPHASE/2) res = 1.0f;
			else res = -1.0f;
			break;
		case 3:			/* Unipolar square wave */
			if (phs<MAXPHASE/2) res = 1.0f;
			else res = 0.0f;
			break;
		case 4:			/* Saw Tooth */
			res = (Number)phs/(Number)MAXPHASE;
			break;
		case 5:			/* Reverse Saw Tooth */
			res = 1.0f - (Number)phs/(Number)MAXPHASE;
			break;
		}
		phs += inc;
		phs &= MAXMASK;
		*ar++ = res * amp;
	} while (--nsmps);
	p->phs = phs;
}

Opcode opcodes[] = 

{
	LFO_OPCODE_LIST,
	{ NULL }
};
