/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: harmon.cc,v 1.1 1999/11/17 00:45:00 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>

#include "harmon.h"

void
harmset(HARMON * p)
{
	Number minfrq = *p->ilowest;

	if (minfrq < 64.0f) {
		p->error ("harmon: minimum frequency too low");
	}

	if (p->auxch.auxp == 0 || minfrq < p->minfrq) {
		int32 nbufs = (int32) (ekr * 3.0f / minfrq) + 1;
		int32 nbufsmps = nbufs * ksmps;
		int32 maxprd = (int32) (esr / minfrq);
		int32 totalsiz = nbufsmps * 4 + maxprd;

		p->auxch.alloc (totalsiz * sizeof (Number));

		p->bufp = (Number *) p->auxch.auxp;
		p->midp = p->bufp + nbufsmps;	/* each >= maxprd * 3 */
		p->bufq = p->midp + nbufsmps;
		p->midq = p->bufq + nbufsmps;
		p->autobuf = p->midq + nbufsmps;	/* size of maxprd */
		p->nbufsmps = nbufsmps;
		p->n2bufsmps = nbufsmps * 2;
		p->lomaxdist = maxprd;
		p->minfrq = minfrq;
	}
	if ((p->autoktim = (int32) (*p->iptrkprd * ekr + 0.5f)) < 1)
		p->autoktim = 1;
	p->autokcnt = 1;	/* init for immediate autocorr attempt */
	p->sicvt = 65536.0f * onedsr;
	p->cpsmode = ((*p->icpsmode != 0.0f));
	p->inp1 = p->bufp;
	p->inp2 = p->midp;
	p->inq1 = p->bufq;
	p->inq2 = p->midq;
	p->puls1 = 0;
	p->puls2 = 0;
	p->puls3 = 0;
	p->prvest = 0.0f;
	p->prvq = 0.0f;
	p->phase1 = 0;
	p->phase2 = 0;
	p->rngflg = 0;
}

void
harmon(HARMON * p)
{
	Number *src1, *src2, *src3, *inp1, *inp2, *outp;
	Number c1, c2, qval, *inq1, *inq2;
	Number sum, minval, *minqp = 0;
	Number *minq1, *minq2, *endp;
	Number *pulstrt, lin1, lin2, lin3;
	int32 cnt1, cnt2, cnt3;
	int32 nn, nsmps, phase1, phase2, phsinc1, phsinc2, period;

	inp1 = p->inp1;
	inp2 = p->inp2;
	inq1 = p->inq1;
	inq2 = p->inq2;
	qval = p->prvq;
	if (*p->kest != p->prvest && *p->kest != 0.0f) {	/* if new pitch 
								   estimate */
		Number estperiod = esr / *p->kest;
		Number b = 2.0f - (Number) cos(*p->kest * tpidsr);
		p->c2 = b - (Number) sqrt((double) (b * b) - 1.0);	/*  
									   recalc lopass coefs */
		p->c1 = 1.0f - p->c2;
		p->prvest = *p->kest;
		p->estprd = estperiod;
		p->prvar = 0.0f;
	}
	if (*p->kvar != p->prvar) {
		Number oneplusvar = 1.0f + *p->kvar;
		p->mindist = (int32) (p->estprd / oneplusvar);	/* prd
								   window is prd +/- var int */
		p->maxdist = (int32) (p->estprd * oneplusvar);
		if (p->maxdist > p->lomaxdist)
			p->maxdist = p->lomaxdist;
		p->max2dist = p->maxdist * 2;
		p->prvar = *p->kvar;
	}
	c1 = p->c1;
	c2 = p->c2;
	for (src1 = p->asig, nsmps = ksmps; nsmps--; src1++) {
		*inp1++ = *inp2++ = *src1;	/* dbl store the wavform */
		if (*src1 > 0.0f)
			qval = c1 * *src1 + c2 * qval;	/*  & its
							   half-wave rect */
		else
			qval = c2 * qval;
		*inq1++ = *inq2++ = qval;
	}
	if (!(--p->autokcnt)) {	/* if time for new autocorr  */
		Number *mid1, *mid2, *src4;
		Number *autop, *maxp;
		Number dsum, dinv, win, windec, maxval;
		int32 dist;
		p->autokcnt = p->autoktim;
		mid2 = inp2 - p->max2dist;
		mid1 = mid2 - 1;
		autop = p->autobuf;
		for (dist = p->mindist; dist <= p->maxdist; dist++) {
			dsum = 0.0f;
			dinv = 1.0f / dist;
			src1 = mid1;
			src3 = mid1 + dist;
			src2 = mid2;
			src4 = mid2 + dist;
			for (win = 1.0f, windec = dinv, nn = dist; nn--;) {
				dsum += win * (*src1 * *src3 + *src2 * *src4);
				src1--;
				src2++;
				src3--;
				src4++;
				win -= windec;
			}
			*autop++ = dsum * dinv;
		}
		maxval = 0.0f;
		maxp = autop = p->autobuf;
		endp = autop + p->maxdist - p->mindist;
		while (autop < endp) {
			if (*autop > maxval) {	/* max autocorr gives new 
						   period */
				maxval = *autop;
				maxp = autop;
			}
			autop++;
		}
		period = p->mindist + maxp - p->autobuf;
		if (period != p->period) {
			p->period = period;
			if (!p->cpsmode)
				p->sicvt = 65536.0f / period;
			p->pnt1 = (int32) ((Number) period * 0.2f);
			p->pnt2 = (int32) ((Number) period * 0.8f);
			p->pnt3 = period;
			p->inc1 = 1.0f / p->pnt1;
			p->inc2 = 1.0f / (period - p->pnt2);
		}
	} else
		period = p->period;

	minval = (Number) HUGE_VAL;	/* Suitably large ! */
	minq2 = inq2 - period;	/* srch the qbuf for minima */
	minq1 = minq2 - period;	/* which are 1 period apart */
	endp = inq2;		/* move srch over 1 period  */
	while (minq2 < endp) {
		if ((sum = *minq1 + *minq2) < minval) {
			minval = sum;
			minqp = minq1;
		}
		minq1++;
		minq2++;
	}
	src1 = minqp - p->n2bufsmps;	/* get src equiv of 1st min  */
	while (src1 + ksmps > inp2)	/* if not enough smps presnt */
		src1 -= period;	/*      back up 1 prd        */
	pulstrt = src1;		/* curr available pulse beg  */

	src1 = p->puls1;	/* insert pulses into output */
	src2 = p->puls2;
	src3 = p->puls3;
	lin1 = p->lin1;
	lin2 = p->lin2;
	lin3 = p->lin3;
	cnt1 = p->cnt1;
	cnt2 = p->cnt2;
	cnt3 = p->cnt3;
	phase1 = p->phase1;
	phase2 = p->phase2;
	phsinc1 = (int32) (*p->kfrq1 * p->sicvt);
	phsinc2 = (int32) (*p->kfrq2 * p->sicvt);
	outp = p->ar;
	nsmps = ksmps;
	do {
		Number sum;
		if (src1 != 0) {
			if (++cnt1 < p->pnt11) {
				sum = *src1++ * lin1;
				lin1 += p->inc11;
			} else if (cnt1 <= p->pnt12)
				sum = *src1++;
			else if (cnt1 <= p->pnt13) {
				sum = *src1++ * lin1;
				lin1 -= p->inc12;
			} else {
				sum = 0.0f;
				src1 = 0;
			}
		} else
			sum = 0.0f;
		if (src2 != 0) {
			if (++cnt2 < p->pnt21) {
				sum += *src2++ * lin2;
				lin2 += p->inc21;
			} else if (cnt2 <= p->pnt22)
				sum += *src2++;
			else if (cnt2 <= p->pnt23) {
				sum += *src2++ * lin2;
				lin2 -= p->inc22;
			} else
				src2 = 0;
		}
		if (src3 != 0) {
			if (++cnt3 < p->pnt31) {
				sum += *src3++ * lin3;
				lin3 += p->inc31;
			} else if (cnt3 <= p->pnt32)
				sum += *src3++;
			else if (cnt3 <= p->pnt33) {
				sum += *src3++ * lin3;
				lin3 -= p->inc32;
			} else
				src3 = 0;
		}
		if ((phase1 += phsinc1) & 0xFFFF0000L) {
			phase1 &= 0x0000FFFFL;
			if (src1 == 0) {
				src1 = pulstrt;
				cnt1 = 0;
				lin1 = p->inc1;
				p->inc11 = p->inc1;
				p->inc12 = p->inc2;
				p->pnt11 = p->pnt1;
				p->pnt12 = p->pnt2;
				p->pnt13 = p->pnt3;
			} else if (src2 == 0) {
				src2 = pulstrt;
				cnt2 = 0;
				lin2 = p->inc1;
				p->inc21 = p->inc1;
				p->inc22 = p->inc2;
				p->pnt21 = p->pnt1;
				p->pnt22 = p->pnt2;
				p->pnt23 = p->pnt3;
			} else if (src3 == 0) {
				src3 = pulstrt;
				cnt3 = 0;
				lin3 = p->inc1;
				p->inc31 = p->inc1;
				p->inc32 = p->inc2;
				p->pnt31 = p->pnt1;
				p->pnt32 = p->pnt2;
				p->pnt33 = p->pnt3;
			} else if (++p->rngflg > 200) {
				info << "harmon: out of range" << endmsg;
				p->rngflg = 0;
			}
		}
		if ((phase2 += phsinc2) & 0xFFFF0000L) {
			phase2 &= 0x0000FFFFL;
			if (src1 == 0) {
				src1 = pulstrt;
				cnt1 = 0;
				lin1 = p->inc1;
				p->inc11 = p->inc1;
				p->inc12 = p->inc2;
				p->pnt11 = p->pnt1;
				p->pnt12 = p->pnt2;
				p->pnt13 = p->pnt3;
			} else if (src2 == 0) {
				src2 = pulstrt;
				cnt2 = 0;
				lin2 = p->inc1;
				p->inc21 = p->inc1;
				p->inc22 = p->inc2;
				p->pnt21 = p->pnt1;
				p->pnt22 = p->pnt2;
				p->pnt23 = p->pnt3;
			} else if (src3 == 0) {
				src3 = pulstrt;
				cnt3 = 0;
				lin3 = p->inc1;
				p->inc31 = p->inc1;
				p->inc32 = p->inc2;
				p->pnt31 = p->pnt1;
				p->pnt32 = p->pnt2;
				p->pnt33 = p->pnt3;
			} else if (++p->rngflg > 200) {
				info << "harmon: out of range" << endmsg;
				p->rngflg = 0;
			}
		}
		*outp++ = sum;
	} while (--nsmps);
	if (inp1 >= p->midp) {
		p->inp1 = p->bufp;
		p->inp2 = p->midp;
		p->inq1 = p->bufq;
		p->inq2 = p->midq;
		if (src1 != 0)
			src1 -= p->nbufsmps;
		if (src2 != 0)
			src2 -= p->nbufsmps;
		if (src3 != 0)
			src3 -= p->nbufsmps;
	} else {
		p->inp1 = inp1;
		p->inp2 = inp2;
		p->inq1 = inq1;
		p->inq2 = inq2;
	}
	p->puls1 = src1;
	p->puls2 = src2;
	p->puls3 = src3;
	p->lin1 = lin1;
	p->lin2 = lin2;
	p->lin3 = lin3;
	p->cnt1 = cnt1;
	p->cnt2 = cnt2;
	p->cnt3 = cnt3;
	p->phase1 = phase1;
	p->phase2 = phase2;
	p->prvq = qval;
}

Opcode opcodes[] =
{
	HARMON_OPCODE_LIST,
	{ 0 }
};
