/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: generators.cc,v 1.2 1999/11/17 13:32:13 pbd Exp $
*/

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/quasimodo.h>
#include <quasimodo/tableerror.h>
#include <quasimodo/generators.h>

#include "random.h"

void gen02 (FunctionTableRequest *req, FunctionTable *ftp) 
{
        /* read ftable values directly from p-args */

	Number *fp = ftp->ftable;
	Number *pp;
	size_t nvals = req->argc;

	if (nvals < 1) {
		return;
	}

	pp = &req->args[1];

	if (nvals > FunctionTable::MaximumTableSize) {
		nvals = FunctionTable::MaximumTableSize - 1;	
	}

	do {
		*fp++ = *pp++;	/*   copy into ftable   */
	} while (--nvals);
}

void gen03 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int ncoefs;
	Number xintvl, xscale;
	int xloc, nlocs;
	Number *fp = ftp->ftable;
	Number x, sum, *coefp, *coef0, *coeflim;
	
	if ((ncoefs = req->argc) <= 0) {
		error << "fgen03: no coefficients present" << endmsg;

		return;
	}

	coef0 = &req->args[3];

	coeflim = coef0 + ncoefs;
	if ((xintvl = req->args[2] - req->args[1]) <= 0) {
		error << "fgen03: illegal x interval" << endmsg;

		return;
	}

	xscale = xintvl / (Number) ftp->flen;
	xloc = (int) (req->args[1] / xscale);	/* initial xloc */
	nlocs = ftp->flen + 1;

	do {			/* for each loc:        */
		x = xloc++ * xscale;
		coefp = coeflim;
		sum = *--coefp;	/* init sum to coef(n)  */
		while (coefp > coef0) {
			sum *= x;	/*  & accum by Horner's rule */
			sum += *--coefp;
		}
		*fp++ = sum;
	} while (--nlocs);
}

void gen04 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	return;
}

void gen05 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int	nsegs, seglen;
        Number	*valp, *fp, *finp;
        Number	amp1, mult;

	if (((nsegs = req->argc) >> 1) <= 0)	{ /* nsegs = nargs-1 /2 */
		return;
	}

	valp = &req->args[1];
	fp = ftp->ftable;
	finp = fp + ftp->flen;

	if (*valp == 0) {
		error << "illegal input vals for gen call, beginning:" << endmsg;

		return;
	}

	do {	
		amp1 = *valp++;
		if (!(seglen = (int)*valp++)) {
			continue;
		}

		if (seglen < 0) {
			error << "gen call has negative segment size:" << endmsg;

			return;
		}

		if ((mult = *valp/amp1) <= 0) {
			error << "gen call has negative segment size:" << endmsg;

			return;
		}
		mult = (float)pow( (double)mult, 1.0/(double)seglen );
		while (seglen--) {
			*fp++ = amp1;
			amp1 *= mult;
			if (fp > finp) return;
		}
	} while (--nsegs);

	if (fp == finp)	{		/* if 2**n pnts, add guardpt */
		*fp = amp1;
	}

	return;
}

void gen06 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	Number	*segp, *extremp, *inflexp, *segptsp, *fp, *finp;
	Number	y, diff2;
        int	pntno, pntinc, nsegs, npts;

	if (((nsegs = req->argc) >>1) < 1) {
		error << "fgen06: insufficient args" << endmsg;

		return;
	}

	fp = ftp->ftable;
	finp = fp + ftp->flen;

	pntinc = 1;

	for (segp = &req->args[1]; nsegs > 0; nsegs--) {
		segptsp = segp + 1;
		if ((npts = (int)*segptsp) < 0) {
			error << "fgen06: negative segsiz" << endmsg;

			return;
		}
		if (pntinc > 0) {
			pntno = 0;
			inflexp = segp + 2;
			extremp = segp;
		}
		else {
			pntno = npts;
			inflexp = segp;
			extremp = segp + 2;
		}
		diff2 = (*inflexp - *extremp) / 2.0f;
		for ( ; npts > 0 && fp < finp; pntno += pntinc, npts--) {
			y = (float)pntno / *segptsp;
			*fp++ = (3.0f-y) * y * y * diff2 + *extremp;
		}
		pntinc = -pntinc;
		segp += 2;
	}
	*fp = *(segp + 2);			/* write last target point */
}

void gen07 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int	nsegs, seglen;
        Number	*valp, *fp, *finp;
        Number	amp1, incr;
	TableError err;

	if ((nsegs = (req->argc >> 1)) <= 0) {
		err << "gen07: incorrect or insufficient "
			"number of arguments ("
		    << req->argc
		    << ')'
		    << endmsg;
	}

	valp = &req->args[1];
	fp = ftp->ftable;
	finp = fp + ftp->flen;

	do {	
		amp1 = *valp++;
		if (!(seglen = (int)*valp++)) {
			continue;
		}

		if (seglen < 0) {
			error << "gen call has negative segment size:" << endmsg;

			return;
		}

		incr = (*valp - amp1) / seglen;
		while (seglen--) {
			*fp++ = amp1;
			amp1 += incr;

			if (fp > finp) {
				return;
			}
		}
	} while (--nsegs);

	if (fp == finp)			/* if 2**n pnts, add guardpt */
		*fp = amp1;

        return;
}

void gen08 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	Number	R, x, c3, c2, c1, c0, *fp, *fplim, *valp;
	Number	f2 = 0;
	Number  f1, f0, df1, df0, dx01, dx02, curx;
	Number  dx12 = 0;
	Number	slope, resd1, resd0;
	int	nsegs, npts;

	if (((nsegs = req->argc) >> 1) <= 0) {
		error << "fgen08: insufficient args" << endmsg;
		return;
	}

	valp = &req->args[1];
	fp = ftp->ftable;
	fplim = fp + ftp->flen;

	f0 = *valp++;			/* 1st 3 params give vals at x0, x1  */
	if ((dx01 = *valp++) <= 0.0f) {	/*	and dist between	     */
		error << "fgen08: illegal x interval" << endmsg;

		return;
	}
	f1 = *valp++;
	curx = df0 = 0.0f;		/* init x to origin; slope at x0 = 0 */
	do {				/* for each spline segmnt (x0 to x1) */
	    if (nsegs > 1) {			/* if another seg to follow  */
		if ((dx12 = *valp++) <= 0.) {	/*    read its distance	     */
			error << "fgen08: illegal x interval" << endmsg;

			return;
		}
		f2 = *valp++;			/*    and the value at x2    */
		dx02 = dx01 + dx12;
		df1 = ( f2*dx01*dx01 + f1*(dx12-dx01)*dx02 - f0*dx12*dx12 )
			/ (dx01*dx02*dx12);
	    }				   /* df1 is slope of parabola at x1 */
	    else df1 = 0.0f;
	    if ((npts = (int)(dx01 - curx)) > fplim - fp)
		npts = fplim - fp;
	    if (npts > 0) {			/* for non-trivial segment: */
		slope = (f1 - f0) / dx01;	/*   get slope x0 to x1	    */
		resd0 = df0 - slope;		/*   then residual slope    */
		resd1 = df1 - slope;		/*     at x0 and x1	    */
		c3 = (resd0 + resd1) / (dx01*dx01);
		c2 = - (resd1 + 2.0f*resd0) / dx01;
		c1 = df0;			/*   and calc cubic coefs   */
		c0 = f0;
		for (x = curx; npts>0; --npts, x += 1.0f) {
		    R = c3;
		    R *= x;
		    R += c2;	     /* f(x) = ((c3 x + c2) x + c1) x + c0  */
		    R *= x;
		    R += c1;
		    R *= x;
		    R += c0;
		    *fp++ = R;			/* store n pts for this seg */
		}
		curx = x;
	    }
	    curx -= dx01;		/* back up x by length last segment */
	    dx01 = dx12;		/* relocate to the next segment	*/
	    f0 = f1;			/*   by assuming its parameters	*/
	    f1 = f2;
	    df0 = df1;
	}
	while (--nsegs && fp<fplim);	/* loop for remaining segments	*/
	while (fp <= fplim)
	    *fp++ = f0;			/* & repeat the last value	*/

	return;
}

void gen09 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int	hcnt;
        Number	*valp, *fp, *finp;
	Dumber	phs, inc, amp;

	if ((hcnt = (req->argc / 3)) <= 0) { /* hcnt = nargs / 3 */
		error << "fgen09: insufficient arguments" << endmsg;

		return;
	}

	valp = &req->args[1];
	finp = &ftp->ftable[ftp->flen];

	do {
		for (inc=(*valp++)*ftp->tpdlen, amp=(*valp++),
			     phs=(*valp++)*tpd360, 
			     fp=ftp->ftable; fp<=finp; fp++) {
			*fp += (float)(sin(phs) * amp);

			if ((phs += inc) >= twopi) {
				phs -= twopi;
			}
		}

	} while (--hcnt);

	return;
}

void gen10 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	long	phs, hcnt;
        Number	amp, *fp, *finp;

	if ((hcnt = req->argc) <= 0) { 
		error << "fgen10: insufficient arguments" << endmsg;

		return;
	}

	finp = &ftp->ftable[ftp->flen];

	do {
		if ((amp = req->args[hcnt]) != 0) { /* for non-0 amps,  */

			for (phs = 0, fp = ftp->ftable; fp <= finp; fp++) {
                                /* accum sin pts  */
				*fp += (float)sin(phs*ftp->tpdlen) * amp; 
				phs += hcnt;      /* phsinc is hno */
				phs &= ftp->lenmask;
			}
		}
	} while (--hcnt);

	return;
}

void gen11 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	Number  *fp, *finp;
        int32   phs;
	Dumber	x;
	Number	denom, r, scale;
	int16	n, k;
	int16   nargs;

	nargs = req->argc;

	if (nargs < 1) {
		error << "fgen11: insufficient arguments" << endmsg;

		return;
	}

	if ((n = (int) req->args[1]) < 1) {
		error << "fgen11: nh partials < 1" << endmsg;

		return;
	}
	k = 1;
	r = 1.0f;

	if (nargs > 1) {
		k = (int)req->args[2];
	}
	if (nargs > 2) {
		r = req->args[3];
	}

	fp = ftp->ftable;
	finp = fp + ftp->flen;

	if (nargs == 1 || k == 1 && r == 1.) {     /* simple "buzz" case */
		int16 tnp1;
		Number pdlen;

		tnp1 = (n << 1) + 1;
		scale = 0.5f / n;
		pdlen = (Number)ftp->tpdlen / 2.0f;
		for (phs = 0; fp <= finp; phs++) {
			x = phs * pdlen;
			if (!(denom = (Number)sin(x)))
				*fp++ = 1.0f;
			else *fp++ = ((Number)sin(tnp1 * x) / denom - 1.0f) * scale;
		}
	}
	else {                                   /* complex "gbuzz" case */
		Number numer, twor, rsqp1, rtn, rtnp1, absr;
		int16   km1, kpn, kpnm1;

		km1   = k - 1;
		kpn   = k + n;
		kpnm1 = kpn - 1;
		twor  = r * 2.0f;
		rsqp1 = r * r + 1.0f;
		rtn   = (Number)pow((double) r, (double) n);
		rtnp1 = rtn * r;
		if ((absr = (Number)fabs(r)) > .999f && absr < 1.001f)
			scale = 1.0f / n;
		else scale = (1.0f - absr) / (1.0f - (Number)fabs(rtn));
		for (phs=0; fp <= finp; phs++) {
			x = phs * ftp->tpdlen;
			numer = (Number)cos(x*k) - r *
				(Number)cos(x*km1) - rtn * (Number)cos(x*kpn)
				+ rtnp1 * (Number)cos(x*kpnm1);
			if ((denom = rsqp1 - twor*(Number)cos(x)) > .0001f
			  || denom < -.0001f)
			  	*fp++ = numer / denom * scale;
			else *fp++ = 1.0f;
		}
	}

	return;
}

void gen12 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	static double coefs[] = { 3.5156229, 3.0899424, 1.2067492,
			  0.2659732, 0.0360768, 0.0045813 };
	double *coefp, sum, tsquare, evenpowr, *cplim = coefs + 6;
        size_t    n;
        Number	*fp;
        double xscale;

	if (req->argc < 1) {
		error << "fgen12: insufficient arguments" << endmsg;

		return;
	}

	xscale = (double) req->args[1] / ftp->flen / 3.75;

	for (n=0,fp=ftp->ftable; n<=ftp->flen; n++) {
	        tsquare = (double) n * xscale;
		tsquare *= tsquare;
		for (sum=evenpowr=1.0, coefp=coefs; coefp<cplim; coefp++) {
			evenpowr *= tsquare;
			sum += *coefp * evenpowr;
	        }
		*fp++ = (float) log(sum);
	}
	return;
}

static void gen1314 (FunctionTableRequest *req, FunctionTable *ftp,
		     Number mxval, Number mxscal)
{
	int32	nh, nn;
        Number	*mp, *mspace, *hp, *oddhp;
	Number	xamp, xintvl, scalfac, sum, prvm;
	TableError err;

	if ((nh = req->argc - 2) <= 0) {
		err << "gen13/14: insufficient args" << endmsg;
	}
	if ((xintvl = req->args[5]) <= 0) {
		err << "gen13/14: illegal xint value" << endmsg;
	}
	if ((xamp = req->args[6]) <= 0) {
		err << "gen13/14: illegal xamp value" << endmsg;
	}

	req->args[5] = -xintvl;
	req->args[6] = xintvl;

        nn = nh * sizeof(Number) / 2;	    /* alloc spc for terms 3,5,7,... */
	mp = mspace = new Number[nn];/* of 1st row of matrix, and */
	for (nn = (nh + 1) >>1; --nn; ) {   /* form array of non-0 terms */
		*mp++ = mxval = -mxval;     /*  -val, val, -val, val ... */
	}
	scalfac = 2 / xamp;
	hp = &req->args[7];                 /* beginning with given h0,  */

	do {
		mp = mspace;
		oddhp = hp;
		sum = *oddhp++;			/* sum = diag(=1) * this h   */
		for (nn = (nh+1) >>1; --nn; ) {
			oddhp++;		/*  + odd terms * h+2,h+4,.. */
			sum += *mp++ * *oddhp++;
		}
		*hp++ = sum * mxscal;		/* repl this h w. coef (sum) */
		mp = mspace;
		prvm = 1.0f;
		for (nn = nh>>1; --nn > 0; mp++)/* calc nxt row matrix terms */
			*mp = prvm = *mp - prvm;
		mxscal *= scalfac;
	} while (--nh);				/* loop til all h's replaced */

	delete [] mspace;

	gen03 (req, ftp); /* then call gen03 to write */
}

void gen13 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	gen1314 (req, ftp, 2.0f, 0.5f);
}

void gen14 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	gen1314 (req, ftp, 1.0f, 1.0f);
}

void gen15 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	return;
}

void gen17 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int	nsegs, ndx, nxtndx;
        float	*valp, *fp, *finp;
        float	val;

	if ((nsegs = req->argc >> 1) <= 0) {	     /* nsegs = nargs /2 */
	    goto gn17err;
	}
	valp = &req->args[5];

	fp = ftp->ftable;
	finp = fp + ftp->flen;

	if ((ndx = (int)*valp++) != 0) {
	    goto gn17err;
	}

	while (--nsegs) {
	    val = *valp++;
	    if ((nxtndx = (int)*valp++) <= ndx) {
		goto gn17err;
	    }

	    do {
	        *fp++ = val;
		if (fp > finp) return;
	    } while (++ndx < nxtndx);
	}
	val = *valp;

	while (fp <= finp) {			/* include 2**n + 1 guardpt */
		*fp++ = val;
	}

        return;

  gn17err: 
	TableError err;
	err << "gen17: illegal x-ordinate values:" << endmsg;
}

void gen19 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int16	hcnt;
        Number	*valp, *fp, *finp;
	Dumber	phs, inc, amp, dc;
	Dumber  tpdlen;
	if ((hcnt = req->argc / 4) <= 0) {		/* hcnt = nargs / 4 */
		TableError err;
		err << "gen19: wrong number of arguments" << endmsg;
	}

	valp = &req->args[5];
	finp = &ftp->ftable[ftp->flen];
	tpdlen = ftp->tpdlen;

	do {
		for (inc=(*valp++) *tpdlen, amp=(*valp++),
			     phs=(*valp++)*tpd360, dc=(*valp++),
			     fp=ftp->ftable; 
		     fp <= finp; 
		     fp++) {

			/* dc after str scale */
			*fp += (Number)(sin(phs) * amp + dc);
			if ((phs += inc) >= TWOPI)
			    phs -= TWOPI;
		}
	} while (--hcnt);
}

static Number besseli (Dumber x)
{
	Number ax, ans;
	Dumber y;
	
	if (( ax = (Number)fabs( x)) < 3.75f)     {
		y = x / 3.75;
		y *= y;
		ans = (Number)(1.0 + y * ( 3.5156229 +
					   y * ( 3.0899424 +
						 y * ( 1.2067492 +
						       y * ( 0.2659732 +
							     y * ( 0.360768e-1 +
								   y * 0.45813e-2))))));
	}
	else {
		y = 3.75 / ax;
		ans = (Number)((exp ( ax) / sqrt(ax))
			       * (0.39894228 +
				  y * (0.1328592e-1 +
				       y * (0.225319e-2 +
					    y * (-0.157565e-2 +
						 y * (0.916281e-2 +
						      y * (-0.2057706e-1 +
							   y * (0.2635537e-1 +
								y * (-0.1647633e-1 +
								     y * 0.392377e-2)))))))));
	}
	return ans;
}

/*  GEN20 and GEN21 by Paris Smaragdis 1994 B.C.M. Csound development team  */

void gen20 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	
	Number cf[4], *ft;
	Dumber arg, i, xarg, beta;
	size_t nargs;
	size_t flen;

	ft = ftp->ftable;
	flen = ftp->flen;

	nargs = req->argc;

	if (nargs > 2) {
		xarg = req->args[2];
		beta = req->args[3];
	} else if (nargs == 2) {
		xarg = req->args[2];
		beta = 0;
	} else {
		xarg = 1.0;
		beta = 0;
	}


	switch((int) req->args[1])  {
	case 1:
		cf[0] = .54f;
		cf[1] = .46f;
		cf[2] = cf[3] = 0.0f;
		break;
	case 2:
		cf[0] = cf[1] = .5f;
		cf[2] = cf[3] = 0.0f;
		break;
	case 3:
		arg = 2.0/flen;
		for (i = 0.0 ; i < flen/2.0 ; i++)
		    *ft++ = (Number)(i*arg*xarg);
		for (; i < flen ; i++)
		    *ft++ = (Number)((2.0 - i*arg)*xarg);
		return;
	case 4:
		cf[0] = .42f;
		cf[1] = .5f;
		cf[2] = .08f;
		cf[3] = 0.0f;
		break;
	case 5:
		cf[0] = .35878f;
		cf[1] = .48829f;
		cf[2] = .14128f;
		cf[3] = .01168f;
		break;
	case 6:
		arg = 12.0/flen;
		for (i = -6.0 ; i < 0.0 ; i += arg)
		    *ft++ = (Number)(xarg * (pow( 2.71828, -(i*i)/2.0)));
		for (i = arg ; i < 6.0 ; i += arg)
		    *ft++ = (Number)(xarg * (pow( 2.71828, -(i*i)/2.0)));
		return;
	case 7:
		for (i = -flen/2.0 + .1 ; i < flen/2 ; i++)
		    *ft++ = (Number)(xarg *
				     besseli((beta * sqrt(1.0-pow((2.0*i/(flen - 1)), 2.0)))) /
				     besseli( beta));
		return;
	case 8:
		for (i = 0 ; i < flen ; i++)
		    *ft++ = 1.0f;
		return;
	case 9:
		arg = TWOPI / flen;
		for (i = -PI ; i < 0 ; i += arg)
		    *ft++ = (Number)(xarg * sin(i) / i);
		*ft++ = 1.0f;
		for (i = arg ; i < PI ; i += arg)
		    *ft++ = (Number)(xarg * sin(i) / i);
		return;
	default:
		TableError err;
		err << "gen20: no such window" << endmsg;
	}

	arg = TWOPI/flen;

	for (i = 0.0 ; i < TWOPI ; i += arg)
	    *ft++ = (Number)(xarg * (cf[0] - cf[1]*cos(i) +
				     cf[2]*cos(2.0*i) - cf[3]*cos(3.0*i)));
}


void gen21 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	size_t i;
	size_t flen;
	Number *ft;
	TableError err;

	ft = ftp->ftable;
	flen = ftp->flen;

	if (req->argc < 1)  {		/* All need at least 1 argument */
		TableError err;
		err << "gen21: wrong number of input arguments ("
		    << req->argc
		    << ')'
		    << endmsg;
	}

	switch ((int)req->args[5])  {
	case 1:			/* Uniform distribution */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = unifrand((Number)req->args[6]);
		break;
	case 2:			/* Linear distribution */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = linrand( (Number) req->args[6]);
		break;
	case 3:			/* Triangular about 0.5 */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = trirand( (Number) req->args[6]);
		break;
	case 4:			/* Exponential */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = exprand( (Number) req->args[6]);
		break;
	case 5:			/* Bilateral exponential */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = biexprand( (Number) req->args[6]);
		break;
	case 6:			/* Gaussian distribution */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = gaussrand( (Number) req->args[6]);
		break;
	case 7:			/* Cauchy distribution */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = cauchrand( (Number) req->args[6]);
		break;
	case 8:			/* Positive Cauchy */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = pcauchrand( (Number) req->args[6]);
		break;
	case 9:			/* Beta distribution */
		if (req->argc < 3)  {
			err << "gen21: Wrong number of input arguments"
			    << endmsg;
		}
		for (i = 0 ; i < flen ; i++)
		    *ft++ = betarand((Number)req->args[6],(Number)req->args[7],(Number)req->args[8]);
		break;
	case 10:			/* Weibull Distribution */
		if (req->argc < 2)  {
			err << "gen21: Wrong number of input arguments"
			    << endmsg;
		}
		for (i = 0 ; i < flen ; i++)
		    *ft++ = weibrand( (Number) req->args[6], (Number) req->args[7]);
		break;
	case 11:			/* Poisson Distribution */
		for (i = 0 ; i < flen ; i++)
		    *ft++ = poissrand( (Number) req->args[6]);
		break;
	default:
		err << "gen21: unknown distribution ("
		    << (int) req->args[5]
		    << endmsg;
	}

}

void gen22 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	return;
}

void gen22raw (FunctionTableRequest *req, FunctionTable *ftp) 
{
	return;
}

void gen23 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	return;
}

void gen24 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	return;
}

void gen25 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int16	nsegs,  seglen;
	Number	*valp, *fp, *finp;
	Number	x1, x2, y1, y2, mult;
	TableError err;
	size_t flen;

	if ((nsegs = ((req->argc / 2) - 1) )<= 0) return;
	valp = &req->args[2];
	fp = ftp->ftable;
	flen = ftp->flen;
	finp = fp + flen;

	do {
		x1 = *valp++;
		y1 =  *valp++;
		x2 = *valp++;
		if (nsegs > 1)
		    y2 =  *valp++;
		else
		    y2 = *valp;
		if (x2 < x1) goto gn25err;
		if (x1 > flen || x2 > flen) goto gn25err2;
		seglen = (int16)(x2-x1);
		if (y1 <= 0 || y2 <= 0) goto gn25err3;
		mult = y2/y1;
		mult = (Number)pow( (Dumber)mult, 1.0/(Dumber)seglen );
		while (seglen--) {
			*fp++ = y1;
			y1 *= mult;
			if (fp > finp) return;
		}
		valp -= 2;
	} while (--nsegs);
	if (fp == finp)			/* if 2**n pnts, add guardpt */
	    *fp = y1;
	return;

  gn25err:
	err << "gen25: x coordindates must all be in increasing order:"
	    << endmsg;

  gn25err2:
	err << "gen25: x coordindate greater than function size:"
	    << endmsg;

  gn25err3:
	err << "gen25: illegal input val (y <= 0) for gen call, beginning:"
	    << endmsg;
}

void gen27 (FunctionTableRequest *req, FunctionTable *ftp) 
{
	int16	nsegs;
	Number	*valp, *fp, *finp;
	Number	x1, x2, y1, y2, seglen, incr;
	TableError err;
	size_t flen;

	if ((nsegs = ((req->argc / 2) - 1) )<= 0) return;
	valp = &req->args[2];
	fp = ftp->ftable;
	flen = ftp->flen;
	finp = fp + flen;

	do {
		x1 = *valp++;
		y1 = *valp++;
		x2 = *valp++;
		if (nsegs > 1)
		    y2 =  *valp++;
		else
		    y2 = *valp;
		if (x2 < x1) goto gn27err;
		if (x1 > flen || x2 > flen) goto gn27err2;
		seglen = x2-x1;
		incr = (y2 - y1) / seglen;
		while (seglen--) {
			*fp++ = y1;
			y1 += incr;
			if (fp > finp) return;
		}
		valp -= 2;
	} while (--nsegs);
	if (fp == finp)			/* if 2**n pnts, add guardpt */
	    *fp = y1;
	return;

  gn27err:
	err << "gen27: x coordindates must all be in increasing order:"
	    << endmsg;
  gn27err2:
	err << "gen27: x coordindate greater than function size:" 
	    << endmsg;
}

void gen28 (FunctionTableRequest *req, FunctionTable *ftp) 
{
#if NEED_STL_CONTAINER_NOT_REALLOC

	Number	*fp, *finp;
	int16 	seglen, resolution=100;
	FILE	*filp;
	int16 i=0, j=0;
	Number *x, *y, *z;
	int16 arraysize = 1000;
	Number x1, y1, z1, x2, y2, z2, incrx, incry;
	TableError err;
	size_t flen;


	if ((filp = fopen(req->filename, "r" )) == NULL) goto gen28err1;
    
	x = new Number[arraysize];
	y = new Number[arraysize];
	z = new Number[arraysize];

	while (fscanf( filp, "%f%f%f", &z[i], &x[i], &y[i])!= EOF) {
		i++;
		if (i>=arraysize) {
			arraysize += 1000;
			x = (Number*)mrealloc(x, arraysize*sizeof(Number));
			y = (Number*)mrealloc(y, arraysize*sizeof(Number));
			z = (Number*)mrealloc(z, arraysize*sizeof(Number));
		}
	}
	--i;

	flen = (long)(z[i]*resolution*2) + 2;

	ftp->ftable = (Number *) mcalloc (flen*sizeof(Number));
	ftp->flen = flen;
	ftp->lenmask = flen;

	fp = ftp->ftable;
	finp = fp + flen;

	do {
		x1 = x[j];
		y1 = y[j];
		x2 = x[j+1];
		y2 = y[j+1];
		z1 = z[j];
		z2 = z[j+1];
	
		if (z2 < z1) goto gen28err2;
		seglen = (int16)((z2-z1) * resolution); 
		incrx = (x2 - x1) / (Number)seglen;
		incry = (y2 - y1) / (Number)seglen;
		while (seglen--) {
			*fp++ = x1;
			x1 += incrx;
			*fp++ = y1;
			y1 += incry;
		}
      
		j++;
	} while (--i);
	do {
		*fp++ = x[j];
		*fp++ = y[j+1];
	} while (fp < finp);

	mfree(x); 
	mfree(y); 
	mfree(z);

	fclose(filp);

	return;

  gen28err1: 
	error << "gen28: couldn't open space file" << endmsg;

  gen28err2: 
	error << "gen28: Time values must be in increasing order" << endmsg;
#else
	return;

#endif // NEED_STL_CONTAINER_NOT_REALLOC

}

void gen29 (FunctionTableRequest *req, FunctionTable *ftp)

{
}

Generator generators[] =

{
      /* FUNC   NAME TYPE       optional: FLAGS ELEMENT SIZE */

	{ gen02, "Arguments", Generator::Arguments },
	{ gen03, "Polynomial", Generator::Polynomial },
	{ gen04, "NormalizedTable", Generator::NormalizedTable },
	{ gen05, "SegmententedExponentials", 
	          Generator::SegmententedExponentials },
	{ gen06, "CubicPolynomial", Generator::CubicPolynomial },
	{ gen07, "SegmentedStraightLines", 
	          Generator::SegmentedStraightLines },
	{ gen08, "CubicSpline", Generator::CubicSpline },
	{ gen09, "WeightedSumsOfSinusoids3", 
	          Generator::WeightedSumsOfSinusoids3 },
	{ gen10, "WeightedSumsOfSinusoids1", 
	          Generator::WeightedSumsOfSinusoids1 },
	{ gen11, "CosinePartials", Generator::CosinePartials },
	{ gen12, "LogOfBessel", Generator::LogOfBessel },
	{ gen13, "FirstKindChebysevCoefficients", 
	          Generator::FirstKindChebysevCoefficients },
	{ gen14, "SecondKindChebysevCoefficients", 
	          Generator::SecondKindChebysevCoefficients },
	{ gen15, "StorePolynomials", Generator::StorePolynomials },
	{ gen17, "StepFunction", Generator::StepFunction },
	{ gen19, "WeightedSumsOfSinusoids4", 
	          Generator::WeightedSumsOfSinusoids4 },
	{ gen20, "Windows", Generator::Windows },
	{ gen21, "RandomDistribution", Generator::RandomDistribution },
	{ gen22, "SoundFileToShort", Generator::SoundFileToShort,
	          Generator::usesFile|Generator::canDefer, sizeof (short) },
	{ gen25, "BreakPointSegmentedExponentials", 
	          Generator::BreakPointSegmentedExponentials },
	{ gen27, "BreakPointSegmentedStraightLines", 
	          Generator::BreakPointSegmentedStraightLines },
	{ gen28, "XYTSpaceFile", Generator::XYTSpaceFile,
	          Generator::usesFile|Generator::canDefer },
	{ gen29, "DataFile", Generator::DataFile, Generator::usesFile },
};
