/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: cross2.cc,v 1.2 1999/11/10 03:44:12 pbd Exp $
*/

#include <math.h>
#include <string.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include "cross2.h"

#define GOOD_TRIG

#if defined(GOOD_TRIG)
#define FHT_SWAP(a,b,t) {(t)=(a);(a)=(b);(b)=(t);}

#define TRIG_VARS	long t_lam=0;

#define TRIG_INIT(k, c, s)				        \
	{							\
		long i;						\
		for (i=2 ; i<=k ; i++)				\
			{coswrk[i]=costab[i];sinwrk[i]=sintab[i];}	\
		t_lam = 0;					\
		c = 1.0f;					\
		s = 0.0f;					\
	}

#define TRIG_NEXT(k,c,s)					\
	{							\
		long i,j;	                                \
		(t_lam)++;					\
		for (i=0 ; !((1L<<i)&t_lam) ; i++);		\
			i = k-i;				\
			s = sinwrk[i];				\
			c = coswrk[i];				\
			if (i>1)	{			\
			  for (j=k-i+2 ; (1L<<j)&t_lam ; j++);	\
			  j = k - j;				\
			  sinwrk[i] = halsec[i] * (sinwrk[i-1] + sinwrk[j]);\
			  coswrk[i] = halsec[i] * (coswrk[i-1] + coswrk[j]);\
			}					\
	}
#endif

#if defined(FAST_TRIG)
#define TRIG_VARS	float t_c,t_s;

#define TRIG_INIT(k,c,s)				\
	{						\
		t_c = costab[k];			\
		t_s = sintab[k];			\
		c = 1;					\
		s = 0;					\
	}

#define TRIG_NEXT(k,c,s)				\
	{						\
		float t = c;				\
		c   = t*t_c - s*t_s;			\
		s   = t*t_s + s*t_c;			\
	}
#endif

#define SQRT2_2		(0.70710678118654752440084436210484f)
#define SQRT2		(2.0f * 0.70710678118654752440084436210484f)

static float halsec[20]=
    {
     0.0f,
     0.0f,
     .54119610014619698439972320536638942006107206337801f,
     .50979557910415916894193980398784391368261849190893f,
     .50241928618815570551167011928012092247859337193963f,
     .50060299823519630134550410676638239611758632599591f,
     .50015063602065098821477101271097658495974913010340f,
     .50003765191554772296778139077905492847503165398345f,
     .50000941253588775676512870469186533538523133757983f,
     .50000235310628608051401267171204408939326297376426f,
     .50000058827484117879868526730916804925780637276181f,
     .50000014706860214875463798283871198206179118093251f,
     .50000003676714377807315864400643020315103490883972f,
     .50000000919178552207366560348853455333939112569380f,
     .50000000229794635411562887767906868558991922348920f,
     .50000000057448658687873302235147272458812263401372f
    };
static float costab[20]=
    {
     .00000000000000000000000000000000000000000000000000f,
     .70710678118654752440084436210484903928483593768847f,
     .92387953251128675612818318939678828682241662586364f,
     .98078528040323044912618223613423903697393373089333f,
     .99518472667219688624483695310947992157547486872985f,
     .99879545620517239271477160475910069444320361470461f,
     .99969881869620422011576564966617219685006108125772f,
     .99992470183914454092164649119638322435060646880221f,
     .99998117528260114265699043772856771617391725094433f,
     .99999529380957617151158012570011989955298763362218f,
     .99999882345170190992902571017152601904826792288976f,
     .99999970586288221916022821773876567711626389934930f,
     .99999992646571785114473148070738785694820115568892f,
     .99999998161642929380834691540290971450507605124278f,
     .99999999540410731289097193313960614895889430318945f,
     .99999999885102682756267330779455410840053741619428f
    };
static float sintab[20]=
    {
     1.0000000000000000000000000000000000000000000000000f,
     .70710678118654752440084436210484903928483593768846f,
     .38268343236508977172845998403039886676134456248561f,
     .19509032201612826784828486847702224092769161775195f,
     .09801714032956060199419556388864184586113667316749f,
     .04906767432741801425495497694268265831474536302574f,
     .02454122852291228803173452945928292506546611923944f,
     .01227153828571992607940826195100321214037231959176f,
     .00613588464915447535964023459037258091705788631738f,
     .00306795676296597627014536549091984251894461021344f,
     .00153398018628476561230369715026407907995486457522f,
     .00076699031874270452693856835794857664314091945205f,
     .00038349518757139558907246168118138126339502603495f,
     .00019174759731070330743990956198900093346887403385f,
     .00009587379909597734587051721097647635118706561284f,
     .00004793689960306688454900399049465887274686668768f
    };
static float coswrk[20]=
    {
     .00000000000000000000000000000000000000000000000000f,
     .70710678118654752440084436210484903928483593768847f,
     .92387953251128675612818318939678828682241662586364f,
     .98078528040323044912618223613423903697393373089333f,
     .99518472667219688624483695310947992157547486872985f,
     .99879545620517239271477160475910069444320361470461f,
     .99969881869620422011576564966617219685006108125772f,
     .99992470183914454092164649119638322435060646880221f,
     .99998117528260114265699043772856771617391725094433f,
     .99999529380957617151158012570011989955298763362218f,
     .99999882345170190992902571017152601904826792288976f,
     .99999970586288221916022821773876567711626389934930f,
     .99999992646571785114473148070738785694820115568892f,
     .99999998161642929380834691540290971450507605124278f,
     .99999999540410731289097193313960614895889430318945f,
     .99999999885102682756267330779455410840053741619428f
    };
static float sinwrk[20]=
    {
     1.0000000000000000000000000000000000000000000000000f,
     .70710678118654752440084436210484903928483593768846f,
     .38268343236508977172845998403039886676134456248561f,
     .19509032201612826784828486847702224092769161775195f,
     .09801714032956060199419556388864184586113667316749f,
     .04906767432741801425495497694268265831474536302574f,
     .02454122852291228803173452945928292506546611923944f,
     .01227153828571992607940826195100321214037231959176f,
     .00613588464915447535964023459037258091705788631738f,
     .00306795676296597627014536549091984251894461021344f,
     .00153398018628476561230369715026407907995486457522f,
     .00076699031874270452693856835794857664314091945205f,
     .00038349518757139558907246168118138126339502603495f,
     .00019174759731070330743990956198900093346887403385f,
     .00009587379909597734587051721097647635118706561284f,
     .00004793689960306688454900399049465887274686668768f
    };


#define CH_THRESH	1.19209e-7
#define CHOP(a)	(a < CH_THRESH ? CH_THRESH : a)

int32
plog2(int32 x)
{
	int32 mask, i;

	if (x == 0)
		return (-1);
	x--;

	for (mask = ~1, i = 0;; mask = mask + mask, i++) {
		if (x == 0)
			return (i);
		x = x & mask;
	}
}


void
getmag(Number *x, int32 size)
{
	Number *i = x + 1, *j = x + size - 1, max = 0.0f;
	int32 n = size / 2 - 1;

	do {
		Number ii = *i;
		Number jj = *j;
		ii = (Number) sqrt((double) (ii * ii + jj * jj));
		if (ii > max)
			max = ii;
		*i = ii;
		i++;
		j--;
	} while (--n);

	n = size / 2 + 1;
	do
		*x++ /= max;
	while (--n);
}


void
mult(Number *x, Number *y, int32 size, Number w)
{
	Number *j = x + size - 1;

	size = size / 2 + 1;
	do {
		Number z = w * *y++;
		*x++ *= z;
		*j-- *= z;
	} while (--size);
}

void
lineaprox(Number *x, int32 size, int32 m)
{
	int32 i, c;
	Number a, f;
	Number rm = 1.0f / (Number) m;

	f = x[0];
	for (i = 0; i < size; i += m) {
		a = 0.0f;
		for (c = 0; c < m; c++) {
			if (a < (Number) fabs((double) (x[i + c])))
				a = x[i + c];
		}
		x[i] = a;
	}
	a = (x[0] + f) * rm;
	for (c = 0; c < m; c++)
		x[c] = a * c + f;
	for (i = m; i < size; i += m) {
		a = (x[i] - x[i - 1]) * rm;
		for (c = 0; c < m; c++)
			x[i + c] = a * c + x[i - 1];
	}
}

void
pfht(Number *fz, int32 n)
{
	int32 i, k, k1, k2, k3, k4, kx;
	Number *fi, *fn, *gi;
	TRIG_VARS;

	k1 = 1;
	k2 = 0;
	do {
		Number a;

		for (k = n >> 1; !((k2 ^= k) & k); k >>= 1);
		if (k1 > k2) {
			a = fz[k1];
			fz[k1] = fz[k2];
			fz[k2] = a;
		}
		k1++;
	} while (k1 < n);

	k = 0;
	i = 1;
	do
		k++;
	while ((i << k) < n);

	k &= 1;
	if (k == 0) {
		fi = fz;
		fn = fz + n;
		do {
			Number f0, f1, f2, f3;
			f1 = fi[0] - fi[1];
			f0 = fi[0] + fi[1];
			f3 = fi[2] - fi[3];
			f2 = fi[2] + fi[3];
			fi[2] = f0 - f2;
			fi[0] = f0 + f2;
			fi[3] = f1 - f3;
			fi[1] = f1 + f3;
			fi += 4;
		} while (fi < fn);
	} else {
		fi = fz;
		fn = fz + n;
		gi = fi + 1;
		do {
			Number s1, c1, s2, c2, s3, c3, s4, c4, g0, f0,
			 f1, g1, f2, g2, f3, g3;

			c1 = fi[0] - gi[0];
			s1 = fi[0] + gi[0];
			c2 = fi[2] - gi[2];
			s2 = fi[2] + gi[2];
			c3 = fi[4] - gi[4];
			s3 = fi[4] + gi[4];
			c4 = fi[6] - gi[6];
			s4 = fi[6] + gi[6];
			f1 = s1 - s2;
			f0 = s1 + s2;
			g1 = c1 - c2;
			g0 = c1 + c2;
			f3 = s3 - s4;
			f2 = s3 + s4;
			g3 = SQRT2 * c4;
			g2 = SQRT2 * c3;
			fi[4] = f0 - f2;
			fi[0] = f0 + f2;
			fi[6] = f1 - f3;
			fi[2] = f1 + f3;
			gi[4] = g0 - g2;
			gi[0] = g0 + g2;
			gi[6] = g1 - g3;
			gi[2] = g1 + g3;

			fi += 8;
			gi += 8;
		} while (fi < fn);
	}
	if (n < 16)
		return;
	do {
		Number s1, c1;

		k += 2;
		k1 = 1L << k;
		k2 = k1 << 1;
		k4 = k2 << 1;
		k3 = k2 + k1;
		kx = k1 >> 1;
		fi = fz;
		gi = fi + kx;
		fn = fz + n;
		do {
			Number g0, f0, f1, g1, f2, g2, f3, g3;

			f1 = fi[0] - fi[k1];
			f0 = fi[0] + fi[k1];
			f3 = fi[k2] - fi[k3];
			f2 = fi[k2] + fi[k3];

			fi[k2] = f0 - f2;
			fi[0] = f0 + f2;
			fi[k3] = f1 - f3;
			fi[k1] = f1 + f3;

			g1 = gi[0] - gi[k1];
			g0 = gi[0] + gi[k1];
			g3 = SQRT2 * gi[k3];
			g2 = SQRT2 * gi[k2];

			gi[k2] = g0 - g2;
			gi[0] = g0 + g2;
			gi[k3] = g1 - g3;
			gi[k1] = g1 + g3;

			gi += k4;
			fi += k4;
		} while (fi < fn);

		TRIG_INIT(k, c1, s1);

		i = 1;
		do {
			Number c2, s2;
			TRIG_NEXT(k, c1, s1);

			c2 = c1 * c1 - s1 * s1;
			s2 = 2 * c1 * s1;
			fn = fz + n;
			fi = fz + i;
			gi = fz + k1 - i;

			do {
				Number a, b, g0, f0, f1, g1, f2, g2,
				 f3, g3;

				b = s2 * fi[k1] - c2 * gi[k1];
				a = c2 * fi[k1] + s2 * gi[k1];
				f1 = fi[0] - a;
				f0 = fi[0] + a;
				g1 = gi[0] - b;
				g0 = gi[0] + b;

				b = s2 * fi[k3] - c2 * gi[k3];
				a = c2 * fi[k3] + s2 * gi[k3];
				f3 = fi[k2] - a;
				f2 = fi[k2] + a;
				g3 = gi[k2] - b;
				g2 = gi[k2] + b;

				b = s1 * f2 - c1 * g3;
				a = c1 * f2 + s1 * g3;
				fi[k2] = f0 - a;
				fi[0] = f0 + a;
				gi[k3] = g1 - b;
				gi[k1] = g1 + b;

				b = c1 * g2 - s1 * f3;
				a = s1 * g2 + c1 * f3;
				gi[k2] = g0 - a;
				gi[0] = g0 + a;
				fi[k3] = f1 - b;
				fi[k1] = f1 + b;

				gi += k4;
				fi += k4;
			} while (fi < fn);

			i++;
		} while (i < kx);
	} while (k4 < n);
}

void
do_fht(Number *real, int32 n)
{
	Number a, b;
	int32 i, j, k;

	pfht(real, n);
	for (i = 1, j = n - 1, k = n / 2; i < k; i++, j--) {
		a = real[i];
		b = real[j];
		real[j] = (a - b) * 0.5f;
		real[i] = (a + b) * 0.5f;
	}
}

void
do_ifht(Number *real, int32 n)
{
	Number a, b;
	int32 i, j, k;

	for (i = 1, j = n - 1, k = n / 2; i < k; i++, j--) {
		a = real[i];
		b = real[j];
		real[j] = (a - b);
		real[i] = (a + b);
	}
	for (i = 0; i < n; i++)
		real[i] /= n;
	pfht(real, n);
}

void
Xsynthset(CON * p)
{
	int32 flen;
	size_t bufsize;
	Number *b;
	RCPointer<FunctionTable> ftp;

	flen = (int32) *p->len;
	p->m = plog2(flen);
	flen = 1L << p->m;

	if (*p->ovlp < 2.0f)
		*p->ovlp = 2.0f;
	else if (*p->ovlp > (Number) (flen + flen))
		*p->ovlp = (Number) (flen + flen);
	*p->ovlp = (Number) (1 << (int) plog2((int32) *p->ovlp));

	bufsize = 10 * flen * sizeof (Number);

	if (p->mem.auxp == NULL || bufsize > p->mem.size) {
		p->mem.alloc (bufsize);
	}

	b = (Number *) p->mem.auxp;
	memset(p->mem.auxp, 0, (size_t) bufsize);	/* Replaces loop */

	p->buffer_in1 = b;
	b += 2 * flen;
	p->buffer_in2 = b;
	b += 2 * flen;
	p->buffer_out = b;
	b += 2 * flen;
	p->in1 = b;
	b += 2 * flen;
	p->in2 = b;
	b += 2 * flen;

	if ((ftp = ftfind(p->iwin)) == NULL) {
		p->error ("cross2: window table not found");
	}

	p->win = ftp;
	p->count = 0;
}

void
Xsynth(CON * p)
{
	Number *s, *f, *out, *buf1, *buf2, *outbuf, rfn;
	int32 n, size, div;
	size_t samps;
	int32 m;

	s = p->as;
	f = p->af;
	out = p->out;

	outbuf = p->buffer_out;
	buf1 = p->buffer_in1;
	buf2 = p->buffer_in2;

	size = (int32) *p->len;
	div = size / (int32) *p->ovlp;
	rfn = (Number) p->win->flen / (Number) size;	/* Moved here for
							   efficiency */

	n = p->count;
	m = n % div;
	for (samps = 0; samps < ksmps; samps++) {
		*(buf1 + n) = *s++;
		*(buf2 + n) = *f++;

		*out++ = *(outbuf + n);
		n++;
		m++;
		if (m == div)
			m = 0;

		if (m == 0) {
			int32 i, mask, index;
			Number window;
			Number *x, *y, *win;

			mask = size - 1;
			win = p->win->ftable;
			x = p->in1;
			y = p->in2;

			for (i = 0; i < size; i++) {

				window = *(win + (int32) (i * rfn));
				index = (i + n) & mask;

				x[i] = *(buf1 + index) * window;
				y[i] = *(buf2 + index) * window;
			}

			for (; i < 2 * size; i++) {
				x[i] = 0.0f;
				y[i] = 0.0f;
			}

			if (*p->bias != 0) {
				int nsize = (int) (size + size);

				do_fht(x, nsize);
				do_fht(y, nsize);
				getmag(y, nsize);

				lineaprox(y, nsize, 16);
				mult(x, y, nsize, *p->bias);

				do_ifht(x, nsize);

			}
			for (i = n + size - div; i < n + size; i++)
				*(outbuf + (i & mask)) = 0.0f;

			window = 5.0f / *p->ovlp;

			for (i = 0; i < size; i++)
				*(outbuf + ((i + n) & mask)) += x[i] * window;

		}
		if (n == size)
			n = 0;	/* Moved to here from inside loop */
	}

	p->count = n;
}

Opcode opcodes[] =
{
	CROSS2_OPCODE_LIST,
	{ NULL }
};
