/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: butter.cc,v 1.1 1999/11/01 04:29:06 pbd Exp $
*/

/*              Butterworth filters coded by Paris Smaragdis 1994       */
/*              Berklee College of Music Csound development team        */
/*              Copyright (c) May 1994.  All rights reserved            */

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include "butter.h"


#define ROOT2 (1.4142135623730950488f)

void butter_filter(float *, float *, float *);

static float pidsr;

void
butset(BFIL * p)
{				/*        Hi/Lo pass set-up       */
	if (*p->istor == 0.0f) {
		p->a[6] = p->a[7] = 0.0f;
		p->lkf = 0.0f;
	}
	pidsr = pi / esr;	/* Constant really */
}

void
bbutset(BBFIL * p)
{				/*        Band set-up     */
	if (*p->istor == 0.0f) {
		p->a[6] = p->a[7] = 0.0f;
		p->lkb = 0.0f;
		p->lkf = 0.0f;
	}
	pidsr = pi / esr;	/* Constant really */
}

void
hibut(BFIL * p)
{				/*        Hipass filter   */
	float *out, *in;

	in = p->ain;
	out = p->sr;

	if (*p->kfc <= 0.0f) {
		long n = ksmps;
		do
			*out++ = *in++;
		while (--n);
		return;
	}
	if (*p->kfc != p->lkf) {
		float *a, c;

		a = p->a;
		c = (float) tan((double) (pidsr * *p->kfc));
		a[1] = 1.0f / (1.0f + ROOT2 * c + c * c);
		a[2] = -(a[1] + a[1]);
		a[3] = a[1];
		a[4] = 2.0f * (c * c - 1.0f) * a[1];
		a[5] = (1.0f - ROOT2 * c + c * c) * a[1];
		p->lkf = *p->kfc;
	}
	butter_filter(in, out, p->a);
}

void
lobut(BFIL * p)
{				/*        Lopass filter   */
	float *out, *in;

	in = p->ain;
	out = p->sr;

	if (*p->kfc <= 0.0f) {
		long n = ksmps;

		do
			*out++ = 0.0f;
		while (--n);

		return;
	}
	if (*p->kfc != p->lkf) {
		float *a, c;

		a = p->a;
		c = 1.0f / (float) tan((double) (pidsr * *p->kfc));
		a[1] = 1.0f / (1.0f + ROOT2 * c + c * c);
		a[2] = a[1] + a[1];
		a[3] = a[1];
		a[4] = 2.0f * (1.0f - c * c) * a[1];
		a[5] = (1.0f - ROOT2 * c + c * c) * a[1];
		p->lkf = *p->kfc;
	}
	butter_filter(in, out, p->a);
}

void
bpbut(BBFIL * p)
{				/*        Bandpass filter */
	float *out, *in;

	in = p->ain;
	out = p->sr;
	if (*p->kbw <= 0.0f) {
		long n = ksmps;
		do
			*out++ = 0.0f;
		while (--n);
		return;
	}
	if (*p->kbw != p->lkb || *p->kfo != p->lkf) {
		float *a, c, d;
		a = p->a;
		c = 1.0f / (float) tan((double) (pidsr * *p->kbw));
		d = 2.0f * (float) cos(2.0 * (double) (pidsr * *p->kfo));
		a[1] = 1.0f / (1.0f + c);
		a[2] = 0.0f;
		a[3] = -a[1];
		a[4] = -c * d * a[1];
		a[5] = (c - 1.0f) * a[1];
		p->lkf = *p->kfo;
		p->lkb = *p->kbw;
	}
	butter_filter(in, out, p->a);
}

void
bcbut(BBFIL * p)
{				/*        Band reject filter      */
	float *out, *in;

	in = p->ain;
	out = p->sr;

	if (*p->kbw <= 0.0f) {
		long n = ksmps;

		do
			*out++ = *in++;
		while (--n);

		return;
	}
	if (*p->kbw != p->lkb || *p->kfo != p->lkf) {
		float *a, c, d;

		a = p->a;
		c = (float) tan((double) (pidsr * *p->kbw));
		d = 2.0f * (float) cos(2.0 * (double) (pidsr * *p->kfo));
		a[1] = 1.0f / (1.0f + c);
		a[2] = -d * a[1];
		a[3] = a[1];
		a[4] = a[2];
		a[5] = (1.0f - c) * a[1];
		p->lkf = *p->kfo;
		p->lkb = *p->kbw;
	}
	butter_filter(in, out, p->a);
}

void
butter_filter(float *in, float *out, float *a)
{				/*        Filter loop     */
	float t, y;
	long n = ksmps;

	do {
		t = *in++ - a[4] * a[6] - a[5] * a[7];
		y = t * a[1] + a[2] * a[6] + a[3] * a[7];
		a[7] = a[6];
		a[6] = t;
		*out++ = y;
	} while (--n);
}

Opcode opcodes[] =
{
	BUTTER_OPCODE_LIST,
	{NULL}
};
