/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: singwave.cc,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/function_tables.h>

#include "physutil.h"
#include "singwave.h"


/*******************************************/
/*  "Singing" Looped Soundfile Class,      */
/*  by Perry R. Cook, 1995-96              */
/*  This Object contains all that's needed */
/*  to make a pitched musical sound, like  */
/*  a simple voice or violin.  In general, */
/*  it will not be used alone (because of  */
/*  of munchinification effects from pitch */
/*  shifting.  It will be used as an       */
/*  excitation source for other instruments */
/*******************************************/


void OneZero_setCoeff(OneZero *, float);
float Wave_tick(Number *, int32 len, Number *, float, float);

void
make_SubNoise(SubNoise * p, int32 subSample)
{
	p->lastOutput = 0.0f;
	p->howOften = subSample - 1;
	p->counter = subSample - 1;
}

float
SubNoise_tick(SubNoise * p)
{
	if (p->counter == 0) {
		p->lastOutput = Noise_tick(&p->lastOutput);
		p->counter = p->howOften;
	} else
		(p->counter)--;
	return p->lastOutput;
}

/*******************************************/
/*  Modulator Class, Perry R. Cook, 1995-96 */
/*  This Object combines random and        */
/*  periodic modulations to give a nice    */
/*  natural human modulation function.     */
/*******************************************/

#define POLE_POS  (0.999f)
#define RND_SCALE (10.0f)

void
make_Modulatr(Modulatr * p, Number *i)
{
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(i)) != 0)
		p->wave = ftp;
	else
	    fatal << "No table for Modulatr" << endmsg;

	/* Expect sine wave */

	p->v_time = 0.0f;
	make_SubNoise(&p->noise, 330);	/* Surely this should be scaled
					   to esr?? */
	make_OnePole(&p->onepole);
	OnePole_setPole(&p->onepole, POLE_POS);
	OnePole_setGain(&p->onepole, RND_SCALE * 0.005f /* p->rndAmt */ );
}

#define Modulatr_setVibFreq(p,vibFreq)	\
                   (p.v_rate = vibFreq * (Number)p.wave->flen/esr)
#define Modulatr_setVibAmt(p,vibAmount)	(p.vibAmt = vibAmount)

float
Modulatr_tick(Modulatr * p)
{
	Number lastOutput;
	lastOutput = Wave_tick(&p->v_time, p->wave->flen, p->wave->ftable,
			       p->v_rate, 0.0f);
	lastOutput *= p->vibAmt;	/*  Compute periodic and */
	lastOutput += OnePole_tick(&p->onepole, SubNoise_tick(&p->noise));
	/*   random modulations  */
	return lastOutput;
}

void
Modulatr_print(Modulatr * p)
{
	info << "Modulatr:"
	     << " v_rate=" << p->v_rate
	     << " v_time=" << p->v_time 
	     << " vibAmt=" << p->vibAmt
	     << endmsg;
}

void
make_SingWave(SingWave * p, Number *ifn, Number *ivfn)
{
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(ifn)) != 0)
		p->wave = ftp;
	else
	    fatal << "No table for Singwave" << endmsg;

	p->mytime = 0.0f;
	p->rate = 1.0f;
	p->sweepRate = 0.001f;
	make_Modulatr(&p->modulator, ivfn);
	Modulatr_setVibFreq(p->modulator, 6.0f);
	Modulatr_setVibAmt(p->modulator, 0.04f);
	make_Envelope(&p->envelope);
	make_Envelope(&p->pitchEnvelope);
	SingWave_setFreq(p, 75.0f);
	Envelope_setRate(&p->pitchEnvelope, 1.0f);
	SingWave_tick(p);
	SingWave_tick(p);
	Envelope_setRate(&p->pitchEnvelope, p->sweepRate * p->rate);
}

void
SingWave_setFreq(SingWave * p, Number aFreq)
{
	Number temp = p->rate;

	p->rate = (Number) p->wave->flen * aFreq / esr;
	temp -= p->rate;
	if (temp < 0)
		temp = -temp;
	Envelope_setTarget(&p->pitchEnvelope, p->rate);
	Envelope_setRate(&p->pitchEnvelope, p->sweepRate * temp);
}

#define SingWave_setVibFreq(p, vibFreq)	\
         Modulatr_setVibFreq(p.modulator, vibFreq)

#define SingWave_setVibAmt(p, vibAmount) \
       (Modulatr_setVibAmt(p.modulator, vibAmount)

float
SingWave_tick(SingWave * p)
{
	Number lastOutput;
	uint32 temp, temp1;
	Number alpha, temp_rate;
	Number mytime = p->mytime;

	temp_rate = Envelope_tick(&p->pitchEnvelope);
	mytime += temp_rate;	/*  Update current time            */
	mytime += temp_rate * Modulatr_tick(&p->modulator);	/*  Add
								   vibratos   */

	while (mytime >= (Number) p->wave->flen) {	/*  Check for end of
							   sound  */
		mytime -= p->wave->flen;	/*  loop back to beginning  */
	}
	while (mytime < 0.0) {	/*  Check for end of sound  */
		mytime += p->wave->flen;	/*  loop back to beginning  */
	}

	temp = (int32) mytime;	/*  Integer part of time address    */
	alpha = mytime - (Number) temp;	/*  fractional part of time
					   address */

	temp1 = temp + 1;
	if (temp1 == p->wave->flen)
		temp1 = temp;	/* Wrap!! */
	lastOutput = alpha * p->wave->ftable[temp1];	/*  Do linear  */
	lastOutput += (1.0f - alpha) * p->wave->ftable[temp];	/*
								   interpolation */
	lastOutput *= Envelope_tick(&p->envelope);

	p->mytime = mytime;
	return lastOutput;
}

void
SingWave_print(SingWave * p)
{
	info << "SingWave:"
	     << " rate=" << p->rate
	     << " sweepRate=" << p->sweepRate
	     << " mytime=" << p->mytime
	     << endmsg;

	Modulatr_print(&p->modulator);
	Envelope_print(&p->envelope);
	Envelope_print(&p->pitchEnvelope);
}

/*******************************************/
/*  4 Formant Synthesis Module         */
/*  by Perry R. Cook, 1995-96              */
/*  This module contains an excitation */
/*  singing wavetable (looping wave with   */
/*  random and periodic vibrato, smoothing */
/*  on frequency, etc.), excitation noise, */
/*  and four sweepable complex resonances. */
/*                                         */
/*  Measured Formant data (from me) is     */
/*  included, and enough data is there to  */
/*  support either parallel or cascade     */
/*  synthesis.  In the floating point case */
/*  cascade synthesis is the most natural  */
/*  so that's what you'll find here.       */
/*                                         */
/*  For right now, there's a simple command */
/*  line score interface consisting of 3   */
/*  letter symbols for the phonemes, =xx   */
/*  sets the pitch to x, + and - add and   */
/*  subtract a half step, and ... makes it */
/*  keep doing what it's doing for longer. */
/*******************************************/

char phonemes[32][4] =
{"eee", "ihh", "ehh", "aaa",
 "ahh", "aww", "ohh", "uhh",
 "uuu", "ooo", "rrr", "lll",
 "mmm", "nnn", "nng", "ngg",
 "fff", "sss", "thh", "shh",
 "xxx", "hee", "hoo", "hah",
 "bbb", "ddd", "jjj", "ggg",
 "vvv", "zzz", "thz", "zhh"};

#define VoicForm_setFormantAll(p,w,f,r,g) \
	FormSwep_setTargets(& p->filters[w],f,r,g)

void
VoicForm_setPhoneme(VOICF * p, int32 i, Number sc)
{
	if (i > 16)
		i = i % 16;
	VoicForm_setFormantAll(p, 0, sc * phonParams[i][0][0],
			       phonParams[i][0][1],
			 (Number) pow(10.0, phonParams[i][0][2] / 20.0f));
	VoicForm_setFormantAll(p, 1, sc * phonParams[i][1][0],
			       phonParams[i][1][1], 1.0f);
	VoicForm_setFormantAll(p, 2, sc * phonParams[i][2][0],
			       phonParams[i][2][1], 1.0f);
	VoicForm_setFormantAll(p, 3, sc * phonParams[i][3][0],
			       phonParams[i][3][1], 1.0f);
	VoicForm_setVoicedUnVoiced(p, phonGains[i][0], phonGains[i][1]);

	info << "Found Formant: " << phonemes[i]
	     << " (number " << i << ')' << endmsg;
}

void
VoicForm_setVoicedUnVoiced(VOICF * p, Number vGain, Number nGain)
{
	Envelope_setTarget(&(p->voiced.envelope), vGain);
	Envelope_setTarget(&p->noiseEnv, nGain);
}

void
VoicForm_quiet(VOICF * p)
{
	Envelope_keyOff(&(p->voiced.envelope));
	Envelope_setTarget(&p->noiseEnv, 0.0f);
}

void
VoicForm_noteOff(VOICF * p)
{
	Envelope_keyOff(&p->voiced.envelope);
}

void
voicprint(VOICF * p)
{
	SingWave_print(&p->voiced);
	OneZero_print(&p->onezero);
	OnePole_print(&p->onepole);
}

static int32 step = 0;

void
voicformset(VOICF * p)
{
	Number amp = (*p->amp) * AMP_RSCALE;	/* Normalise */

	make_SingWave(&p->voiced, p->ifn, p->ivfn);
	Envelope_setRate(&(p->voiced.envelope), 0.001f);
	Envelope_setTarget(&(p->voiced.envelope), 0.0f);

	make_Noise(p->noise);

	make_FormSwep(&p->filters[0]);
	make_FormSwep(&p->filters[1]);
	make_FormSwep(&p->filters[2]);
	make_FormSwep(&p->filters[3]);
	FormSwep_setSweepRate(p->filters[0], 0.001f);
	FormSwep_setSweepRate(p->filters[1], 0.001f);
	FormSwep_setSweepRate(p->filters[2], 0.001f);
	FormSwep_setSweepRate(p->filters[3], 0.001f);

	make_OneZero(&p->onezero);
	OneZero_setCoeff(&p->onezero, -0.9f);
	make_OnePole(&p->onepole);
	OnePole_setPole(&p->onepole, 0.9f);

	make_Envelope(&p->noiseEnv);
	Envelope_setRate(&p->noiseEnv, 0.001f);
	Envelope_setTarget(&p->noiseEnv, 0.0f);

	p->oldform = *p->formant;
	p->ph = (int) (0.5f + *p->phoneme);
	VoicForm_setPhoneme(p, p->ph, p->oldform);
	FormSwep_clear(p->filters[0]);
	FormSwep_clear(p->filters[1]);
	FormSwep_clear(p->filters[2]);
	FormSwep_clear(p->filters[3]);
	Envelope_setTarget(&(p->voiced.envelope), amp);
	OnePole_setPole(&p->onepole, 0.95f - (amp * 0.2f) / 128.0f);
	p->basef = *p->frequency;
	SingWave_setFreq(&p->voiced, p->basef);
	step = 1;
}

void
voicform(VOICF * p)
{
	Number *ar = p->ar;
	int32 nsmps = ksmps;
	Number temp;
	Number lastOutput;

	if (p->basef != *p->frequency) {
		p->basef = *p->frequency;
		SingWave_setFreq(&p->voiced, p->basef);
	}
	SingWave_setVibFreq(p->voiced, *p->vibf);
	Modulatr_setVibAmt(p->voiced.modulator, *p->vibAmt);
	/* Set phoneme */
	if (p->oldform != *p->formant || p->ph != (int) (0.5 + *p->phoneme)) {
		p->oldform = *p->formant;
		p->ph = (int) (0.5 + *p->phoneme);
		info << "Setting Phoneme: " << p->ph << p->oldform << endmsg;
		VoicForm_setPhoneme (p, (int) *p->phoneme, p->oldform);
	}

	do {
		temp = OnePole_tick(&p->onepole,
				    OneZero_tick(&p->onezero,
					     SingWave_tick(&p->voiced)));
		temp += Envelope_tick(&p->noiseEnv) * Noise_tick(&p->noise);
		lastOutput = FormSwep_tick(&p->filters[0], temp);
		lastOutput = FormSwep_tick(&p->filters[1], lastOutput);
		lastOutput = FormSwep_tick(&p->filters[2], lastOutput);
		lastOutput = FormSwep_tick(&p->filters[3], lastOutput);
		lastOutput *= 0.07f * 1.5f;	/* JPff rescaled */
		*ar++ = lastOutput * AMP_SCALE;
	} while (--nsmps);

}

Opcode opcodes[] = {
	SINGWAVE_OPCODE_LIST,
	{ NULL }
};
