/* resonrz.c
 *
 * Copyright 1999, by Sean M. Costello
 *
 * resonr and resonz are implementations of second-order 
 * bandpass resonators, with added zeroes in the transfer function.
 * The algorithms are based upon the work of Julius O. Smith and
 * John Stautner at Stanford, and Ken Steiglitz at Princeton.
 *
 */

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>

#include "resonz.h"
 
void resonzset(RESONZ *p)
{
	/* error message code derived from code for reson in ugens5.c */

	int32 scaletype;

	p->scaletype = scaletype = (int32)*p->iscl;
	if (scaletype && scaletype != 1 && scaletype != 2) {
		p->error ("illegal reson iscl value, %f", *p->iscl);
	}
	if (!(*p->istor))
		p->xnm1 = p->xnm2 = p->ynm1 = p->ynm2 = 0.0f;
}

void resonr(RESONZ *p)
{
	/* 
	 *
	 * An implementation of the 2-pole, 2-zero reson filter
	 * described by Julius O. Smith and James B. Angell in 
	 * "A Constant Gain Digital Resonator Tuned by a Single
	 * Coefficient," Computer Music Journal, Vol. 6, No. 4,
	 * Winter 1982, p.36-39. resonr implements the version
	 * where the zeros are located at + and - the square root
	 * of r, where r is the pole radius of the reson filter.
	 *
	 */

	Number r, scale; /* radius & scaling factor */
	Number c1, c2;   /* filter coefficients */
	Number *out, *in, xn, yn, *xnm1, *xnm2, *ynm1, *ynm2;
	Number kcf = *p->kcf, kbw = *p->kbw;
	int nsmps = ksmps;

	r = (Number)exp(-PI * (double)(kbw / esr));
	c1 = 2.0f * r * (Number)cos(2.0 * PI * (double)(kcf / esr));
	c2 = r * r;

	/* calculation of scaling coefficients */
	if (p->scaletype == 1)
		scale = 1.0f - r;
	else if (p->scaletype == 2)
		scale = (Number)sqrt(1.0 - (double)r);
	else scale = 1.0f;
    
	out = p->out;
	in = p->in; 
	xnm1 = &(p->xnm1);
	xnm2 = &(p->xnm2);
	ynm1 = &(p->ynm1);
	ynm2 = &(p->ynm2);
    
	do {
		xn = *in;
		*out++ = yn = scale * (*in++ - r * *xnm2) + c1 *
			*ynm1 - c2 * *ynm2;
		*xnm2 = *xnm1;
		*xnm1 = xn;
		*ynm2 = *ynm1;
		*ynm1 = yn;
	} while (--nsmps);
}

void resonz(RESONZ *p)
{
	/* 
	 *
	 * An implementation of the 2-pole, 2-zero reson filter
	 * described by Julius O. Smith and James B. Angell in 
	 * "A Constant Gain Digital Resonator Tuned by a Single
	 * Coefficient," Computer Music Journal, Vol. 6, No. 4,
	 * Winter 1982, p.36-39. resonr implements the version
	 * where the zeros are located at z = 1 and z = -1.
	 *
	 */
    
	Number r, scale; /* radius & scaling factor */
	Number c1, c2;   /* filter coefficients */
	Number *out, *in, xn, yn, *xnm1, *xnm2, *ynm1, *ynm2;
	Number kcf = *p->kcf, kbw = *p->kbw;
	int32 nsmps = ksmps;
    
	r = (Number)exp(-PI * (double)(kbw / esr));
	c1 = 2.0f * r * (Number)cos(2.0 * PI * (double)(kcf / esr));
	c2 = r * r;
        
	/* Normalizing factors derived from equations in Ken Steiglitz,
	 * "A Note on Constant-Gain Digital Resonators," Computer
	 * Music Journal, vol. 18, no. 4, pp. 8-10, Winter 1982.
	 */

	if (p->scaletype == 1)
		scale = (1.0f - c2) * 0.5f;
	else if (p->scaletype == 2)
		scale = (Number)sqrt((1.0 - (double)c2) * 0.5);
	else scale = 1.0f;
    
	out = p->out;
	in = p->in; 
	xnm1 = &(p->xnm1);
	xnm2 = &(p->xnm2);
	ynm1 = &(p->ynm1);
	ynm2 = &(p->ynm2);
    
	do {
		xn = *in;
		*out++ = yn = scale * (*in++ - *xnm2) + c1 * 
			*ynm1 - c2 * *ynm2;
		*xnm2 = *xnm1;
		*xnm1 = xn;
		*ynm2 = *ynm1;
		*ynm1 = yn;
	} while (--nsmps);
}

Opcode opcodes[] = {
	RESONZ_OPCODE_LIST,
	{ NULL }
};
