/*
    Copyright (C) 1995-97 Perry Cook & John ffitch

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: physutil.h,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

#ifndef __qm_physutil_h__
#define __qm_physutil_h__

/* Various filters etc for Physical models */

#include <math.h>

#include <quasimodo/types.h>
#include <quasimodo/opcode_defs.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/dynamic_memory.h>

#define AMP_SCALE	(32768.0f)
#define AMP_RSCALE	(.000030517578125f)

/*******************************************/
/*  Noise Generator Class,                 */
/*  by Perry R. Cook, 1995-96              */ 
/*  White noise as often as you like.      */
/*  Recoded by John ffitch 1997            */
/*******************************************/

typedef Number Noise;

#define make_Noise(n) n = 0.0f
#define Noise_lastOut(n) (n)

extern Number Noise_tick (Noise *n);

/*******************************************/
/*  Linearly Interpolating Delay Line      */
/*  Object by Perry R. Cook 1995-96        */
/*  Recoded by John ffitch 1997            */
/*******************************************/

struct DLineL {
    AUXCH	inputs;
    Number	lastOutput;
    int32	inPoint;
    int32	outPoint;
    int32	length;
    Number	alpha;
    Number	omAlpha;
};

#define DLineL_lastOut(d)	((d)->lastOutput)

extern void make_DLineL (DLineL *p, int32 max_length);
extern void DLineL_setDelay (DLineL *p, Number lag);
extern Number DLineL_tick (DLineL *p, Number sample);

/*******************************************/
/*                                         */
/*  AllPass Interpolating Delay Line       */
/*  Object by Perry R. Cook 1995-96        */
/*  This one uses a delay line of maximum  */
/*  length specified on creation, and      */ 
/*  interpolates fractional length using   */
/*  an all-pass filter.  This version is   */
/*  more efficient for computing static    */
/*  length delay lines (alpha and coeff    */
/*  are computed only when the length      */  
/*  is set, there probably is a more       */ 
/*  efficient computational form if alpha  */
/*  is changed often (each sample)).       */
/*                                         */
/*******************************************/

struct DLineA {
    AUXCH	inputs;
    Number	lastOutput;
    int32	inPoint;
    int32	outPoint;
    int32	length;
    Number	alpha;
    Number	coeff;
    Number	lastIn;
};

void make_DLineA(DLineA *, int32 max_length);
void DLineA_clear(DLineA *);
void DLineA_setDelay(DLineA *, Number length);
Number DLineA_tick(DLineA *, Number sample);

/*******************************************/
/*  Envelope Class, Perry R. Cook, 1995-96 */ 
/*  This is the base class for envelopes.  */
/*  This one is capable of ramping state   */
/*  from where it is to a target value by  */
/*  a rate.  It also responds to simple    */
/*  KeyOn and KeyOff messages, ramping to  */         
/*  1.0 on keyon and to 0.0 on keyoff.     */
/*  There are two tick (update value)      */
/*  methods, one returns the value, and    */
/*  other returns 0 if the envelope is at  */
/*  the target value (the state bit).      */
/*******************************************/

#define RATE_NORM	(22050.0f/esr)

struct Envelope {
    Number	value;
    Number	target;
    Number	rate;
    int32       state;
};

extern void make_Envelope (Envelope *e);
extern void Envelope_keyOn (Envelope *e);
extern void Envelope_keyOff (Envelope *e);
extern void Envelope_setRate (Envelope *e, Number aRate);
extern void Envelope_setTarget (Envelope *e, Number aTarget);
extern void Envelope_setValue (Envelope *e, Number aValue);
extern Number Envelope_tick (Envelope *e);
extern void Envelope_print (Envelope *p);

/*******************************************/
/*  One Pole Filter Class,                 */
/*  by Perry R. Cook, 1995-96              */ 
/*  The parameter gain is an additional    */
/*  gain parameter applied to the filter   */  
/*  on top of the normalization that takes */
/*  place automatically.  So the net max   */
/*  gain through the system equals the     */
/*  value of gain.  sgain is the combina-  */
/*  tion of gain and the normalization     */
/*  parameter, so if you set the poleCoeff */
/*  to alpha, sgain is always set to       */
/*  gain * (1.0 - fabs(alpha)).            */
/*******************************************/

struct OnePole {
    Number gain; 
    Number outputs;
    Number poleCoeff;
    Number sgain;
};

extern void make_OnePole (OnePole *p);
extern void OnePole_setPole (OnePole *p, Number aValue);
extern void OnePole_setGain (OnePole *p, Number aValue);
extern Number OnePole_tick (OnePole *p, Number sample);
extern void OnePole_print (OnePole *p);

/*******************************************/
/*  DC Blocking Filter                     */ 
/*  by Perry R. Cook, 1995-96              */ 
/*  This guy is very helpful in, uh,       */
/*  blocking DC.  Needed because a simple  */
/*  low-pass reflection filter allows DC   */
/*  to build up inside recursive           */ 
/*  structures.                            */
/*******************************************/

struct DCBlock {
    Number	gain;
    Number	outputs;
    Number	inputs;
};

extern void make_DCBlock (DCBlock *d);
extern Number DCBlock_tick (DCBlock *d, Number sample);


/*******************************************/
/*  ADSR Subclass of the Envelope Class,   */
/*  by Perry R. Cook, 1995-96              */ 
/*  This is the traditional ADSR (Attack   */
/*  Decay, Sustain, Release) envelope.     */
/*  It responds to simple KeyOn and KeyOff */
/*  messages, keeping track of it's state. */         
/*  There are two tick (update value)      */
/*  methods, one returns the value, and    */
/*  other returns the state (0 = A, 1 = D, */
/*  2 = S, 3 = R)                          */
/*******************************************/



#define ATTACK	(0)
#define DECAY	(1)
#define SUSTAIN (2)
#define RELEASE (3)
#define CLEAR	(4)

struct ADSR {
    Number	value; 
    Number	target;
    Number	rate;
    int32 	state;
    Number	attackRate;
    Number	decayRate;
    Number	sustainLevel;
    Number	releaseRate;
};

void make_ADSR(ADSR*);
void dest_ADSR(ADSR*);
void ADSR_keyOn(ADSR*);
void ADSR_keyOff(ADSR*);
void ADSR_setAttackRate(ADSR*, Number);
void ADSR_setDecayRate(ADSR*, Number);
void ADSR_setSustainLevel(ADSR*, Number);
void ADSR_setReleaseRate(ADSR*, Number);
void ADSR_setAll(ADSR*, float, float, float, Number);
void ADSR_setTarget(ADSR*, Number);
void ADSR_setValue(ADSR*, Number);
Number ADSR_tick(ADSR*);
int32 ADSR_informTick(ADSR*);  
Number ADSR_lastOut(ADSR*);

extern void make_ADSR (ADSR *a);
extern void ADSR_keyOn (ADSR *a);
extern void ADSR_keyOff (ADSR *a);
extern void ADSR_setAttackRate (ADSR *a, Number aRate);
extern void ADSR_setDecayRate (ADSR *a, Number aRate);
extern void ADSR_setSustainLevel (ADSR *a, Number aLevel);
extern void ADSR_setReleaseRate (ADSR *a, Number aRate);
extern void ADSR_setAll (ADSR *a, Number attRate, Number decRate, Number susLevel, Number relRate);
extern void ADSR_setTarget (ADSR *a, Number aTarget);
extern void ADSR_setValue (ADSR *a, Number aValue);
extern Number ADSR_tick (ADSR *a);
extern int32 ADSR_informTick (ADSR *a);


/*******************************************/
/*  BiQuad (2-pole, 2-zero) Filter Class,  */
/*  by Perry R. Cook, 1995-96              */ 
/*  See books on filters to understand     */
/*  more about how this works.  Nothing    */
/*  out of the ordinary in this version.   */
/*******************************************/

struct BiQuad {
    Number	gain;  
    Number	inputs[2];
    Number	lastOutput;
    Number	poleCoeffs[2];
    Number	zeroCoeffs[2];
};

extern void make_BiQuad (BiQuad *b);
extern void BiQuad_clear (BiQuad *b);
extern void BiQuad_setPoleCoeffs (BiQuad *b, Number *coeffs);
extern void BiQuad_setZeroCoeffs (BiQuad *b, Number *coeffs);
extern Number BiQuad_tick (BiQuad *b, Number sample);

#define BiQuad_setGain(b,aValue)	((b).gain = aValue)
#define BiQuad_setEqualGainZeroes(b)	\
	{ (b).zeroCoeffs[1] = -1.0f; (b).zeroCoeffs[0] = 0.0f; }
#define BiQuad_setFreqAndReson(b,freq,reson)	\
	{ (b).poleCoeffs[1]= -((reson)*(reson)); \
          (b).poleCoeffs[0]= 2.0f*(reson)*(Number)cos((double)twopi*(double)(freq)/(double)esr); }
Number BiQuad_tick(BiQuad*, Number);
#define BiQuad_lastOut(x)	(x)->lastOutput


/*******************************************/
/*  One Zero Filter Class,                 */
/*  by Perry R. Cook, 1995-96              */ 
/*  The parameter gain is an additional    */
/*  gain parameter applied to the filter   */  
/*  on top of the normalization that takes */
/*  place automatically.  So the net max   */
/*  gain through the system equals the     */
/*  value of gain.  sgain is the combina-  */
/*  tion of gain and the normalization     */
/*  parameter, so if you set the poleCoeff */
/*  to alpha, sgain is always set to       */
/*  gain / (1.0 - fabs(alpha)).            */
/*******************************************/

struct OneZero {
    Number gain;
    Number inputs;
    Number lastOutput;
    Number zeroCoeff;
    Number sgain;
};

extern void make_OneZero (OneZero *z);
extern Number OneZero_tick (OneZero *z, Number sample);
extern void OneZero_setCoeff (OneZero *z, Number aValue);
extern void OneZero_print (OneZero *p);

/*******************************************/
/*  4 Resonance Modal Synthesis Module */
/*  by Perry R. Cook, 1995-96              */ 
/*  This module contains an excitation */
/*  wavetable, an envelope, and four reso- */
/*  nances (Non-Sweeping BiQuad Filters).  */
/*******************************************/

struct Modal4 {
    Envelope envelope; 
    RCPointer<FunctionTable>	wave;
    int32 	w_myData;
    int32 	w_allDone;
    Number 	w_rate;
    Number 	w_time;
    Number 	w_phaseOffset;
    Number 	w_lastOutput;
    BiQuad      filters[4];
    OnePole     onepole;
    RCPointer<FunctionTable>	vibr;
    Number	v_rate;         /* Parameters for vibrato */
    Number	v_time;
    Number	v_phaseOffset;
    Number	v_lastOutput;
    Number	vibrGain;
    Number	masterGain;
    Number	directGain;
    Number	baseFreq;
    Number	ratios[4];
    Number	resons[4];
};

#define Modal4_setMasterGain(m,Gain)	(m->masterGain = aGain)
#define Modal4_setDirectGain(m,aGain)	(m->directGain = aGain)
#define Modal4_setFiltGain(m,whichOne,gain) (BiQuad_setGain(m->filters[whichOne], gain))

extern void make_Modal4 (Modal4 *m, Number *ifn, Number vgain, Number vrate);
extern void Modal4_setFreq (Modal4 *m, Number frequency);
extern void Modal4_setRatioAndReson (Modal4 *m, int32 whichOne, Number ratio, Number reson);
extern void Modal4_strike (Modal4 *m, Number amplitude);
extern void Modal4_damp (Modal4 *m, Number amplitude);
extern Number Modal4_tick (Modal4 *m);

/*******************************************/
/*  Two Zero Filter Class,                 */
/*  by Perry R. Cook, 1995-96              */ 
/*  See books on filters to understand     */
/*  more about how this works.  Nothing    */
/*  out of the ordinary in this version.   */
/*******************************************/

struct TwoZero {
    Number gain;
    Number inputs[2];
    Number lastOutput;
    Number zeroCoeffs[2];
};

extern void make_TwoZero (TwoZero *p);
extern void TwoZero_setZeroCoeffs (TwoZero *p, Number *coeffs);
extern Number TwoZero_tick (TwoZero *p, Number sample);

extern Number Wave_tick (Number *vTime, int32 len, Number *data, 
			 Number rate, Number phase);


/*******************************************/
/*  Sweepable Formant (2-pole)             */
/*  Filter Class, by Perry R. Cook, 1995-96*/ 
/*  See books on filters to understand     */
/*  more about how this works.  Nothing    */
/*  out of the ordinary in this version.   */
/*******************************************/

struct FormSwep {
    Number	gain;
    Number	outputs[2];
    Number	poleCoeffs[2];
    Number	freq;
    Number	reson;
    int32	dirty;
    Number	targetFreq;
    Number	targetReson;
    Number	targetGain;
    Number	currentFreq;
    Number	currentReson;
    Number	currentGain;
    Number	deltaFreq;
    Number	deltaReson;
    Number	deltaGain;
    Number	sweepState;
    Number	sweepRate;
};

void make_FormSwep(FormSwep *);
#define FormSwep_setSweepRate(p,aRate)	(p.sweepRate = aRate)
#define FormSwep_clear(p)		(p.outputs[0]=p.outputs[1]=0.0f)
void FormSwep_setTargets(FormSwep *, Number, Number, Number);
Number FormSwep_tick(FormSwep *, Number);

extern void make_FormSwep (FormSwep *p);
extern void FormSwep_setStates (FormSwep *p, Number aFreq, Number aReson, Number aGain);
extern void FormSwep_setTargets (FormSwep *p, Number aFreq, Number aReson, Number aGain);
extern Number FormSwep_tick (FormSwep *p, Number sample);

struct Wave {
    RCPointer<FunctionTable>	wave;
    Number	rate;
    Number	time;
    Number	phase;
};

extern Number Samp_tick (Wave *p);

extern Number phonParams[32][4][3];
extern Number phonGains[32][2];

#endif __qm_physutil_h__


