/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: physutil.cc,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

/* This file contains a collection of utilities for the Physical Model
   opcodes, in no particular order
 */

#include <math.h>
#include <stdlib.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/process.h>
#include <quasimodo/function_tables.h>
#include "physutil.h"


/*******************************************/
/*  Noise Generator Class,                 */
/*  by Perry R. Cook, 1995-96              */
/*  White noise as often as you like.      */
/*******************************************/

#define ONE_OVER_RANDLIMIT (1.0f/(Number)(RAND_MAX/2))

Number
Noise_tick(Noise * n)
{
	*n = (Number) rand() - (Number) (RAND_MAX / 2);
	*n *= ONE_OVER_RANDLIMIT;
	return *n;
}

/*******************************************/
/*  Linearly Interpolating Delay Line      */
/*  Object by Perry R. Cook 1995-96        */
/*  This one uses a delay line of maximum  */
/*  length specified on creation, and      */
/*  linearly interpolates fractional       */
/*  length.  It is designed to be more     */
/*  efficient if the delay length is not   */
/*  changed very often.                    */
/*******************************************/

void
make_DLineL(DLineL * p, int32 max_length)
{
	int32 i;

	p->length = max_length;
	p->inputs.alloc (max_length * sizeof (Number));
	for (i = 0; i < max_length; i++)
		((Number *) p->inputs.auxp)[i] = 0.0f;
	p->outPoint = 0;
	p->lastOutput = 0.0f;
	p->inPoint = max_length >> 1;
}

void
DLineL_setDelay(DLineL * p, Number lag)
{
	Number outputPointer;

	outputPointer = p->inPoint - lag;	/* read chases write, + 1 for
						   interp. */
	while (outputPointer < 0.0f)
		outputPointer += (Number) p->length;	/* modulo maximum
							   length */
	while (outputPointer >= (Number) p->length)
		outputPointer -= (Number) p->length;	/* modulo maximum
							   length */
	p->outPoint = (long) outputPointer;	/* integer part */
	p->alpha = outputPointer - (Number) p->outPoint;		/*
								   fractional part */
	p->omAlpha = 1.0f - p->alpha;	/* 1.0 - fractional part */
}

Number
DLineL_tick(DLineL * p, Number sample)
{				/* Take one, yield one */
	Number lastOutput;

	((Number *) p->inputs.auxp)[p->inPoint++] = sample;	/*  Input
								   next sample */
	if (p->inPoint == p->length)	/* Check for end condition */
		p->inPoint -= p->length;
	/* first 1/2 of interpolation */
	lastOutput = ((Number *) p->inputs.auxp)[p->outPoint++] * p->omAlpha;
	if (p->outPoint < p->length) {	/*  Check for end condition */
		/* second 1/2 of interpolation    */
		lastOutput += ((Number *) p->inputs.auxp)[p->outPoint] *
		    p->alpha;
	} else {		/*  if at end . . .  */
		/* second 1/2 of interpolation */
		lastOutput += ((Number *) p->inputs.auxp)[0] * p->alpha;
		p->outPoint -= p->length;
	}

	return (p->lastOutput = lastOutput);
}


/****************************************************************************/
/*                                                                          */
/*  AllPass Interpolating Delay Line Object by Perry R. Cook 1995-96        */
/*  This one uses a delay line of maximum length specified on creation,     */
/*  and interpolates fractional length using an all-pass filter.  This      */
/*  version is more efficient for computing static length delay lines       */
/*  (alpha and coeff are computed only when the length is set, there        */
/*  probably is a more efficient computational form if alpha is changed     */
/*  often (each sample)).                                                   */
/****************************************************************************/

void
make_DLineA(DLineA * p, int32 max_length)
{
	int32 i;
	p->length = max_length;
	p->inputs.alloc (max_length * sizeof (Number));
	for (i = 0; i < max_length; i++)
		((Number *) p->inputs.auxp)[i] = 0.0f;
	p->lastIn = 0.0f;
	p->lastOutput = 0.0f;
	p->inPoint = 0;
	p->outPoint = max_length >> 1;
}

void
DLineA_clear(DLineA * p)
{
	int32 i;
	for (i = 0; i < p->length; i++)
		((Number *) p->inputs.auxp)[i] = 0.0f;
	p->lastIn = 0.0f;
	p->lastOutput = 0.0f;
}

void
DLineA_setDelay(DLineA * p, Number lag)
{
	Number outputPointer;
	outputPointer = p->inPoint - lag + 2;	/* outPoint chases inpoint 
						 */
	/*   + 2 for interp and other     */

	if (p->length <= 0) {
		error << "DlineA not initialised" << endmsg;
		return;
	}

	while (outputPointer < 0)
		outputPointer += p->length;	/* modulo table length       
						 */
	p->outPoint = (long) outputPointer;	/* Integer part of delay     
						 */
	p->alpha = 1.0f + p->outPoint - outputPointer;	/* fractional
							   part of delay */
	if (p->alpha < 0.1) {
		outputPointer += 1.0f;	/*  Hack to avoid pole/zero       
					 */
		p->outPoint += 1;	/*  cancellation.  Keeps allpass  */
		p->alpha += 1.0f;	/*  delay in range of .1 to 1.1   */
	}
	p->coeff = (1.0f - p->alpha) / (1.0f + p->alpha);	/* coefficient 
								   for all pass */
}

Number
DLineA_tick(DLineA * p, Number sample)
{				/*   Take sample, yield sample */
	Number temp;
	((Number *) p->inputs.auxp)[p->inPoint++] = sample;	/* Write
								   input sample  */
	if (p->inPoint == p->length)	/* Increment input pointer */
		p->inPoint -= p->length;	/* modulo length           */
	temp = ((Number *) p->inputs.auxp)[p->outPoint++];	/* filter
								   input         */
	if (p->outPoint == p->length)	/* Increment output pointer */
		p->outPoint -= p->length;	/* modulo length           */
	p->lastOutput = -p->coeff * p->lastOutput;	/* delayed output     
							 */
	p->lastOutput += p->lastIn + (p->coeff * temp);		/* input 
								   + delayed Input */
	p->lastIn = temp;
	return p->lastOutput;	/* save output and return  */
}

/*******************************************/
/*  Envelope Class, Perry R. Cook, 1995-96 */
/*  This is the base class for envelopes.  */
/*  This one is capable of ramping state   */
/*  from where it is to a target value by  */
/*  a rate.                                */
/*******************************************/

void
make_Envelope(Envelope * e)
{
	e->target = 0.0f;
	e->value = 0.0f;
	e->rate = 0.001f;
	e->state = 1;
}

void
Envelope_keyOn(Envelope * e)
{
	e->target = 1.0f;
	if (e->value != e->target)
		e->state = 1;
}

void
Envelope_keyOff(Envelope * e)
{
	e->target = 0.0f;
	if (e->value != e->target)
		e->state = 1;
}

void
Envelope_setRate(Envelope * e, Number aRate)
{
	if (aRate < 0.0f) {
		error << "Envelope_setRate" 
			 "negative rates not allowed!!, correcting" 
		      << endmsg;
		e->rate = -aRate;
	} else {
		e->rate = aRate;
	}
}

void
Envelope_setTarget(Envelope * e, Number aTarget)
{
	e->target = aTarget;
	if (e->value != e->target)
		e->state = 1;
}

void
Envelope_setValue(Envelope * e, Number aValue)
{
	e->state = 0;
	e->target = aValue;
	e->value = aValue;
}

Number
Envelope_tick(Envelope * e)
{
	if (e->state) {
		if (e->target > e->value) {
			e->value += e->rate;
			if (e->value >= e->target) {
				e->value = e->target;
				e->state = 0;
			}
		} else {
			e->value -= e->rate;
			if (e->value <= e->target) {
				e->value = e->target;
				e->state = 0;
			}
		}
	}

	return e->value;
}

void
Envelope_print(Envelope * p)
{
	info << "Envelope: value="
	     << p->value
	     << " target=" << p->target
	     << " rate=" << p->rate
	     << " state=" << p->state 
	     << endmsg;
}


/*******************************************/
/*  One Pole Filter Class,                 */
/*  by Perry R. Cook, 1995-96              */
/*  The parameter gain is an additional    */
/*  gain parameter applied to the filter   */
/*  on top of the normalization that takes */
/*  place automatically.  So the net max   */
/*  gain through the system equals the     */
/*  value of gain.  sgain is the combina-  */
/*  tion of gain and the normalization     */
/*  parameter, so if you set the poleCoeff */
/*  to alpha, sgain is always set to       */
/*  gain * (1.0 - fabs(alpha)).            */
/*******************************************/

void
make_OnePole(OnePole * p)
{
	p->poleCoeff = 0.9f;
	p->gain = 1.0f;
	p->sgain = 0.1f;
	p->outputs = 0.0f;
}

void
OnePole_setPole(OnePole * p, Number aValue)
{
	p->poleCoeff = aValue;
	if (p->poleCoeff > 0.0f)	/*  Normalize gain to 1.0 max */
		p->sgain = p->gain * (1.0f - p->poleCoeff);
	else
		p->sgain = p->gain * (1.0f + p->poleCoeff);
}

void
OnePole_setGain(OnePole * p, Number aValue)
{
	p->gain = aValue;
	if (p->poleCoeff > 0.0f)
		p->sgain = p->gain * (1.0f - p->poleCoeff);	/* 
								   Normalize gain to 1.0 max */
	else
		p->sgain = p->gain * (1.0f + p->poleCoeff);
}

Number
OnePole_tick(OnePole * p, Number sample)
{				/*   Perform Filter Operation */
	p->outputs = (p->sgain * sample) + (p->poleCoeff * p->outputs);
	return p->outputs;
}

void
OnePole_print(OnePole * p)
{
	info << "OnePole:"
	     << " gain=" << p->gain
	     << " outputs=" << p->outputs
	     << " poleCoeff=" << p->poleCoeff
	     << " sgain=" << p->sgain
	     << endmsg;
}


/*******************************************/
/*  DC Blocking Filter                     */
/*  by Perry R. Cook, 1995-96              */
/*  This guy is very helpful in, uh,       */
/*  blocking DC.  Needed because a simple  */
/*  low-pass reflection filter allows DC   */
/*  to build up inside recursive           */
/*  structures.                            */
/*******************************************/

void
make_DCBlock(DCBlock * d)
{
	d->outputs = 0.0f;
	d->inputs = 0.0f;
}

Number
DCBlock_tick(DCBlock * d, Number sample)
{
	d->outputs = sample - d->inputs + (0.99f * d->outputs);
	d->inputs = sample;
	return d->outputs;
}


/*******************************************/
/*  ADSR Subclass of the Envelope Class,   */
/*  by Perry R. Cook, 1995-96              */
/*  This is the traditional ADSR (Attack   */
/*  Decay, Sustain, Release) ADSR.         */
/*  It responds to simple KeyOn and KeyOff */
/*  messages, keeping track of it's state. */
/*  There are two tick (update value)      */
/*  methods, one returns the value, and    */
/*  other returns the state (0 = A, 1 = D, */
/*  2 = S, 3 = R)                          */
/*******************************************/

void
make_ADSR(ADSR * a)
{
	make_Envelope((Envelope *) a);
	a->target = 0.0f;
	a->value = 0.0f;
	a->attackRate = 0.001f;
	a->decayRate = 0.001f;
	a->sustainLevel = 0.5f;
	a->releaseRate = 0.01f;
	a->state = ATTACK;
}

void
ADSR_keyOn(ADSR * a)
{
	a->target = 1.0f;
	a->rate = a->attackRate;
	a->state = ATTACK;
}

void
ADSR_keyOff(ADSR * a)
{
	a->target = 0.0f;
	a->rate = a->releaseRate;
	a->state = RELEASE;
}

void
ADSR_setAttackRate(ADSR * a, Number aRate)
{
	if (aRate < 0.0) {
		info << "negative rates not allowed!!, correcting" << endmsg;
		a->attackRate = -aRate;
	} else
		a->attackRate = aRate;
	a->attackRate *= RATE_NORM;
}

void
ADSR_setDecayRate(ADSR * a, Number aRate)
{
	if (aRate < 0.0) {
		info << "negative rates not allowed!!, correcting" << endmsg;
		a->decayRate = -aRate;
	} else
		a->decayRate = aRate;
	a->decayRate *= RATE_NORM;
}

void
ADSR_setSustainLevel(ADSR * a, Number aLevel)
{
	if (aLevel < 0.0f) {
		info << "Sustain level out of range!!, correcting" << endmsg;
		a->sustainLevel = 0.0f;
	} else
		a->sustainLevel = aLevel;
}

void
ADSR_setReleaseRate(ADSR * a, Number aRate)
{
	if (aRate < 0.0f) {
		info << "negative rates not allowed!!, correcting" << endmsg;
		a->releaseRate = -aRate;
	} else
		a->releaseRate = aRate;
	a->releaseRate *= RATE_NORM;

}

void
ADSR_setAll(ADSR * a, Number attRate, Number decRate,
	    Number susLevel, Number relRate)
{
	ADSR_setAttackRate(a, attRate);
	ADSR_setDecayRate(a, decRate);
	ADSR_setSustainLevel(a, susLevel);
	ADSR_setReleaseRate(a, relRate);
}

void
ADSR_setTarget(ADSR * a, Number aTarget)
{
	a->target = aTarget;
	if (a->value < a->target) {
		a->state = ATTACK;
		ADSR_setSustainLevel(a, a->target);
		a->rate = a->attackRate;
	}
	if (a->value > a->target) {
		ADSR_setSustainLevel(a, a->target);
		a->state = DECAY;
		a->rate = a->decayRate;
	}
}

void
ADSR_setValue(ADSR * a, Number aValue)
{
	a->state = SUSTAIN;
	a->target = aValue;
	a->value = aValue;
	ADSR_setSustainLevel(a, aValue);
	a->rate = 0.0f;
}

Number
ADSR_tick(ADSR * a)
{
	if (a->state == ATTACK) {
		a->value += a->rate;
		if (a->value >= a->target) {
			a->value = a->target;
			a->rate = a->decayRate;
			a->target = a->sustainLevel;
			a->state = DECAY;
		}
	} else if (a->state == DECAY) {
		a->value -= a->decayRate;
		if (a->value <= a->sustainLevel) {
			a->value = a->sustainLevel;
			a->rate = 0.0f;
			a->state = SUSTAIN;
		}
	} else if (a->state == RELEASE) {
		a->value -= a->releaseRate;
		if (a->value <= 0.0f) {
			a->value = 0.0f;
			a->state = CLEAR;
		}
	}
	return a->value;
}

int32
ADSR_informTick(ADSR * a)
{
	ADSR_tick(a);
	return a->state;
}

/*******************************************/
/*  BiQuad (2-pole, 2-zero) Filter Class,  */
/*  by Perry R. Cook, 1995-96              */
/*  See books on filters to understand     */
/*  more about how this works.  Nothing    */
/*  out of the ordinary in this version.   */
/*******************************************/

void
make_BiQuad(BiQuad * b)
{
	b->zeroCoeffs[0] = 0.0f;
	b->zeroCoeffs[1] = 0.0f;
	b->poleCoeffs[0] = 0.0f;
	b->poleCoeffs[1] = 0.0f;
	b->gain = 1.0f;
	b->inputs[0] = 0.0f;
	b->inputs[1] = 0.0f;
	b->lastOutput = 0.0f;
}

void
BiQuad_clear(BiQuad * b)
{
	b->inputs[0] = 0.0f;
	b->inputs[1] = 0.0f;
	b->lastOutput = 0.0f;
}

void
BiQuad_setPoleCoeffs(BiQuad * b, Number *coeffs)
{
	b->poleCoeffs[0] = coeffs[0];
	b->poleCoeffs[1] = coeffs[1];
}

void
BiQuad_setZeroCoeffs(BiQuad * b, Number *coeffs)
{
	b->zeroCoeffs[0] = coeffs[0];
	b->zeroCoeffs[1] = coeffs[1];
}

Number
BiQuad_tick(BiQuad * b, Number sample)
{				/*   Perform Filter Operation   *//*  Biquad
				   is two pole, two zero filter  */
	Number temp;		/*  Look it up in your favorite DSP text */

	temp = sample * b->gain;	/* Here's the math for the  */
	temp += b->inputs[0] * b->poleCoeffs[0];	/* version which
							   implements */
	temp += b->inputs[1] * b->poleCoeffs[1];	/* only 2 state
							   variables.  */

	b->lastOutput = temp;	/* This form takes   */
	b->lastOutput += (b->inputs[0] * b->zeroCoeffs[0]);	/* 5
								   multiplies and  */
	b->lastOutput += (b->inputs[1] * b->zeroCoeffs[1]);	/* 4 adds    
								 */
	b->inputs[1] = b->inputs[0];	/* and 3 moves       */
	b->inputs[0] = temp;	/* like the 2 state-var form */

	return b->lastOutput;

}


/*******************************************/
/*  One Zero Filter Class,                 */
/*  by Perry R. Cook, 1995-96              */
/*  The parameter gain is an additional    */
/*  gain parameter applied to the filter   */
/*  on top of the normalization that takes */
/*  place automatically.  So the net max   */
/*  gain through the system equals the     */
/*  value of gain.  sgain is the combina-  */
/*  tion of gain and the normalization     */
/*  parameter, so if you set the poleCoeff */
/*  to alpha, sgain is always set to       */
/*  gain / (1.0 - fabs(alpha)).            */
/*******************************************/

void
make_OneZero(OneZero * z)
{
	z->gain = 1.0f;
	z->zeroCoeff = 1.0f;
	z->sgain = 0.5f;
	z->inputs = 0.0f;
	z->lastOutput = 0.0f;
}

Number
OneZero_tick(OneZero * z, Number sample)
{				/*   Perform Filter Operation  */
	Number temp;
	temp = z->sgain * sample;
	z->lastOutput = (z->inputs * z->zeroCoeff) + temp;
	z->inputs = temp;
	return z->lastOutput;
}

void
OneZero_setCoeff(OneZero * z, Number aValue)
{
	z->zeroCoeff = aValue;
	if (z->zeroCoeff > 0.0f)	/*  Normalize gain to 1.0 max  */
		z->sgain = z->gain / (1.0f + z->zeroCoeff);
	else
		z->sgain = z->gain / (1.0f - z->zeroCoeff);
}

void
OneZero_print(OneZero * p)
{
	info << "OneZero:"
	     << " gain=" << p->gain
	     << " inputs=" << p->inputs
	     << " zeroCoeff=" << p->zeroCoeff
	     << " sgain=" << p->sgain
	     << " last=" << p->lastOutput
	     << endmsg;
}

/*******************************************/
/*  4 Resonance Modal Synthesis Module */
/*  by Perry R. Cook, 1995-96              */
/*  This module contains an excitation */
/*  wavetable, an envelope, and four reso- */
/*  nances (Non-Sweeping BiQuad Filters).  */
/*******************************************/

void
make_Modal4(Modal4 * m, Number *ifn, Number vgain, Number vrate)
{
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(ifn)) != 0) {
		m->vibr = ftp;
	} else {
		error << "No table for Modal4 case" << endmsg;
		return;
	}

	make_Envelope(&m->envelope);
	/*  We don't make the excitation wave here yet,   */
	/*  because we don't know what it's going to be.  */
	make_BiQuad(&m->filters[0]);
	make_BiQuad(&m->filters[1]);
	make_BiQuad(&m->filters[2]);
	make_BiQuad(&m->filters[3]);
	make_OnePole(&m->onepole);

	m->v_rate = vrate;	/* 6.0; */
	m->vibrGain = vgain;	/* 0.05; */

	m->masterGain = 1.0f;
	BiQuad_clear(&m->filters[0]);
	BiQuad_clear(&m->filters[1]);
	BiQuad_clear(&m->filters[2]);
	BiQuad_clear(&m->filters[3]);
	BiQuad_setEqualGainZeroes(m->filters[0]);
	BiQuad_setEqualGainZeroes(m->filters[1]);
	BiQuad_setEqualGainZeroes(m->filters[2]);
	BiQuad_setEqualGainZeroes(m->filters[3]);
}

void
Modal4_setFreq(Modal4 * m, Number frequency)
{
	m->baseFreq = frequency;
	Modal4_setRatioAndReson(m, 0, m->ratios[0], m->resons[0]);
	Modal4_setRatioAndReson(m, 1, m->ratios[1], m->resons[1]);
	Modal4_setRatioAndReson(m, 2, m->ratios[2], m->resons[2]);
	Modal4_setRatioAndReson(m, 3, m->ratios[3], m->resons[3]);
}

void
Modal4_setRatioAndReson(Modal4 * m, int32 whichOne, Number ratio, Number reson)
{
	Number temp;
	if (ratio * m->baseFreq < esr * 0.5f) {
		m->ratios[whichOne] = ratio;
	} else {
		temp = ratio;
		while (temp * m->baseFreq > esr / 2.0f)
			temp *= 0.5f;
		m->ratios[whichOne] = temp;
	}
	m->resons[whichOne] = reson;
	if (ratio < 0)
		temp = -ratio;
	else
		temp = ratio * m->baseFreq;
	BiQuad_setFreqAndReson(m->filters[whichOne], temp, reson);
}

void
Modal4_strike(Modal4 * m, Number amplitude)
{
	int32 i;
	Number temp;
	Envelope_setRate(&m->envelope, 1.0f);
	Envelope_setTarget(&m->envelope, amplitude);
	OnePole_setPole(&m->onepole, 1.0f - amplitude);
	Envelope_tick(&m->envelope);
	m->w_time = 0.0f;
	m->w_lastOutput = 0.0f;
	m->w_allDone = 0;
	/*     wave->reset(); */
	for (i = 0; i < 4; i++) {
		if (m->ratios[i] < 0)
			temp = -m->ratios[i];
		else
			temp = m->ratios[i] * m->baseFreq;
		BiQuad_setFreqAndReson(m->filters[i], temp, m->resons[i]);
	}
}


void
Modal4_damp(Modal4 * m, Number amplitude)
{
	int32 i;
	Number temp;
	for (i = 0; i < 4; i++) {
		if (m->ratios[i] < 0)
			temp = -m->ratios[i];
		else
			temp = m->ratios[i] * m->baseFreq;
		BiQuad_setFreqAndReson(m->filters[i], temp, m->resons[i] *
				       amplitude);
	}
}

Number
Modal4_tick(Modal4 * m)
{
	Number temp, temp2;
	int32 itemp;
	Number temp_time, alpha;
	int32 length = (int32) m->wave->flen;

	m->w_time += m->w_rate;	/*  Update current time  */

	if (m->w_time >= length) {	/*  Check for end of sound       */
		m->w_time = (Number) (length - 1);	/*  stick at end       
							 */
		m->w_allDone = 1;	/*  Information for one-shot use */
	} else if (m->w_time < 0.0f)	/*  Check for end of sound       */
		m->w_time = 0.0f;	/*  stick at beg                 */

	temp_time = m->w_time;

#ifdef phase_offset
	if (m->w_phaseOffset != 0.0f) {
		temp_time += m->w_phaseOffset;	/*  Add phase offset      
						 */
		if (temp_time >= length)	/*  Check for end of sound      
						 */
			temp_time = length - 1;		/*  stick at end 
							 */
		else if (temp_time < 0.0f)	/*  check for end of sound    
						 */
			temp_time = 0.0f;	/*  stick at beg               
						 */
	}
#endif

	itemp = (long) temp_time;	/*  Integer part of time address    */

	alpha = temp_time - (Number) itemp;	/*  fractional part of time
						   address */
	m->w_lastOutput = m->wave->ftable[itemp];	/*  Do linear
							   interpolation         */

	m->w_lastOutput = m->w_lastOutput +	/*  same as
						   alpha*data[temp+1]      */
	    (alpha * (m->wave->ftable[itemp + 1] -
		      m->w_lastOutput));	/*  + (1-alpha)data[temp]       
						 */


	temp = m->masterGain *
	    OnePole_tick(&m->onepole, m->w_lastOutput *
			 Envelope_tick(&m->envelope));

	temp2 = BiQuad_tick(&m->filters[0], temp);

	temp2 += BiQuad_tick(&m->filters[1], temp);

	temp2 += BiQuad_tick(&m->filters[2], temp);

	temp2 += BiQuad_tick(&m->filters[3], temp);

	temp2 = temp2 - (temp2 * m->directGain);
	temp2 += m->directGain * temp;


	if (m->vibrGain != 0.0) {

		/* Tick on vibrato table */
		m->v_time += m->v_rate;		/*  Update current time  
						 */
		while (m->v_time >= m->vibr->flen)	/*  Check for end of
							   sound */
			m->v_time -= m->vibr->flen;	/*  loop back to
							   beginning */
		while (m->v_time < 0.0f)	/*  Check for end of sound */
			m->v_time += m->vibr->flen;	/*  loop back to
							   beginning */

		temp_time = m->v_time;

#ifdef phase_offset
		if (m->v_phaseOffset != 0.0f) {
			temp_time += m->v_phaseOffset;	/*  Add phase
							   offset       */
			while (temp_time >= m->vibr->flen)	/*  Check for 
								   end of sound */
				temp_time -= m->vibr->flen;	/*  loop
								   back to beginning */
			while (temp_time < 0.0f)	/*  Check for end of
							   sound */
				temp_time += m->vibr->flen;	/*  loop
								   back to beginning */
		}
#endif

		itemp = (long) temp_time;	/*  Integer part of time
						   address    */
		/*  fractional part of time address */
		alpha = temp_time - (Number) itemp;
		m->v_lastOutput = m->vibr->ftable[itemp];	/* Do linear
								   interpolation */
		/*  same as alpha*data[itemp+1] + (1-alpha)data[temp] */
		m->v_lastOutput = m->v_lastOutput +
		    (alpha * (m->vibr->ftable[itemp + 1] - m->v_lastOutput));
		/* End of vibrato tick */
		temp = 1.0f + (m->v_lastOutput * m->vibrGain);	/* 
								   Calculate AM           */
		temp2 = temp * temp2;	/*  and apply to master out */
	}
	return (temp2 + temp2);
}

/***********************************************************/
/*  Two Zero Filter Class,                                 */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98   */
/*  See books on filters to understand more about how this */
/*  works.  Nothing out of the ordinary in this version.   */
/***********************************************************/

void
make_TwoZero(TwoZero * p)
{
	p->zeroCoeffs[0] = 0.0f;
	p->zeroCoeffs[1] = 0.0f;
	p->gain = 1.0f;
	p->inputs[0] = 0.0f;
	p->inputs[1] = 0.0f;
	p->lastOutput = 0.0f;
}

void
TwoZero_setZeroCoeffs(TwoZero * p, float *coeffs)
{
	p->zeroCoeffs[0] = coeffs[0];
	p->zeroCoeffs[1] = coeffs[1];
}

float
TwoZero_tick(TwoZero * p, float sample)
  /*   Perform Filter Operation            */
{				/*  TwoZero is a two zero filter (duh!)  */
	/*  Look it up in your favorite DSP text */
	p->lastOutput = p->zeroCoeffs[0] * p->inputs[0] +
	    p->zeroCoeffs[1] * p->inputs[1];
	p->inputs[1] = p->inputs[0];
	p->inputs[0] = p->gain * sample;
	p->lastOutput += p->inputs[0];
	return p->lastOutput;
}

Number
Wave_tick(Number *vTime, int32 len, Number *data, Number rate, Number phase)
{	
	int32 temp;
	Number temp_time, alpha;
	Number lastOutput;

	*vTime += rate;		/*  Update current time    */
	while (*vTime >= len)	/*  Check for end of sound */
		*vTime -= len;	/*  loop back to beginning */
	while (*vTime < 0.0f)	/*  Check for end of sound */
		*vTime += len;	/*  loop back to beginning */

	temp_time = *vTime;

	if (phase != 0.0f) {
		temp_time += phase;	/*  Add phase offset       */
		while (temp_time >= len)	/*  Check for end of sound */
			temp_time -= len;	/*  loop back to beginning */
		while (temp_time < 0.0f)	/*  Check for end of sound */
			temp_time += len;	/*  loop back to beginning */
	}
	temp = (int32) temp_time;	/*  Integer part of time address    */
	/*  fractional part of time address */
	alpha = temp_time - (Number) temp;
	lastOutput = data[temp];	/* Do linear interpolation */
	/*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
	lastOutput += (alpha * (data[temp + 1] - lastOutput));
	/* End of vibrato tick */
	return lastOutput;
}

Number
Samp_tick(Wave * p)
{
	uint32 temp, temp1;
	Number temp_time, alpha;
	Number lastOutput;

	p->time += p->rate;	/*  Update current time    */
	while (p->time >= p->wave->flen)	/*  Check for end of sound */
		p->time -= p->wave->flen;	/*  loop back to beginning */
	while (p->time < 0.0)	/*  Check for end of sound */
		p->time += p->wave->flen;	/*  loop back to beginning */

	temp_time = p->time;

	if (p->phase != 0.0) {
		temp_time += p->phase;	/*  Add phase offset       */
		while (temp_time >= p->wave->flen)	/*  Check for end of
							   sound */
			temp_time -= p->wave->flen;	/*  loop back to
							   beginning */
		while (temp_time < 0.0)		/*  Check for end of
						   sound */
			temp_time += p->wave->flen;	/*  loop back to
							   beginning */
	}
	temp = (long) temp_time;	/*  Integer part of time address    */
	temp1 = temp + 1;
	if (temp1 == p->wave->flen)
		temp1 = 0;	/* Wrap!! */

	/*  fractional part of time address */
	alpha = temp_time - (Number) temp;
	lastOutput = p->wave->ftable[temp];	/* Do linear interpolation
						 */

	/* same as alpha*data[temp+1] + (1-alpha)data[temp] */
	lastOutput += (alpha * (p->wave->ftable[temp + 1] - lastOutput));
	/* End of vibrato tick */
	return lastOutput;
}


/********************************************/
/*  Sweepable Formant (2-pole)              */
/*  Filter Class, by Perry R. Cook, 1995-96 */
/*  See books on filters to understand      */
/*  more about how this works.  This drives */
/*  to a target at speed set by rate.       */
/********************************************/

void
make_FormSwep(FormSwep * p)
{
	p->poleCoeffs[0] = p->poleCoeffs[1] = 0.0f;
	p->gain = 1.0f;
	p->freq = p->reson = 0.0f;
	p->currentGain = 1.0f;
	p->currentFreq = p->currentReson = 0.0f;
	p->targetGain = 1.0f;
	p->targetFreq = p->targetReson = 0.0f;
	p->deltaGain = 0.0f;
	p->deltaFreq = p->deltaReson = 0.0f;
	p->sweepState = 0.0f;
	p->sweepRate = 0.002f;
	p->dirty = 0;
	p->outputs[0] = p->outputs[1] = 0.0f;
}

void
FormSwep_setStates(FormSwep * p, Number aFreq, Number aReson, Number aGain)
{
	p->dirty = 0;
	p->freq = p->targetFreq = p->currentFreq = aFreq;
	p->reson = p->targetReson = p->currentReson = aReson;
	p->gain = p->targetGain = p->currentGain = aGain;
}

void
FormSwep_setTargets(FormSwep * p, Number aFreq, Number aReson, Number aGain)
{
	p->dirty = 1;
	p->targetFreq = aFreq;
	p->targetReson = aReson;
	p->targetGain = aGain;
	p->deltaFreq = aFreq - p->currentFreq;
	p->deltaReson = aReson - p->currentReson;
	p->deltaGain = aGain - p->currentGain;
	p->sweepState = 0.0f;
}

Number
FormSwep_tick(FormSwep * p, Number sample)
{				/* Perform Filter Operation */
	Number temp;

	if (p->dirty) {
		p->sweepState += p->sweepRate;
		if (p->sweepState >= 1.0f) {
			p->sweepState = 1.0f;
			p->dirty = 0;
			p->currentReson = p->targetReson;
			p->reson = p->targetReson;
			p->currentFreq = p->targetFreq;
			p->freq = p->targetFreq;
			p->currentGain = p->targetGain;
			p->gain = p->targetGain;
		} else {
			p->currentReson = p->reson + (p->deltaReson *
						      p->sweepState);
			p->currentFreq = p->freq + (p->deltaFreq *
						    p->sweepState);
			p->currentGain = p->gain + (p->deltaGain *
						    p->sweepState);
		}
		p->poleCoeffs[1] = -(p->currentReson * p->currentReson);
		p->poleCoeffs[0] = 2.0f * p->currentReson *
		    (Number) cos((double) (twopi * p->currentFreq / esr));
	}
	temp = p->currentGain * sample;
	temp += p->poleCoeffs[0] * p->outputs[0];
	temp += p->poleCoeffs[1] * p->outputs[1];
	p->outputs[1] = p->outputs[0];
	p->outputs[0] = temp;
	return temp;
}

Number phonGains[32][2] =
{
	{1.0f, 0.0f},		/* eee */
	{1.0f, 0.0f},		/* ihh */
	{1.0f, 0.0f},		/* ehh */
	{1.0f, 0.0f},		/* aaa */

	{1.0f, 0.0f},		/* ahh */
	{1.0f, 0.0f},		/* aww */
	{1.0f, 0.0f},		/* ohh */
	{1.0f, 0.0f},		/* uhh */

	{1.0f, 0.0f},		/* uuu */
	{1.0f, 0.0f},		/* ooo */
	{1.0f, 0.0f},		/* rrr */
	{1.0f, 0.0f},		/* lll */

	{1.0f, 0.0f},		/* mmm */
	{1.0f, 0.0f},		/* nnn */
	{1.0f, 0.0f},		/* nng */
	{1.0f, 0.0f},		/* ngg */

	{0.0f, 1.0f},		/* fff */
	{0.0f, 1.0f},		/* sss */
	{0.0f, 1.0f},		/* thh */
	{0.0f, 1.0f},		/* shh */

	{0.0f, 1.0f},		/* xxx */
	{0.0f, 0.1f},		/* hee */
	{0.0f, 0.1f},		/* hoo */
	{0.0f, 0.1f},		/* hah */

	{1.0f, 0.1f},		/* bbb */
	{1.0f, 0.1f},		/* ddd */
	{1.0f, 0.1f},		/* jjj */
	{1.0f, 0.1f},		/* ggg */

	{1.0f, 1.0f},		/* vvv */
	{1.0f, 1.0f},		/* zzz */
	{1.0f, 1.0f},		/* thz */
	{1.0f, 1.0f}		/* zhh */
};

Number phonParams[32][4][3] =
{
	{
		{273.0f, 0.996f, 0.0f},		/* eee (beet) */
		{2086.0f, 0.945f, -16.0f},
		{2754.0f, 0.979f, -12.0f},
		{3270.0f, 0.440f, -17.0f}},
	{
		{385.0f, 0.987f, 10.0f},	/* ihh (bit) */
		{2056.0f, 0.930f, -20.0f},
		{2587.0f, 0.890f, -20.0f},
		{3150.0f, 0.400f, -20.0f}},
	{
		{515.0f, 0.977f, 10.0f},	/* ehh (bet) */
		{1805.0f, 0.810f, -10.0f},
		{2526.0f, 0.875f, -10.0f},
		{3103.0f, 0.400f, -13.0f}},
	{
		{773.0f, 0.950f, 10.0f},	/* aaa (bat) */
		{1676.0f, 0.830f, -6.0f},
		{2380.0f, 0.880f, -20.0f},
		{3027.0f, 0.600f, -20.0f}},

	{
		{770.0f, 0.950f, 0.0f},		/* ahh (father) */
		{1153.0f, 0.970f, -9.0f},
		{2450.0f, 0.780f, -29.0f},
		{3140.0f, 0.800f, -39.0f}},
	{
		{637.0f, 0.910f, 0.0f},		/* aww (bought) */
		{895.0f, 0.900f, -3.0f},
		{2556.0f, 0.950f, -17.0f},
		{3070.0f, 0.910f, -20.0f}},
	{
		{637.0f, 0.910f, 0.0f},		/* ohh (bone)  NOTE:: 
						   same as aww (bought) */
		{895.0f, 0.900f, -3.0f},
		{2556.0f, 0.950f, -17.0f},
		{3070.0f, 0.910f, -20.0f}},
	{
		{561.0f, 0.965f, 0.0f},		/* uhh (but) */
		{1084.0f, 0.930f, -10.0f},
		{2541.0f, 0.930f, -15.0f},
		{3345.0f, 0.900f, -20.0f}},

	{
		{515.0f, 0.976f, 0.0f},		/* uuu (foot) */
		{1031.0f, 0.950f, -3.0f},
		{2572.0f, 0.960f, -11.0f},
		{3345.0f, 0.960f, -20.0f}},
	{
		{349.0f, 0.986f, -10.0f},	/* ooo (boot) */
		{918.0f, 0.940f, -20.0f},
		{2350.0f, 0.960f, -27.0f},
		{2731.0f, 0.950f, -33.0f}},
	{
		{394.0f, 0.959f, -10.0f},	/* rrr (bird) */
		{1297.0f, 0.780f, -16.0f},
		{1441.0f, 0.980f, -16.0f},
		{2754.0f, 0.950f, -40.0f}},
	{
		{462.0f, 0.990f, +5.0f},	/* lll (lull) */
		{1200.0f, 0.640f, -10.0f},
		{2500.0f, 0.200f, -20.0f},
		{3000.0f, 0.100f, -30.0f}},

	{
		{265.0f, 0.987f, -10.0f},	/* mmm (mom) */
		{1176.0f, 0.940f, -22.0f},
		{2352.0f, 0.970f, -20.0f},
		{3277.0f, 0.940f, -31.0f}},
	{
		{204.0f, 0.980f, -10.0f},	/* nnn (nun) */
		{1570.0f, 0.940f, -15.0f},
		{2481.0f, 0.980f, -12.0f},
		{3133.0f, 0.800f, -30.0f}},
	{
		{204.0f, 0.980f, -10.0f},	/* nng (sang)    NOTE:: same
						   as nnn */
		{1570.0f, 0.940f, -15.0f},
		{2481.0f, 0.980f, -12.0f},
		{3133.0f, 0.800f, -30.0f}},
	{
		{204.0f, 0.980f, -10.0f},	/* ngg (bong)    NOTE:: same
						   as nnn */
		{1570.0f, 0.940f, -15.0f},
		{2481.0f, 0.980f, -12.0f},
		{3133.0f, 0.800f, -30.0f}},

	{
		{1000.0f, 0.300f, 0.0f},	/* fff */
		{2800.0f, 0.860f, -10.0f},
		{7425.0f, 0.740f, 0.0f},
		{8140.0f, 0.860f, 0.0f}},
	{
		{0.0f, 0.000f, 0.0f},	/* sss */
		{2000.0f, 0.700f, -15.0f},
		{5257.0f, 0.750f, -3.0f},
		{7171.0f, 0.840f, 0.0f}},
	{
		{100.0f, 0.900f, 0.0f},		/* thh */
		{4000.0f, 0.500f, -20.0f},
		{5500.0f, 0.500f, -15.0f},
		{8000.0f, 0.400f, -20.0f}},
	{
		{2693.0f, 0.940f, 0.0f},	/* shh */
		{4000.0f, 0.720f, -10.0f},
		{6123.0f, 0.870f, -10.0f},
		{7755.0f, 0.750f, -18.0f}},

	{
		{1000.0f, 0.300f, -10.0f},	/* xxx           NOTE:: Not
						   Really Done Yet */
		{2800.0f, 0.860f, -10.0f},
		{7425.0f, 0.740f, 0.0f},
		{8140.0f, 0.860f, 0.0f}},
	{
		{273.0f, 0.996f, -40.0f},	/* hee (beet)    (noisy eee)
						 */
		{2086.0f, 0.945f, -16.0f},
		{2754.0f, 0.979f, -12.0f},
		{3270.0f, 0.440f, -17.0f}},
	{
		{349.0f, 0.986f, -40.0f},	/* hoo (boot)    (noisy ooo)
						 */
		{918.0f, 0.940f, -10.0f},
		{2350.0f, 0.960f, -17.0f},
		{2731.0f, 0.950f, -23.0f}},
	{
		{770.0f, 0.950f, -40.0f},	/* hah (father)  (noisy ahh)
						 */
		{1153.0f, 0.970f, -3.0f},
		{2450.0f, 0.780f, -20.0f},
		{3140.0f, 0.800f, -32.0f}},

	{
		{2000.0f, 0.700f, -20.0f},	/* bbb           NOTE:: Not
						   Really Done Yet */
		{5257.0f, 0.750f, -15.0f},
		{7171.0f, 0.840f, -3.0f},
		{9000.0f, 0.900f, 0.0f}},
	{
		{100.0f, 0.900f, 0.0f},		/* ddd           NOTE::
						   Not Really Done Yet */
		{4000.0f, 0.500f, -20.0f},
		{5500.0f, 0.500f, -15.0f},
		{8000.0f, 0.400f, -20.0f}},
	{
		{2693.0f, 0.940f, 0.0f},	/* jjj           NOTE:: Not
						   Really Done Yet */
		{4000.0f, 0.720f, -10.0f},
		{6123.0f, 0.870f, -10.0f},
		{7755.0f, 0.750f, -18.0f}},
	{
		{2693.0f, 0.940f, 0.0f},	/* ggg           NOTE:: Not
						   Really Done Yet */
		{4000.0f, 0.720f, -10.0f},
		{6123.0f, 0.870f, -10.0f},
		{7755.0f, 0.750f, -18.0f}},

	{
		{2000.0f, 0.700f, -20.0f},	/* vvv           NOTE:: Not
						   Really Done Yet */
		{5257.0f, 0.750f, -15.0f},
		{7171.0f, 0.840f, -3.0f},
		{9000.0f, 0.900f, 0.0f}},
	{
		{100.0f, 0.900f, 0.0f},		/* zzz           NOTE::
						   Not Really Done Yet */
		{4000.0f, 0.500f, -20.0f},
		{5500.0f, 0.500f, -15.0f},
		{8000.0f, 0.400f, -20.0f}},
	{
		{2693.0f, 0.940f, 0.0f},	/* thz           NOTE:: Not
						   Really Done Yet */
		{4000.0f, 0.720f, -10.0f},
		{6123.0f, 0.870f, -10.0f},
		{7755.0f, 0.750f, -18.0f}},
	{
		{2693.0f, 0.940f, 0.0f},	/* zhh           NOTE:: Not
						   Really Done Yet */
		{4000.0f, 0.720f, -10.0f},
		{6123.0f, 0.870f, -10.0f},
		{7755.0f, 0.750f, -18.0f}}
};
