/*
    Copyright (C) 1998-99 Paul Barton-Davis 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: multisample.cc,v 1.1 1999/11/01 04:25:03 pbd Exp $
*/

#include <quasimodo/qm.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/multisample.h>

static MultiSamples multiSamples;

MultiSample::MultiSample (Number l, Number h, const char *soundfile)

{
	pthread_mutex_initialize (&my_lock, NULL);
	low = l;
	high = h;
	sample_name = soundfile;
	table = NULL;
	refcnt = 0;
}

void
MultiSample::lock ()

{
	pthread_mutex_lock (&my_lock);
}

void
MultiSample::unlock ()

{
	pthread_mutex_unlock (&my_lock);
}

void
MultiSample::erase (MultiSample *ms)

{
	ms->unref ();

	if (!ms->inuse ()) {
		multiSamples.erase (ms->ms_position);
		delete ms;
	}
}

MultiSample *
MultiSample::find (const char *name)

{
	MultiSamples::iterator i;
	
	if ((i = multiSamples.find (name)) == multiSamples.end()) {
		return NULL;
	}
	
	return (*i).second;
}

MultiSample *
MultiSample::make (Process *, const char *msname,
		   Number low, Number high,
		   const char *tablename)

{
	MultiSample *ms;
	pair<const char *, MultiSample *> newpair;

	if ((ms = find (msname)) != NULL) {
		ms->ref ();
		return ms;
	}
		
	newpair.first = msname;
	newpair.second = new MultiSample (low, high, tablename);
	
	multiSamples.insert (newpair);

	return newpair.first;
}

void
multisample_make (MULTISAMPLE *p)

{
	MultiSample *ms;
	int32 low;
	int32 high;
	const char *table_name;
	const char *msname;
	size_t nentries;
	size_t i;

	if (!is_string_ptr (p->msname)) {
		p->error (
			       "multisample: non-string first argument");
	}

	msname = get_string (p->msname);

	if (p->INOCOUNT == 1) {
		MultiSample::erase (msname);
		return;
	} else if (p->INOCOUNT != 4) {
		p->error (
			       "multisample: incorrect number of "
			       "arguments (%d)");
	}


	low = *p->low;
	high = *p->high;

	if (!is_string_ptr (p->tablename)) {
		p->error (
			       "multisample: non-string argument "
			       "for table name %d");
	}
	
	table_name = get_string (p->tablename);
	
	MultiSample::make (p->PROCESS, msname, low, high, table_name);
}

void
multisample_pick (MULTISAMPLE *p)

{
	MultiSample *ms;
	const char *msname;

	if (!is_string_ptr (p->msname)) {
		p->error (
			       "multisample: non-string multisample name");
	}

	msname = get_string (p->msname);

	if ((ms = MultiSample::find (msname)) == NULL) {
		p->error (
			       "multisample: no such multisample \"%s\"",
			       msname);
	}

	*p->table = ms->table;
	*p->base_frequency = ms->base_frequency;
}
			       
