/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: moog1.cc,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/function_tables.h>
#include "physutil.h"
#include "moog1.h"



void
Moog1set(MOOG1 * p)
{
	RCPointer<FunctionTable>	ftp;
	float tempCoeffs[2] =
	{0.0f, -1.0f};

	make_ADSR(&p->adsr);
	make_OnePole(&p->filter);
	make_TwoZero(&p->twozeroes[0]);
	TwoZero_setZeroCoeffs(&p->twozeroes[0], tempCoeffs);
	make_TwoZero(&p->twozeroes[1]);
	TwoZero_setZeroCoeffs(&p->twozeroes[1], tempCoeffs);
	make_FormSwep(&p->filters[0]);
	make_FormSwep(&p->filters[1]);

	if ((ftp = ftfind(p->iatt)) != 0)
		p->attk.wave = ftp;	/* mandpluk */
	if ((ftp = ftfind(p->ifn)) != 0)
		p->loop.wave = ftp;	/* impuls20 */
	if ((ftp = ftfind(p->ivfn)) != 0)
		p->vibr.wave = ftp;	/* sinewave */
	p->attk.time = p->attk.phase = 0.0f;
	p->loop.time = p->loop.phase = 0.0f;
	p->vibr.time = p->vibr.phase = 0.0f;
	p->oldfilterQ = p->oldfilterRate = 0.0f;
	ADSR_setAll(&p->adsr, 0.05f, 0.00003f, 0.6f, 0.0002f);
	ADSR_keyOn(&p->adsr);
}

static float last = 0.0f;

void
Moog1(MOOG1 * p)
{
	float amp = *p->amp * AMP_RSCALE;	/* Normalised */
	float *ar = p->ar;
	long nsmps = ksmps;
	float temp;
	float vib = *p->vibAmt;

	p->baseFreq = *p->frequency;
	p->attk.rate = p->baseFreq * 0.01f * p->attk.wave->flen / esr;
	p->loop.rate = p->baseFreq * p->loop.wave->flen / esr;
	p->attackGain = amp * 0.5f;
	p->loopGain = amp;
	if (*p->filterQ != p->oldfilterQ) {
		p->oldfilterQ = *p->filterQ;
		temp = p->oldfilterQ + 0.05f;
		FormSwep_setStates(&p->filters[0], 2000.0f, temp, 2.0f * (1.0f
								- temp));
		FormSwep_setStates(&p->filters[1], 2000.0f, temp, 2.0f * (1.0f
								- temp));
		temp = p->oldfilterQ + 0.099f;
		FormSwep_setTargets(&p->filters[0], 0.0f, temp, 2.0f * (1.0f -
								  temp));
		FormSwep_setTargets(&p->filters[1], 0.0f, temp, 2.0f * (1.0f -
								  temp));
	}
	if (*p->filterRate != p->oldfilterRate) {
		p->oldfilterRate = *p->filterRate;
		p->filters[0].sweepRate = p->oldfilterRate * RATE_NORM;
		p->filters[1].sweepRate = p->oldfilterRate * RATE_NORM;
	}
	p->vibr.rate = *p->vibf * p->vibr.wave->flen / esr;
	do {
		float temp;
		float output;
		long itemp;
		float temp_time, alpha;

		if (vib != 0.0f) {
			temp = vib * Samp_tick(&p->vibr);
			p->loop.rate = p->baseFreq * (1.0f + temp) *
			    (float) (p->loop.wave->flen) / esr;
		}
		p->attk.time += p->attk.rate;	/*  Update current time    
						 */
		temp_time = p->attk.time;
		if (p->attk.time >= (float) p->attk.wave->flen)
			output = 0.0f;	/* One shot */
		else {
			itemp = (long) temp_time;	/*  Integer part of
							   time address    */
			/*  fractional part of time address */
			alpha = temp_time - (float) itemp;
			output = p->attk.wave->ftable[itemp];	/* Do
								   linear interpolation */
			/*  same as alpha*data[itemp+1] + (1-alpha)data[itemp]
			 */

			output += (alpha * (p->attk.wave->ftable[itemp + 1] -
					    output));
			output *= p->attackGain;
			/* End of attack tick */
		}

		output += p->loopGain * Samp_tick(&p->loop);
		output = OnePole_tick(&p->filter, output);
		output *= ADSR_tick(&p->adsr);
		output = TwoZero_tick(&p->twozeroes[0], output);
		output = FormSwep_tick(&p->filters[0], output);
		output = TwoZero_tick(&p->twozeroes[1], output);
		output = FormSwep_tick(&p->filters[1], output);
		last = output;
		*ar++ = output * AMP_SCALE * 8.0f;
	} while (--nsmps);
}

Opcode opcodes[] = {
	MOOG1_OPCODE_LIST,
	{ NULL }
};
