/*
    Copyright (C) 1998-99 Paul Barton-Davis
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: generators_gpl.cc,v 1.1 1999/11/16 19:43:03 pbd Exp $
*/

#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <errno.h>

#include <quasimodo/qm.h>
#include <quasimodo/defaults.h>
#include <quasimodo/quasimodo.h>
#include <quasimodo/tableerror.h>
#include <quasimodo/qm_soundfile.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/generators.h>

#include "generators_gpl.h"

void gen01 (FunctionTableRequest *req, FunctionTable *ftp) 

{
	size_t n;
	int c;
	char sound_path[PATH_MAX+1];
	RCPointer<SoundFile> sf;
	size_t channel;
	size_t nchannels;
	size_t skiptime;
	size_t first_frame;
	TableError err;
	size_t frames_read;
	size_t samples_in_table;
	size_t frames_in_table;
	bool *channel_mask;
	Number **bufs;

	skiptime = (size_t) req->args[2];

	if (req->filename) {
		strcpy (sound_path, req->filename);
	} else {
		sprintf(sound_path, "soundin.%ld", (long) req->args[5]);
	}

	sf = soundfiles->open (sound_path);

	if (sf == 0) {
		err << "gen01: cannot open or grok sound file "
		    << sound_path
		    << endmsg;
	}

        channel = (size_t) req->args[4];

	if (channel && sf->channels() < channel-1) {
		err << "gen: input file "
		    << sf->short_name ()
		    << " has "
		    << sf->channels()
		    << "so you can't possibly read channel "
		    << channel
		    << endmsg;
	}

	first_frame = (long) (skiptime * sf->sample_rate());
	nchannels = sf->channels();
	channel_mask = new bool[nchannels];

	if (channel == 0) {
		/* read all channels */

		for (n = 0; n < nchannels; n++) {
			channel_mask[n] = true;
		}

	} else {

		/* just one channel */

		memset (channel_mask, 0, sizeof (bool) * nchannels);
		channel_mask[channel-1] = true;
		nchannels = 1;
	}

	/* nelements = 0: load was deferred, load entire file
	   nelememts = 1: not deferred, read the entire file

	   Both imply a guard point that wraps the table.
	*/

	if (req->nelements == 0 || req->nelements == 1) {
		ftp->request_length (sf->frames() * nchannels, nchannels);
		ftp->set_guard_type (FunctionTable::GuardWrap);
	} else {
		ftp->request_length (req->nelements, nchannels);
	}

	samples_in_table = MIN (ftp->flen, sf->samples ());
	frames_in_table = samples_in_table / nchannels;

	info << "gen01: soundfile based table is " << samples_in_table << " samples "
	      "(allocated " << ftp->flen << "+1), " << frames_in_table << " frames based "
	      "on sound file of " << sf->samples() << " samples, " << sf->frames () << " frames" << endmsg;

	      
	bufs = new Number *[nchannels];

	/* setup per-channel buffers for Soundfile::read(). In our
	   case, the buffers are interleaved within the ftable.
	*/
	
	for (n = 0, c = 0; n < sf->channels(); n++) {
		if (channel_mask[n]) {
			bufs[n] = &ftp->ftable[c++];
		} else {
			bufs[n] = NULL;
		}
	}

	frames_read = sf->read (first_frame, 
				bufs,
				frames_in_table,
				nchannels - 1,
				channel_mask, NULL);

	delete [] channel_mask;
	delete [] bufs;

	if (frames_read != frames_in_table && frames_read < 0) {
		err << "gen01: read error for "
		    << sound_path
		    << endmsg;
	}

	ftp->soundend = frames_read;
	ftp->cvtbas = LOFACT * sf->sample_rate() / esr;
	ftp->cpscvt = 0.0f;

	/* XXX currently no looping data available from libsoundfile */
	
	ftp->loopmode1 = 0;
	ftp->loopmode2 = 0;
	ftp->end1 = ftp->flenfrms;

	if (req->rescale) {
		ftp->rescale ();
	}
}

void 
gen30 (FunctionTableRequest *req, FunctionTable *ftp)

{
	/* take function table from an internal vector pointed
	   to by req->filename (dirty trick ... ick)
	*/

	ftp->set_table ((Number *) req->filename, req->nelements, 1);
}

void 
gen31 (FunctionTableRequest *req, FunctionTable *ftp)

{
	int fd;
	TableError err;
	size_t file_size;
	unsigned char *dummy;

	/* XXX THIS FUNCTION DOES NOT WORK YET XXX */

	if ((fd = open (req->filename, O_RDONLY)) < 0) {
		err << "Cannot open wavetable file "
		    << req->filename
		    << " (" 
		    << strerror (errno)
		    << ')'
		    << endmsg;
	}

	file_size = lseek (fd, 0L, SEEK_END);
	lseek (fd, 0L, SEEK_SET);

	if (file_size < sizeof (FunctionTable) + 1) {
		err << "File "
		    << req->filename
		    << " is too short to contain a wavetable"
		    << endmsg;
	}

	/* copy the header data into our dummy */

	dummy = new unsigned char[sizeof (FunctionTable)];

	if (read (fd, dummy, sizeof (FunctionTable)) != sizeof (FunctionTable)) {
		close (fd);
		err << "Cannot read wavetable data from "
		    << req->filename 
		    << " ("
		    << strerror (errno)
		    << ')'
		    << endmsg;
	}

	*ftp = *((FunctionTable *) dummy);

	delete dummy;

	ftp->ftable = (Number *) mmap (0, ftp->flen,
				       PROT_READ, MAP_SHARED,
				       fd, 0);
	
	close (fd);

	if (ftp->ftable < 0) {
		err << "Cannot map wavetable file "
		    << req->filename
		    << endmsg;
	}

	ftp->set_mapped (true);
}

Generator generators[] = {
	{ gen01, "AudioFile", Generator::AudioFile, 
	              Generator::usesFile|Generator::canDefer },
	{ gen30, "InternalVector", Generator::InternalVector },
	{ gen31, "MappedFile", Generator::MappedFile, Generator::usesFile },
	{ 0 }
};
