/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: flute.cc,v 1.2 1999/12/16 03:53:34 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/flute.h>
#include "physutil.h"
#include <quasimodo/process.h>


/******************************************/
/*  WaveGuide Flute ala Karjalainen,      */
/*  Smith, Waryznyk, etc.                 */
/*  with polynomial Jet ala Cook          */
/*  by Perry Cook, 1995-96                */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/******************************************/


/**********************************************/
/* Jet Table Object by Perry R. Cook, 1995-96 */
/* Consult Fletcher and Rossing, Karjalainen, */
/*       Cook, more, for information.         */
/* This, as with many other of my "tables",   */
/* is not a table, but is computed by poly-   */
/* nomial calculation.                        */
/**********************************************/

Number
JetTabl_lookup(Number sample)
{				/* Perform "Table Lookup"  *//* By Polynomial
				   Calculation */
	/* (x^3 - x) approximates sigmoid of jet */
	Number j = sample * (sample * sample - 1.0f);
	if (j > 1.0f)
		j = 1.0f;	/* Saturation at +/- 1.0       */
	else if (j < -1.0f)
		j = -1.0f;
	return j;
}

void
fluteset(FLUTE * f)
{
	RCPointer<FunctionTable>	ftp;
	int32 length = 0;
	Number amp = (*f->amp) * AMP_RSCALE;	/* Normalise */
	Number temp;

	if ((ftp = ftfind(f->ifn)) != 0)
		f->vibr = ftp;
	else
		f->error ("No table for Flute");

	/* Expect sine wave */
	if (*f->lowestFreq >= 0.0f) {	/* Skip initialisation?? */
		if (*f->lowestFreq == 0.0)
			length = (int32) (esr / *f->lowestFreq + 1.0f);
		else if (*f->frequency)
			length = (int32) (esr / *f->frequency + 1.0f);
		else
			f->error ("No base frequency for flute");
		make_DLineL(&f->boreDelay, length);
		length = length >> 1;	/* ??? really */
		make_DLineL(&f->jetDelay, length);
		make_OnePole(&f->filter);
		make_DCBlock(&f->dcBlock);
		make_Noise(f->noise);
		make_ADSR(&f->adsr);
		/* Clear */
		OnePole_setPole(&f->filter, 0.7f - (0.1f * RATE_NORM));
		OnePole_setGain(&f->filter, -1.0f);
		ADSR_setAll(&f->adsr, 0.02f, 0.05f, 0.8f, 0.001f);
		/* Suggested values */
		/*    f->endRefl = 0.5; */
		/*    f->jetRefl = 0.5; */
		/*    f->noiseGain = 0.15; *//* Breath pressure random
		   component   */
		/*    f->vibrGain = 0.05;  *//* breath periodic vibrato
		   component  */
		/*    f->jetRatio = 0.32;  */
		f->lastamp = amp;	/* Remember */
		ADSR_setAttackRate(&f->adsr, amp * 0.02f);	/* This
								   should be controlled by attack */
		f->maxPress = (1.1f + (amp * 0.20f)) / 0.8f;
		f->outputGain = amp + 0.001f;
		ADSR_keyOn(&f->adsr);

		{
			qm_time_t relestim = (qm_time_t) (esr * *f->dettack);
			
			if (relestim > f->PROCESS->extra_time)
			    f->PROCESS->extra_time = relestim;
		}

		f->kloop = (int32) ((f->PROCESS->time_off() * ekr) - 
				    (ekr * *f->dettack));


		f->lastFreq = *f->frequency;
		f->lastJet = *f->jetRatio;
		/* freq = (2/3)*f->frequency as we're overblowing here */
		/* but 1/(2/3) is 1.5 so multiply for speed */
		temp = 1.5f * esr / f->lastFreq - 2.0f;		/*
								   Length - approx. filter delay */
		DLineL_setDelay(&f->boreDelay, temp);	/* Length of bore
							   tube */
		DLineL_setDelay(&f->jetDelay, temp * f->lastJet);	/* jet 
									   delay shorter */
	}
}

void
flute(FLUTE * f)
{
	Number *ar = f->ar;
	int32 nsmps = ksmps;
	Number amp = (*f->amp) * AMP_RSCALE;	/* Normalise */
	Number temp;
	int32 v_len = (int32) f->vibr->flen;
	Number *v_data = f->vibr->ftable;
	Number v_time = f->v_time;
	Number vibGain = *f->vibAmt;

	if (amp != f->lastamp) {	/* If amplitude has changed */
		ADSR_setAttackRate(&f->adsr, amp * 0.02f);	/* This
								   shoudl be controlled by attack */
		f->maxPress = (1.1f + (amp * 0.20f)) / 0.8f;
		f->outputGain = amp + 0.001f;
		f->lastamp = amp;
	}
	f->v_rate = *f->vibFreq * v_len / esr;
	/* Start SetFreq */
	if (f->lastFreq != *f->frequency) {	/* It changed */
		f->lastFreq = *f->frequency;
		f->lastJet = *f->jetRatio;
		/* freq = (2/3)*f->frequency as we're overblowing here */
		/* but 1/(2/3) is 1.5 so multiply for speed */
		temp = 1.5f * esr / f->lastFreq - 2.0f;		/*
								   Length - approx. filter delay */
		DLineL_setDelay(&f->boreDelay, temp);	/* Length of bore
							   tube */
		DLineL_setDelay(&f->jetDelay, temp * f->lastJet);	/* jet 
									   delay shorter */
	} else if (*f->jetRatio != f->lastJet) {	/* Freq same but jet
							   changed */
		f->lastJet = *f->jetRatio;
		temp = 1.5f * esr / f->lastFreq - 2.0f;		/*
								   Length - approx. filter delay */
		DLineL_setDelay(&f->jetDelay, temp * f->lastJet);	/* jet 
									   delay shorter */
	}
	/* End SetFreq */

	if (f->kloop >= 0.0f && f->PROCESS->releasing())
		f->kloop = 1.0f;
	if (f->kloop && (--f->kloop) == 0) {
		f->adsr.releaseRate = f->adsr.value / (*f->dettack * esr);
		ADSR_keyOff (&f->adsr);
	}
	do {
		int32 temp;
		Number temf;
		Number temp_time, alpha;
		Number pressDiff;
		Number randPress;
		Number breathPress;
		Number lastOutput;
		Number v_lastOutput;

		breathPress = f->maxPress * ADSR_tick(&f->adsr);	/*
									   Breath Pressure */
		randPress = *f->noiseGain * Noise_tick(&f->noise);	/*
									   Random Deviation */
		/* Tick on vibrato table */
		v_time += f->v_rate;	/*  Update current time    */
		while (v_time >= v_len)		/*  Check for end of
						   sound */
			v_time -= v_len;	/*  loop back to beginning */
		while (v_time < 0.0f)	/*  Check for end of sound */
			v_time += v_len;	/*  loop back to beginning */

		temp_time = v_time;

		temp = (int32) temp_time;	/*  Integer part of time
						   address    */
		/*  fractional part of time address */
		alpha = temp_time - (Number) temp;
		v_lastOutput = v_data[temp];	/* Do linear interpolation
						 */
		/*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
		v_lastOutput += (alpha * (v_data[temp + 1] - v_lastOutput));
		/* End of vibrato tick */
		randPress += vibGain * v_lastOutput;	/* + breath vibrato 
							 */
		randPress *= breathPress;	/* All scaled by Breath
						   Pressure */
		temf = OnePole_tick(&f->filter, DLineL_lastOut(&f->boreDelay));
		temf = DCBlock_tick(&f->dcBlock, temf);		/* Block 
								   DC on reflection */
		pressDiff = breathPress + randPress	/* Breath Pressure   
							 */
		    - (*f->jetRefl * temf);	/*  - reflected      */
		pressDiff = DLineL_tick(&f->jetDelay, pressDiff);	/* Jet 
									   Delay Line */
		pressDiff = JetTabl_lookup(pressDiff)	/* Non-Lin Jet +
							   reflected */
		    +(*f->endRefl * temf);
		lastOutput = 0.3f * DLineL_tick(&f->boreDelay, pressDiff);
		/* Bore Delay and "bell" filter  */

		lastOutput *= f->outputGain;
		*ar++ = lastOutput * AMP_SCALE * 1.4f;
	} while (--nsmps);

	f->v_time = v_time;
}

Opcode opcodes[] = {
	FLUTE_OPCODE_LIST,
	{ NULL }
};
