#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include "dam.h"

/*
 *   Dynamic Amplitude Modifier.
 *
 *       (C) Marc Resibois 1997
 *
 *   I place this source code in the public domain. Just
 *   let me know if you do something you like with it ;-)
 *
 *   For bugs, question, please write to Marc.Resibois@ping.be
 */


/*
 *    Initialisation code
 */

void
daminit(DAM * p)
{
	int32 i;

	/* Initialise gain value */

	p->gain = 1.0f;

	/* Compute the gain speed changes from parameter given by Csound */
	/* the computed values are stored in the opcode data structure p */
	/* for later use in the main processing                          */

	p->rspeed = (*p->rtime) / esr * 1000.0f;
	p->fspeed = (*p->ftime) / esr * 1000.0f;

	/* Initialize power value and buffer */

	p->power = (*p->kthreshold);
	for (i = 0; i < POWER_BUFSIZE; i++) {
		p->powerBuffer[i] = p->power / (Number) POWER_BUFSIZE;
	}

	p->powerPos = p->powerBuffer;
}

/*
 * Run-time computation code
 */

void
dam(DAM * p)
{
	size_t i;
	sample_t *ain, *aout;
	Number threshold;
	Number gain;
	Number comp1, comp2;
	Number *powerPos;
	Number *powerBuffer;
	Number power;
	Number tg;

	ain = p->ain;
	aout = p->aout;
	threshold = (*p->kthreshold);
	gain = p->gain;
	comp1 = (*p->icomp1);
	comp2 = (*p->icomp2);
	powerPos = p->powerPos;
	powerBuffer = p->powerBuffer;
	power = p->power;

	/* Process ksmps samples */

	for (i = 0; i < ksmps; i++) {

		/* Estimates the current power level */

		*powerPos = (Number) (fabs(ain[i])) / (Number) (POWER_BUFSIZE *
							      sqrt(2.0));
		power += (*powerPos++);
		if ((powerPos - powerBuffer) == POWER_BUFSIZE) {
			powerPos = p->powerBuffer;
		}
		power -= (*powerPos);

		/* Looks where the power is related to the treshold
		   and compute target gain */

		if (power > threshold) {
			tg = ((power - threshold) * comp1 + threshold) / power;
		} else {
			tg = threshold * (Number) (pow((double) (power /
							      threshold),
					  1.0 / (double) comp2)) / power;
		}

		/* move gain toward target */

		if (gain < tg) {
			gain += p->rspeed;
		} else {
			gain -= p->fspeed;
		}

		/* compute output */

		aout[i] = ain[i] * gain;
	}

	/* Store the last gain value for next call */

	p->gain = gain;
	p->power = power;
	p->powerPos = powerPos;

}

Opcode opcodes[] =
{
	DAM_OPCODE_LIST,
	{ 0 }
};
