/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: brass.cc,v 1.2 1999/12/16 03:53:34 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/process.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/process.h>

#include "physutil.h"
#include "brass.h"


/******************************************/
/*  Waveguide Brass Module Model ala  */
/*  Cook (TBone, HosePlayer)              */
/*  by Perry R. Cook, 1995-96             */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/******************************************/

/* ====================================================================== */

/****************************************************************************/
/*  Lip Filter Object by Perry R. Cook, 1995-96                             */
/*  The lip of the brass player has dynamics which are controlled by the    */
/*  mass, spring constant, and damping of the lip.  This filter simulates   */
/*  that behavior and the transmission/reflection properties as well.       */
/*  See Cook TBone and HosePlayer instruments and articles.                 */
/****************************************************************************/

void
make_LipFilt(LipFilt * p)
{
	make_BiQuad(p);
}

void
LipFilt_setFreq(LipFilt * p, Number frequency)
{
	Number coeffs[2];
	coeffs[0] = 2.0f * 0.997f *
	    (Number) cos((double) (twopi * frequency / esr));	/* damping
								   should  */
	coeffs[1] = -0.997f * 0.997f;	/* change with lip */
	BiQuad_setPoleCoeffs(p, coeffs);	/* parameters, but */
	BiQuad_setGain(*p, 0.03f);	/* not yet.        */
}

/*  NOTE:  Here we should add lip tension                 */
/*              settings based on Mass/Spring/Damping     */
/*              Maybe in TookKit97                        */

Number
LipFilt_tick(LipFilt * p, Number mouthSample, Number boreSample)
		/*   Perform "Table Lookup" By Polynomial Calculation */
{
	Number temp;
	Number output;
	temp = mouthSample - boreSample;	/* Differential pressure        
						 */
	temp = BiQuad_tick(p, temp);	/* Force -> position            */
	temp = temp * temp;	/* Simple position to area mapping */
	if (temp > 1.0f)
		temp = 1.0f;	/* Saturation at + 1.0          */
	output = temp * mouthSample;	/* Assume mouth input = area    */
	output += (1.0f - temp) * boreSample;	/* and Bore reflection is
						   compliment */
	return output;
}

/* ====================================================================== */

void
brassset(BRASS * p)
{
	RCPointer<FunctionTable>	ftp;
	Number amp = (*p->amp) * AMP_RSCALE;	/* Normalise */

	if ((ftp = ftfind(p->ifn)) != 0)
		p->vibr = ftp;
	else
		p->error ("No table for Brass");
	
        /* Expect sine wave */
	p->frq = *p->frequency;	/* Remember */
	if (*p->lowestFreq >= 0.0f) {
		if (*p->lowestFreq != 0.0f)
			p->length = (long) (esr / *p->lowestFreq + 1.0f);
		else if (p->frq != 0.0f)
			p->length = (long) (esr / p->frq + 1.0f);
		else
			p->error ("No base frequency for brass");
		make_DLineA(&p->delayLine, p->length);
		make_LipFilt(&p->lipFilter);
		make_DCBlock(&p->dcBlock);
		make_ADSR(&p->adsr);
		ADSR_setAll(&p->adsr, 0.02f, 0.05f, 1.0f, 0.001f);

		ADSR_setAttackRate(&p->adsr, amp * 0.001f);

		p->maxPressure = amp;
		ADSR_keyOn(&p->adsr);

		/* Set frequency */
	        p->slideTarget = (esr / p->frq * 2.0f) + 3.0f;

		/* fudge correction for filter delays */
		DLineA_setDelay(&p->delayLine, p->slideTarget);

		p->lipTarget = 0.0f;
		p->lipT = 0.0f;

		/* End of set frequency */

		{
			qm_time_t relestim = (qm_time_t) (ekr * *p->dettack);
			if (relestim > p->PROCESS->extra_time) {
				p->PROCESS->extra_time = relestim;
			}
		}

		p->kloop = (int32) ((p->PROCESS->time_off() * ekr) - 
				    (ekr * *p->dettack));
	}
}

void
brass(BRASS * p)
{
	Number *ar = p->ar;
	long nsmps = ksmps;
	Number amp = (*p->amp) * AMP_RSCALE;	/* Normalise */
	Number maxPressure = p->maxPressure = amp;
	int32 v_len = (int32) p->vibr->flen;
	Number *v_data = p->vibr->ftable;
	Number vibGain = *p->vibAmt;
	Number vTime = p->v_time;

	p->v_rate = *p->vibFreq * v_len / esr;

	if (p->kloop >= 0 && p->PROCESS->releasing())
		p->kloop = 1;

	if (p->kloop && (--p->kloop) == 0) {
		ADSR_setReleaseRate(&p->adsr, amp * 0.005f);
		ADSR_keyOff(&p->adsr);
	}

	if (p->frq != *p->frequency) {	/* Set frequency if changed */
		p->frq = *p->frequency;
		p->slideTarget = (esr / p->frq * 2.0f) + 3.0f;
		/* fudge correction for filter delays */
		DLineA_setDelay(&p->delayLine, p->slideTarget);
		/*  we'll play a harmonic */
		p->lipTarget = p->frq;
	}			

	if (*p->liptension != p->lipT) {
		p->lipT = *p->liptension;
		LipFilt_setFreq(&p->lipFilter,
		 p->lipTarget * (Number) pow(4.0, (2.0 * p->lipT) - 1.0));
	}

	do {
		Number breathPressure;
		Number lastOutput;
		int32 temp;
		Number temp_time, alpha;
		Number v_lastOutput;

		breathPressure = maxPressure * ADSR_tick(&p->adsr);

		/* Tick on vibrato table */

		vTime += p->v_rate;	/*  Update current time    */
		while (vTime >= v_len)	/*  Check for end of sound */
			vTime -= v_len;		/*  loop back to
						   beginning */
		while (vTime < 0.0f)	/*  Check for end of sound */
			vTime += v_len;		/*  loop back to
						   beginning */

		temp_time = vTime;
		temp = (int32) temp_time;	
		alpha = temp_time - vTime;
		v_lastOutput = v_data[temp];
		v_lastOutput += (alpha * (v_data[temp + 1] - v_lastOutput));

		/* End of vibrato tick */
		breathPressure += vibGain * v_lastOutput;

		lastOutput =
		    DLineA_tick
			(&p->delayLine,	             /* bore delay  */
			 DCBlock_tick(&p->dcBlock,    /* block DC */
				      LipFilt_tick
				      (&p->lipFilter,
				       0.3f * breathPressure, /* mouth input */
				       0.85f * p->delayLine.lastOutput)));
          	 	                        /* and bore reflection */

		*ar++ = lastOutput * AMP_SCALE * 3.5f;
	} while (--nsmps);

	p->v_time = vTime;
}

Opcode opcodes[] = {
	BRASS_OPCODE_LIST,
	{ NULL }
};

