/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: bowed.cc,v 1.2 1999/12/16 03:53:34 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/process.h>

#include "physutil.h"
#include "bowed.h"

/******************************************/
/*  Bowed String model ala Smith          */
/*  after McIntyre, Schumacher, Woodhouse */
/*  by Perry Cook, 1995-96                */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/******************************************/


/******************************************/
/*  Simple Bow Table Object, after Smith  */
/*    by Perry R. Cook, 1995-96           */
/******************************************/

Number
BowTabl_lookup(BowTabl * b, Number sample)
{				/*  Perform Table Lookup    *//*  sample is
				   differential  */
	/*  string vs. bow velocity */
	Number input;
	input = sample /* + b->offSet */ ;	/*  add bias to sample     
						 */
	input *= b->slope;	/*  scale it                */
	b->lastOutput = (Number) fabs(input) + 0.75f;	/*  below min
							   delta, frict = 1 */
	b->lastOutput = (Number) pow((double) b->lastOutput, -4.0);
	if (b->lastOutput > 1.0f)
		b->lastOutput = 1.0f;	/*  maximum friction is 1.0 */
	return b->lastOutput;
}

void bowedset(BOWED *b)
{
	int32	length;
	qm_time_t relestim;
	RCPointer<FunctionTable>	ftp;
	Number	amp = (*b->amp)*AMP_RSCALE; /* Normalise */

	if ((ftp = ftfind (b->ifn)) != 0)	{
		b->vibr = ftp;
	} else {
		b->error ("No table for wgbow vibrato");
	}

	if (*b->lowestFreq < 0.0f) {	/* If no init skip */
		return;
	}
	if (*b->lowestFreq!=0.0f) {
		length = (int32) (esr / *b->lowestFreq + 1.0f);
	} else if (*b->frequency!=0.0f) {
		length = (int32) (esr / *b->frequency + 1.0f);
	} else {
		length = (int32) (esr / 50.0f + 1.0f);
	}

	make_DLineL(&b->neckDelay, length);
	/*    length = length >> 1; */       /* ?? Unsure about this */
	make_DLineL(&b->bridgeDelay, length);

        /* offset is a bias, really not needed unless 
	   friction is different in each direction
	*/

	/*  b->bowTabl.offSet = 0.0f;*/ 

	/* b->bowTabl.slope contrls width of friction pulse, related
           to bowForce 
	*/

	b->bowTabl.slope = 3.0f;
	make_OnePole(&b->reflFilt);
	make_BiQuad(&b->bodyFilt);
	make_ADSR(&b->adsr);

	DLineL_setDelay(&b->neckDelay, 100.0f);
	DLineL_setDelay(&b->bridgeDelay, 29.0f);

	OnePole_setPole(&b->reflFilt, 0.6f - (0.1f * RATE_NORM));
	OnePole_setGain(&b->reflFilt, 0.95f);

	BiQuad_setFreqAndReson(b->bodyFilt, 500.0f, 0.85f);
	BiQuad_setEqualGainZeroes(b->bodyFilt);
	BiQuad_setGain(b->bodyFilt, 0.2f);

	ADSR_setAll(&b->adsr, 0.002f,0.01f,0.9f,0.01f);
	ADSR_keyOn (&b->adsr);

	b->maxVelocity = 0.03f + (0.2f * amp);
	b->lastpress = 0.0f;	/* Set unknown state */
	b->lastfreq = 0.0f;
	b->lastbeta = 0.0f;	/* Remember states */
	b->lastamp = amp;

	relestim = (qm_time_t) (esr * *b->decay);
	
	if (relestim > b->PROCESS->extra_time) {
		b->PROCESS->extra_time = relestim;
	}

	b->kloop = (int32) ((b->PROCESS->time_off() * ekr)-(ekr * *b->decay));
}


void bowed(BOWED *b)
{
	Number	*ar = b->ar;
	int32 	nsmps = ksmps;
	Number 	amp = (*b->amp)*AMP_RSCALE; /* Normalise */
	Number	maxVel;
	int32		freq_changed = 0;

	if (amp != b->lastamp) {
		b->maxVelocity = 0.03f + (0.2f * amp);
		b->lastamp = amp;
	}
	maxVel = b->maxVelocity;
	if (b->lastpress != *b->bowPress)
	    b->bowTabl.slope = b->lastpress = *b->bowPress;
      
	if (b->lastfreq != *b->frequency) {
		/* delay - approx. filter delay */
		b->lastfreq = *b->frequency;
		b->baseDelay = esr / b->lastfreq - 4.0f;
		freq_changed = 1;
	}

	if (b->lastbeta != *b->betaRatio || freq_changed) {
		b->lastbeta = *b->betaRatio;
		DLineL_setDelay(&b->bridgeDelay, /* bow to bridge length */
				b->baseDelay * b->lastbeta);
		DLineL_setDelay(&b->neckDelay, /* bow to nut (finger) length */
				b->baseDelay *(1.0f - b->lastbeta));
	}

	b->v_rate = *b->vibFreq * b->vibr->flen / esr;

	if (b->kloop >= 0 && b->PROCESS->releasing()) { 
		b->kloop=1;
	}

	if (b->kloop && (--b->kloop) == 0) {
		/* WAS: (1.0f - b->adsr.value) * 0.005f); */
		ADSR_setDecayRate (&b->adsr, 
				   b->adsr.value / (*b->decay * esr));
		ADSR_keyOff (&b->adsr);
	}

	do {
		Number	bowVelocity;
		Number	bridgeRefl=0.0f, nutRefl=0.0f;
		Number	newVel=0.0f, velDiff=0.0f, stringVel=0.0f;
		Number	lastOutput;

		bowVelocity = maxVel * ADSR_tick(&b->adsr);

		bridgeRefl = - OnePole_tick(&b->reflFilt, b->bridgeDelay.lastOutput);  /* Bridge Reflection      */
		nutRefl = - b->neckDelay.lastOutput; /* Nut Reflection  */
		stringVel = bridgeRefl + nutRefl; /* Sum is String Velocity */
		velDiff = bowVelocity - stringVel; /* Differential Velocity  */
		newVel = velDiff * BowTabl_lookup(&b->bowTabl, velDiff);  /* Non-Lin Bow Function   */
		DLineL_tick(&b->neckDelay, bridgeRefl + newVel);	/* Do string       */
		DLineL_tick(&b->bridgeDelay, nutRefl + newVel);   /*   propagations  */

		if (*b->vibAmt > 0.0f) {
			int32	temp;
			Number	temp_time, alpha;
                                /* Tick on vibrato table */
			b->v_time += b->v_rate;              /*  Update current time    */
			while (b->v_time >= b->vibr->flen)   /*  Check for end of sound */
			    b->v_time -= b->vibr->flen;        /*  loop back to beginning */
			while (b->v_time < 0.0f)             /*  Check for end of sound */
			    b->v_time += b->vibr->flen;        /*  loop back to beginning */
        
			temp_time = b->v_time;
    
			temp = (int32) temp_time;    /*  Integer part of time address    */
                                /*  fractional part of time address */
			alpha = temp_time - (Number)temp; 
			b->v_lastOutput = b->vibr->ftable[temp]; /* Do linear interpolation */
                        /*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
			b->v_lastOutput = b->v_lastOutput +
				(alpha * (b->vibr->ftable[temp+1] - b->v_lastOutput));
                                /* End of vibrato tick */

			DLineL_setDelay(&b->neckDelay,
					(b->baseDelay * (1.0f - b->lastbeta)) +
					(b->baseDelay * *b->vibAmt * b->v_lastOutput));
		} else {
			DLineL_setDelay(&b->neckDelay,
					(b->baseDelay * (1.0f - b->lastbeta)));
		}

		lastOutput = BiQuad_tick(&b->bodyFilt, 
					 b->bridgeDelay.lastOutput);

		*ar++ = lastOutput*AMP_SCALE*1.8f;

	} while (--nsmps);
}

Opcode opcodes[] = {
	BOWED_OPCODE_LIST,
	{ NULL }
};
