// ------------------------------------------------------------------------
// qecutevent.cpp: Cut specified range to clipboard
// Copyright (C) 2000 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <cstdio>
#include <stdlib.h> /* getenv() */

#include <kvu_temporary_file_directory.h>
#include "qecutevent.h"

using std::string;

QECutEvent::QECutEvent(ECA_CONTROL* ctrl,
		       const string& input,
		       const string& output,
		       long int start_pos, 
		       long int length) 
  : QEBlockingEvent(ctrl),
    ectrl(ctrl),
    start_pos_rep(start_pos),
    length_rep(length),
    input_rep(input),
    output_rep(output) {

  status_info("Cutting selected area...");
  init("cutevent-phase1", "default");
  set_input(input_rep);
  set_input_position(start_pos);
  set_length(length);
  set_default_audio_format(input_rep);
  ectrl->set_chainsetup_output_mode(AUDIO_IO::io_write);
  set_output(output_rep);
  set_output_position(0);
}

void QECutEvent::start(void) {
  // copy cut-area to clipboard
  status_info("Copying selected area to clipboard...");
  blocking_start(); 

  TEMPORARY_FILE_DIRECTORY tempfile_dir_rep;
  string tmpdir ("ecawave-");
  char* tmp_p = getenv("USER");
  if (tmp_p != NULL) {
    tmpdir += string(tmp_p);
    tempfile_dir_rep.reserve_directory(tmpdir);
  }
  if (tempfile_dir_rep.is_valid() != true) {
    status_info("Cut failed...");
    return;
  }

  // copy before-cut to temporary file
  string tmpfile = tempfile_dir_rep.create_filename("cut-tmp", ".wav");
  if (start_pos_rep != 0) {
    init("cutevent-phase2", "default");
    set_input(input_rep);
    set_input_position(0);
    set_default_audio_format(input_rep);
    ectrl->set_chainsetup_output_mode(AUDIO_IO::io_write);
    set_output(tmpfile);
    set_length(start_pos_rep);
    status_info("Copying begin to a temporary file...");
    blocking_start();
  }

  // copy after-cut to temporary file
  init("cutevent-phase3", "default");
  set_input(input_rep);
  set_input_position(start_pos_rep + length_rep);
  set_default_audio_format(input_rep);
  set_length(0);
  ectrl->set_chainsetup_output_mode(AUDIO_IO::io_readwrite);  
  set_output(tmpfile);
  set_output_position(start_pos_rep);
  status_info("Copying end to the temporary file...");
  blocking_start();

  // copy temporary over the input file  
  init("cutevent-phase4", "default");
  ectrl->set_chainsetup_output_mode(AUDIO_IO::io_write);
  set_input(tmpfile);
  set_default_audio_format(tmpfile);
  set_output(input_rep);
  set_length(0);
  status_info("Copying from the temporary file...");
  blocking_start();
  
  // remove the tempfile
  ::remove(tmpfile.c_str());
}
