// ------------------------------------------------------------------------
// qesession.cpp: Class representing an ecawave session
// Copyright (C) 2000,2001 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cstdio>
#include <sys/stat.h> /* stat() */
#include <unistd.h> /* stat() */
#include <stdlib.h> /* getenv() */

#include <qapplication.h>
#include <qmainwindow.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qtimer.h>
#include <qmessagebox.h>
#include <qprogressdialog.h>
#include <qaccel.h>
#include <qvbox.h>
#include <qmenubar.h>
#include <qpopupmenu.h>

#include <ecasound/eca-session.h>
#include <ecasound/eca-control.h>

#include "qebuttonrow.h"
#include "qefile.h"
#include "qeopenfiledialog.h"
#include "qesavefiledialog.h"
#include "qesession.h"
#include "qestatusbar.h"

#include "qeevent.h"
#include "qenonblockingevent.h"
#include "qeplayevent.h"
#include "qesaveevent.h"
#include "qechainopevent.h"
#include "qecopyevent.h"
#include "qepasteevent.h"
#include "qecutevent.h"
#include "qefadeinevent.h"
#include "qefadeoutevent.h"

QESession::QESession (QMainWindow *parent,
		      const char *name)
  : QWidget(parent, name),
    state_rep(state_no_file),
    fileview_repp(0),
    audio_io_repp(0),
    buttonrow_repp(0),
    buttonrow2_repp(0),
    buttonrow3_repp(0),
    nb_event_repp(0),
    mainwindow_repp(parent),
    menubar_repp(parent->menuBar()),
    statusbar_repp(0),
    vlayout_repp(0),
    refresh_toggle_rep(false),
    wcache_toggle_rep(true),
    direct_mode_rep(false) {

  esession_repp = new ECA_SESSION();
  ectrl_repp = new ECA_CONTROL(esession_repp);

  auto_ptr<ECA_SESSION> p (esession_repp);
  auto_esession_rep = p;

  auto_ptr<ECA_CONTROL> q (ectrl_repp);
  auto_ectrl_rep = q;

  startTimer(100);
  QTimer *timer = new QTimer( this );
  connect( timer, SIGNAL(timeout()), this, SLOT(position_update()));
  timer->start(500, false);

  QAccel* a = new QAccel (this);
  a->connectItem(a->insertItem(ALT+CTRL+Key_D), this, SLOT(debug_event()));

  no_file_label_repp = new QLabel("No active file.", this);
  no_file_label_repp->setAlignment(QLabel::AlignHCenter | QLabel::AlignVCenter);

  init_menubar();
  init_layout();

  init_temp_directory();

  // --------
  ENSURE(ectrl_repp != 0);
  ENSURE(esession_repp != 0);
  // --------
}

void QESession::init_temp_directory(void) {
  string tmpdir ("ecawave-");
  char* tmp_p = getenv("USER");
  if (tmp_p != NULL) {
    tmpdir += string(tmp_p);
    tempfile_dir_rep.reserve_directory(tmpdir);
  }
  if (tempfile_dir_rep.is_valid() != true) {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", 
		      QString("Warning! Can't access temporary directory  ") + 
		      QString(tempfile_dir_rep.get_directory_prefix().c_str()) +
		      QString("/") +
		      QString(tmpdir.c_str()) +
		      QString(". Ecawave may not work properly."),
		      0);
  }
}

void QESession::open_file(const string& filename) {
  stop_event();
  if (temp_files_created() == true) remove_temps();

  set_active_filename(filename);
  if (active_filename_rep.empty() == true) {
    change_state_to_no_file();
  }
  else {
    fileview_repp = new QEFileView(active_filename_rep,
				   this,
				   "fileview");
    if (fileview_repp->is_valid() == true) {
      set_orig_filename(active_filename_rep);
      change_state_to_named();
    }
  }
 
  init_layout();

  // --------
  ENSURE(state_rep == state_named || state_rep == state_no_file);
  // --------
}

void QESession::timerEvent( QTimerEvent * ) {
  if (nb_event_repp != 0) {
    if (nb_event_repp->is_finished() == true &&
	nb_event_repp->is_triggered() == true) {
      nb_event_repp->stop();
      emit engine_status("stopped");
    }
    else
      if (nb_event_repp->is_triggered()) emit engine_status("running");
  }
}

QESession::~QESession(void) {
  if (nb_event_repp != 0) {
    if (nb_event_repp->is_triggered() &&
  	ectrl_repp->is_running()) nb_event_repp->stop();
    emit engine_status("stopped");
    delete nb_event_repp;
  }
  if (temp_files_created() == true) remove_temps();
}

void QESession::set_active_filename(const string& name) {
  active_filename_rep = name;
}

void QESession::set_orig_filename(const string& name) {
  orig_filename_rep = name;
  emit filename_changed(orig_filename_rep);
}

void QESession::remove_temps(void) {
  // --------
  REQUIRE(temp_files_created() == true);
  // --------

  for(unsigned int n = 0; n < temp_filenames_rep.size(); n++) {
    if (temp_filenames_rep[n].size() > 0) {
      ::remove(temp_filenames_rep[n].c_str());
      ::remove((temp_filenames_rep[n] + ".ews").c_str());
    }
  }
  temp_filenames_rep.clear();
  temp_files_created_rep = false;

  // --------
  ENSURE(temp_filenames_rep.size() == 0);
  ENSURE(temp_files_created() != true);
  // --------
}

void QESession::position_update(void) {
  static bool toggle = false;
  long int pos = 0;
  if (ectrl_repp->is_running() == true) {
    toggle = true;
    if (nb_event_repp != 0)
      pos = nb_event_repp->position_in_samples() - ectrl_repp->get_chainsetup()->buffersize();
    else
      pos = ectrl_repp->position_in_samples() - ectrl_repp->get_chainsetup()->buffersize();
  }
  if (toggle == true) {
    if (fileview_repp != 0) {
      if (pos > 0) fileview_repp->current_position(pos);
      else fileview_repp->current_position(0);
    }
  }
  if (ectrl_repp->is_running() != true) toggle = false;
}

void QESession::init_menubar(void) {

  QPopupMenu *file_menu = new QPopupMenu();
  file_menu->insertItem("New file", this, SLOT(new_event()), CTRL+Key_F);
  file_menu->insertItem("New session", this, SLOT(new_session()), CTRL+Key_N);
  file_menu->insertItem("Open", this, SLOT(open_event()), CTRL+Key_O);
  file_menu->insertItem("Save", this, SLOT(save_event()), CTRL+Key_V);
  file_menu->insertItem("Save As", this, SLOT(save_as_event()), CTRL+Key_A);
  file_menu->insertItem("Close", this, SLOT(close_session()), CTRL+Key_C);
  menubar_repp->insertItem("File", file_menu);

  QPopupMenu *control_menu = new QPopupMenu();
  control_menu->insertItem("Start", this, SLOT(play_event()), CTRL+Key_T);
  control_menu->insertItem("Stop", this, SLOT(stop_event()), CTRL+Key_S);
  menubar_repp->insertItem("Control", control_menu);

  QPopupMenu *edit_menu = new QPopupMenu();
  edit_menu->insertItem("Cut", this, SLOT(cut_event()), CTRL+Key_U);
  edit_menu->insertItem("Copy", this, SLOT(copy_event()), CTRL+Key_Y);
  edit_menu->insertItem("Paste", this, SLOT(paste_event()), CTRL+Key_P);
  edit_menu->insertItem("Effect menu", this, SLOT(effect_event()), CTRL+Key_E);
  edit_menu->insertItem("Fade in", this, SLOT(fade_in_event()), CTRL+Key_I);
  edit_menu->insertItem("Fade out", this, SLOT(fade_out_event()), CTRL+Key_D);
  menubar_repp->insertItem("Edit", edit_menu);

  menubar_repp->setSeparator(QMenuBar::InWindowsStyle);
}

/**
 * Initializes the widget layout. Depending on the current session 
 * state, different layouts are used. If init_layout() is called 
 * multiple times, the result should be same if the session state
 * remains the same between individual calls.
 */
void QESession::init_layout(void) {
  if (vlayout_repp != 0) {
    delete vlayout_repp;
  }
  vlayout_repp = new QVBoxLayout(this);

  if (buttonrow_repp == 0) {
    buttonrow_repp = new QEButtonRow(this, "buttonrow");
    buttonrow_repp->add_button(new QPushButton("(N)ew session",buttonrow_repp), 
			  CTRL+Key_N,
			  this, SLOT(new_session()));
    buttonrow_repp->add_button(new QPushButton("New (f)ile",buttonrow_repp), 
			  CTRL+Key_F, this, SLOT(new_event()));
    buttonrow_repp->add_button(new QPushButton("(O)pen",buttonrow_repp), 
			  CTRL+Key_O, this, SLOT(open_event()));
    buttonrow_repp->add_button(new QPushButton("Sa(v)e",buttonrow_repp), 
			  CTRL+Key_V, this, SLOT(save_event()));
    buttonrow_repp->add_button(new QPushButton("Save (a)s",buttonrow_repp), 
			  CTRL+Key_A, this, SLOT(save_as_event()));
    buttonrow_repp->add_button(new QPushButton("(C)lose",buttonrow_repp), 
			  CTRL+Key_C, this, SLOT(close_session()));

  }
  vlayout_repp->addWidget(buttonrow_repp);

  if (buttonrow2_repp == 0) {
    buttonrow2_repp = new QEButtonRow(this, "buttonrow2_repp");
    buttonrow2_repp->add_button(new QPushButton("S(t)art",buttonrow2_repp), CTRL+Key_T,
				this, SLOT(play_event()));
    buttonrow2_repp->add_button(new QPushButton("(S)top",buttonrow2_repp), CTRL+Key_S,
				this, SLOT(stop_event()));
    buttonrow2_repp->add_button(new QPushButton("(E)ffect",buttonrow2_repp), CTRL+Key_E,
				this, SLOT(effect_event()));
    buttonrow2_repp->add_button(new QPushButton("Fade (i)n",buttonrow2_repp), CTRL+Key_I,
				this, SLOT(fade_in_event()));
    buttonrow2_repp->add_button(new QPushButton("Fa(d)e out",buttonrow2_repp), CTRL+Key_D,
				this, SLOT(fade_out_event()));
    buttonrow2_repp->add_button(new QPushButton("Cop(y)",buttonrow2_repp), CTRL+Key_Y,
				this, SLOT(copy_event()));
    buttonrow2_repp->add_button(new QPushButton("C(u)t",buttonrow2_repp), CTRL+Key_U,
				this, SLOT(cut_event()));
    buttonrow2_repp->add_button(new QPushButton("(P)aste",buttonrow2_repp), CTRL+Key_P,
				this, SLOT(paste_event()));
  }
  vlayout_repp->addWidget(buttonrow2_repp);

  if (state_rep == state_no_file) {
    vlayout_repp->addWidget(no_file_label_repp, 2);
    no_file_label_repp->show();
    if (buttonrow3_repp != 0) buttonrow3_repp->hide();
    if (fileview_repp != 0) fileview_repp->hide();
  }
  else {
    if (fileview_repp != 0) {
      if (buttonrow3_repp != 0) delete buttonrow3_repp;
      buttonrow3_repp = new QEButtonRow(this, "buttonrow3_repp");
      buttonrow3_repp->add_button(new QPushButton("(Z)oom in",buttonrow3_repp), 
				  CTRL+Key_Z,
				  fileview_repp, SLOT(zoom_to_marked()));
      buttonrow3_repp->add_button(new QPushButton("Zoo(m) out",buttonrow3_repp), 
				  CTRL+Key_M,
				  fileview_repp, SLOT(zoom_out()));
      buttonrow3_repp->add_button(new QPushButton("Ma(r)k all",buttonrow3_repp),
				  CTRL+Key_R,
				  fileview_repp, SLOT(mark_all()));
      buttonrow3_repp->add_button(new QPushButton("(U)nmark",buttonrow3_repp), 
				  CTRL+Key_U,
				  fileview_repp, SLOT(unmark()));
      buttonrow3_repp->add_button(new QPushButton("Redra(w)",buttonrow3_repp), 
				  CTRL+Key_W,
				  fileview_repp, SLOT(update_wave_form_data()));
      vlayout_repp->addWidget(buttonrow3_repp);
      buttonrow3_repp->show();
    
      QObject::connect(this, 
		       SIGNAL(filename_changed(const string&)), 
		       fileview_repp,
		       SLOT(title(const string&)));
      vlayout_repp->addWidget(fileview_repp, 1);
      fileview_repp->show();
    }
    no_file_label_repp->hide();
  }
  
  if (statusbar_repp == 0) {
    statusbar_repp = new QEStatusBar(mainwindow_repp, "statusbar");
  }
  
  if (state_rep != state_no_file) {
    statusbar_repp->visible_area(ECA_AUDIO_TIME(0, fileview_repp->samples_per_second()),
				 ECA_AUDIO_TIME(fileview_repp->length(), 
						fileview_repp->samples_per_second()));
    
    QObject::connect(fileview_repp, 
		     SIGNAL(visible_area_changed(ECA_AUDIO_TIME,
						 ECA_AUDIO_TIME)), 
		     statusbar_repp, 
		     SLOT(visible_area(ECA_AUDIO_TIME, ECA_AUDIO_TIME)));
    QObject::connect(fileview_repp, 
		     SIGNAL(marked_area_changed(ECA_AUDIO_TIME,
						ECA_AUDIO_TIME)), 
		     statusbar_repp, 
		     SLOT(marked_area(ECA_AUDIO_TIME, ECA_AUDIO_TIME)));
    QObject::connect(fileview_repp, 
		     SIGNAL(current_position_changed(ECA_AUDIO_TIME)), 
		     statusbar_repp,
		     SLOT(current_position(ECA_AUDIO_TIME)));

    QObject::connect(this,
		     SIGNAL(engine_status(const string&)),
		     statusbar_repp,
		     SLOT(status(const string&)));
  }
}

void QESession::debug_event(void) {
  cerr << "----------------" << endl;
  cerr << "- ecawave-debug:" << endl;
  cerr << "orig_filename_rep: " << orig_filename_rep << endl;
  cerr << "active_filename_rep: " << active_filename_rep << endl;
  cerr << "state_rep: " << static_cast<int>(state_rep) << endl;

  if (temp_files_created()) cerr << "temp files created: true" << endl;
  else cerr << "temp files created: false" << endl;

  cerr << "start_pos_rep: " << start_pos_rep << endl;
  cerr << "sel_length_rep: " << sel_length_rep << endl;

  if (fileview_repp != 0) cerr << "qefile-filename: " <<
			    fileview_repp->filename() << endl;
  else cerr << "qefile not created." << endl;

  if (refresh_toggle_rep) cerr << "refresh_toggle_rep: true" << endl;
  else cerr << "refresh_toggle_rep: false" << endl;

  if (wcache_toggle_rep) cerr << "wcache_toggle_rep: true" << endl;
  else cerr << "wcache_toggle_rep: false" << endl;

  if (direct_mode_rep) cerr << "direct_mode_rep: true" << endl;
  else cerr << "direct_mode_rep: false" << endl;
 
  if (ectrl_repp == 0) cerr << "ectrl not created." << endl;
  else {
    cerr << "ectrl status: " << ectrl_repp->engine_status() << endl;
    cerr << "ectrl position: " << ectrl_repp->position_in_seconds_exact()
	 << "sec" << endl;
  }
}

void QESession::new_session(void) { emit new_session_request(); }
void QESession::close_session(void) { 
  // FIXME: ask whether to delete stuff stored into ecawave clipboard?
  if (temp_files_created() == true) remove_temps();
  emit session_closed(); 
}

void QESession::prepare_event(void) { 
  start_pos_rep = 0;
  sel_length_rep = 0;
  if (fileview_repp != 0 && fileview_repp->is_valid() == true) {
    if (fileview_repp->is_marked() == true) {
      start_pos_rep = fileview_repp->marked_area_start();
      sel_length_rep = fileview_repp->marked_area_end() - fileview_repp->marked_area_start(); 
    }
    else {
      start_pos_rep = fileview_repp->current_position();
      sel_length_rep = fileview_repp->length();
    }
    
    if (start_pos_rep > fileview_repp->length() ||
	start_pos_rep < 0) {
      start_pos_rep = 0;
    }
    if (sel_length_rep == 0 ||
	start_pos_rep + sel_length_rep > fileview_repp->length()) {
      sel_length_rep = fileview_repp->length() - start_pos_rep;
    }
  }

  // --------
  ENSURE(start_pos_rep >= 0);
  ENSURE(sel_length_rep >= 0);
  ENSURE(fileview_repp == 0 || start_pos_rep + sel_length_rep <= fileview_repp->length());
  // --------
}

void QESession::update_wave_view(void) {
  if (fileview_repp == 0) {
    fileview_repp = new QEFileView(active_filename_rep,
				   this, 
				   "fileview");
    init_layout();
  }
  else {
    if (fileview_repp->filename() != active_filename_rep ||
	fileview_repp->is_valid() != true) {
      fileview_repp->new_file(active_filename_rep);
      init_layout();
    }
  }
  fileview_repp->update_wave_form_data();
  fileview_repp->emit_status();
}

bool QESession::temp_files_created(void) const { return(temp_files_created_rep); }

/**
 * Make a copy of the current active file, and sets the 
 * newly created file active.
 */
void QESession::create_temps(void) {
  bool unexpected_error = false;
  string temp1 = tempfile_dir_rep.create_filename("workfile", ".wav");
  if (temp1.size() == 0) {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Error while creating temporary file ") + QString(temp1.c_str()),0);
  }
  else {
    temp_filenames_rep.push_back(temp1);
    string& temp2 = temp_filenames_rep.back();

    if (state_rep == state_no_file) {
      set_active_filename(temp2);
    }
    else {
      struct stat stattemp1;
      struct stat stattemp2;
      stat(active_filename_rep.c_str(), &stattemp1);
      stat(temp2.c_str(), &stattemp2);
      if (stattemp1.st_size != stattemp2.st_size) {
	QECopyEvent p (ectrl_repp, active_filename_rep, temp2, 0, 0);
	p.status_info("Creating temporary file for processing...");
	if (p.is_valid() == true) {
	  emit engine_status("running"); 
	  p.start(); 
	  emit engine_status("stopped");
	  if (ectrl_repp->engine_status() == "error") {
	    QMessageBox* mbox = new QMessageBox(this, "mbox");
	    mbox->information(this, "ecawave", QString("Error while creating temporary file ") + QString(temp2.c_str()),0);
	    unexpected_error = true;
	  }
	}
      }
      if (stattemp1.st_size != stattemp2.st_size) 
	copy_file(active_filename_rep + ".ews", temp2 + ".ews");
      stat(temp2.c_str(), &stattemp2);
      if (stattemp2.st_size == 0) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");
	mbox->information(this, "ecawave", QString("Error while creating temporary file ") + QString(temp2.c_str()),0);
	unexpected_error = true;
      }
      if (unexpected_error != true)
	set_active_filename(temp2);
    }
    if (unexpected_error != true)
      temp_files_created_rep = true;
  }

  // --------
  ENSURE(temp_files_created() == true || unexpected_error == true);
  ENSURE(active_filename_rep.empty() != true);
  // --------
}

string QESession::get_clipboard_location(void) {
  string cliploc = ecawaverc_rep.resource("clipboard-location");
  if (cliploc == "default") {
    cliploc = tempfile_dir_rep.get_reserved_directory() + "/clipboard.wav";
  }
  return(cliploc);
}

void QESession::change_state_to_modified(void) {
  if (direct_mode_rep != true &&
      temp_files_created() != true) create_temps();

  if (direct_mode_rep == true ||
      temp_files_created() == true) {
    state_rep = state_modified;
    emit state_change_modified();
  }
  
  // --------
  ENSURE(direct_mode_rep == true ||
	 (direct_mode_rep != true && active_filename_rep != orig_filename_rep) || 
	 temp_files_created() != true);
  ENSURE(state_rep == state_modified ||
	 temp_files_created() != true);
  // --------
}

void QESession::change_state_to_named(void) {
  state_rep = state_named;
  emit state_change_named();

  // --------
  ENSURE(state_rep == state_named);
  // --------
}

void QESession::change_state_to_no_file(void) {
  if (temp_files_created() == true) remove_temps();
  orig_filename_rep = "";
  set_active_filename("");
  emit filename_changed(active_filename_rep);
  state_rep = state_no_file;
  emit state_change_no_file();

  // --------
  ENSURE(state_rep == state_no_file);
  // --------
}

bool QESession::active_file_valid(void) {
  if (state_rep == state_no_file ||
      fileview_repp->is_valid() == false) {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Can't perform action. No active file or file in invalid state."));
    return(false);
  }
  return(true);
}

void QESession::play_event(void) { 
  if (active_file_valid() != true) return;
  stop_event();
  prepare_event();

  QEPlayEvent* p;
  p = new QEPlayEvent(ectrl_repp, active_filename_rep, ecawaverc_rep.resource("default-output"), start_pos_rep, sel_length_rep);

  if (p->is_valid() == true) {
    p->start();
    nb_event_repp = p;
  }
  else 
    nb_event_repp = 0;
}

void QESession::open_event(void) {
  QEOpenFileDialog* fdialog = new QEOpenFileDialog();
  if (fdialog->exec() == QEOpenFileDialog::Accepted) {
    stop_event();
    if (temp_files_created() == true) remove_temps();

    set_active_filename(fdialog->result_filename());
    set_orig_filename(active_filename_rep);
    change_state_to_named();
    if (fileview_repp == 0) {
      fileview_repp = new QEFileView(active_filename_rep,
				     this,
				     "fileview");
    }
    else {
      fileview_repp->new_file(active_filename_rep);
      fileview_repp->update_wave_form_data();
      fileview_repp->emit_status();
    }
    init_layout();

    ECA_AUDIO_FORMAT frm (fdialog->result_channels(), 
			  (long int)fdialog->result_srate(), 
			  ECA_AUDIO_FORMAT::sfmt_s16);
    if (fdialog->result_bits() == 8)
      frm.set_sample_format(ECA_AUDIO_FORMAT::sfmt_u8);
    
    fileview_repp->toggle_wave_cache(fdialog->result_wave_cache_toggle());
    fileview_repp->toggle_cache_refresh(fdialog->result_cache_refresh_toggle());
    direct_mode_rep = fdialog->result_direct_mode_toggle();
  }
}

void QESession::new_event(void) {
  change_state_to_no_file();
  init_layout();
}

void QESession::save_event(void) { 
  if (state_rep != state_modified) {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", "File not modified, save file cancelled.",0);
    return;
  }

  if (orig_filename_rep.empty() == true) save_as_event();
  else {
    if (active_file_valid() != true) return;
    stop_event();
    QESaveEvent p (ectrl_repp, active_filename_rep, orig_filename_rep);
    if (p.is_valid() == true) {
      emit engine_status("running"); 
      p.start(); 
      emit engine_status("stopped");
    }
  }
  change_state_to_named();
}

void QESession::save_as_event(void) { 
  QESaveFileDialog* fdialog = new QESaveFileDialog();
  if (fdialog->exec() == QESaveFileDialog::Accepted) {
    if (active_file_valid() != true) return;
    stop_event();
    
    QESaveEvent p (ectrl_repp, active_filename_rep, fdialog->result_filename());
    if (p.is_valid() == true) {
      emit engine_status("running"); 
      p.start(); 
      emit engine_status("stopped");
      set_orig_filename(fdialog->result_filename());
      update_wave_view();
      change_state_to_named();
    }
  }
}

void QESession::effect_event(void) {
  if (active_file_valid() != true) return;
  stop_event();
  prepare_event();

  /* make sure that active file exists and is in a valid state */
  change_state_to_modified();
  if (state_rep == state_modified) {
    QEChainopEvent* p = new QEChainopEvent(ectrl_repp, active_filename_rep, active_filename_rep,
					   start_pos_rep, sel_length_rep);
    
    QObject::connect(p, SIGNAL(finished()), this, SLOT(update_wave_view()));
    p->show();
    nb_event_repp = p;
  }
  else {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Can't perform action (ecawave couldn't create the necessary tempfiles)."));
  }
}

void QESession::copy_event(void) { 
  if (active_file_valid() != true) return;
  stop_event();
  prepare_event();
  if (state_rep == state_no_file ||
      fileview_repp->is_valid() == false) return;

  QECopyEvent p (ectrl_repp, active_filename_rep, get_clipboard_location(), start_pos_rep, sel_length_rep);
  
  if (p.is_valid() == true) {
    emit engine_status("running"); 
    p.start(); 
    emit engine_status("stopped");
  }
}

void QESession::paste_event(void) { 
  stop_event();
  prepare_event();

  /* make sure that active file exists and is in a valid state */
  change_state_to_modified();

  /* make sure that source and target files exists */
  string clipboard = get_clipboard_location();
  if (clipboard.empty() != true &&
      state_rep == state_modified) {
    QEPasteEvent p (ectrl_repp, 
		    clipboard,
		    active_filename_rep, 
		    start_pos_rep);
    if (p.is_valid() == true) {
      emit engine_status("running"); 
      p.start(); 
      emit engine_status("stopped");
      update_wave_view();
    }
  }
  else {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Can't perform action. Either the clipboard is empty or ecawave couldn't create the necessary tempfiles."));
  }
}

void QESession::cut_event(void) { 
  if (active_file_valid() != true) return;
  stop_event();
  prepare_event();

  /* make sure that active file exists and is in a valid state */
  change_state_to_modified();

  if (state_rep == state_modified) {
    QECutEvent p (ectrl_repp, 
		  active_filename_rep,
		  get_clipboard_location(),
		  start_pos_rep,
		  sel_length_rep);
    if (p.is_valid() == true) {
      emit engine_status("running"); 
      p.start(); 
      emit engine_status("stopped");
      fileview_repp->unmark();
      update_wave_view();
    }
  }
  else {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Can't perform action (ecawave couldn't create the necessary tempfiles)."));
  }
}

void QESession::fade_in_event(void) { 
  if (active_file_valid() != true) return;
  stop_event();
  prepare_event();

  /* make sure that active file exists and is in a valid state */
  change_state_to_modified();

  if (state_rep == state_modified) {
    QEFadeInEvent p (ectrl_repp, 
		     active_filename_rep,
		     active_filename_rep,
		     start_pos_rep,
		     sel_length_rep);
    if (p.is_valid() == true) {
      emit engine_status("running"); 
      p.start(); 
      emit engine_status("stopped");
      update_wave_view();
    }
  }
  else {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Can't perform action (ecawave couldn't create the necessary tempfiles)."));
  }
}

void QESession::fade_out_event(void) { 
  if (active_file_valid() != true) return;
  stop_event();
  prepare_event();

  /* make sure that active file exists and is in a valid state */
  change_state_to_modified();

  if (state_rep == state_modified) {
    QEFadeOutEvent p (ectrl_repp, 
		      active_filename_rep,
		      active_filename_rep,
		      start_pos_rep,
		      sel_length_rep);
    if (p.is_valid() == true) {
      emit engine_status("running"); 
      p.start(); 
      emit engine_status("stopped");
      update_wave_view();
    }
  }
  else {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Can't perform action (ecawave couldn't create the necessary tempfiles)."));
  }
}

void QESession::stop_event(void) { 
  if (nb_event_repp != 0) {
    if (ectrl_repp->is_running()) nb_event_repp->stop();
    emit engine_status("stopped");
    delete nb_event_repp;
  }
  nb_event_repp = 0;
}

void QESession::copy_file(const string& a, const string& b) {
  FILE *f1, *f2;
  f1 = fopen(a.c_str(), "r");
  f2 = fopen(b.c_str(), "w");
  char buffer[16384];

  if (!f1 || !f2) {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Error while creating temporary file ") + QString(b.c_str()),0);
    return;
  }

  fseek(f1, 0, SEEK_END);
  long int len = ftell(f1);
  fseek(f1, 0, SEEK_SET);

  QProgressDialog progress ("Creating temporary file for processing...", 0,
			    (int)(len / 1000), 0, 0, true);
  progress.setProgress(0);
  progress.show();

  while(!feof(f1) && !ferror(f2)) {
    fwrite(buffer, fread(buffer, 1, 16384, f1), 1, f2);
    progress.setProgress((int)(ftell(f1) / 1000));
  }
  fclose(f1);
  fclose(f2);
}
