// ------------------------------------------------------------------------
// qesession.cpp: Class representing an ecawave session
// Copyright (C) 2000 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cstdio>
#include <sys/stat.h>
#include <unistd.h>

#include <qapplication.h>
#include <qmainwindow.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qtimer.h>
#include <qmessagebox.h>
#include <qprogressdialog.h>
#include <qaccel.h>
#include <qvbox.h>
#include <qmenubar.h>
#include <qpopupmenu.h>

#include <ecasound/eca-session.h>
#include <ecasound/eca-control.h>

#include <qtecasound/qebuttonrow.h>

#include "qefile.h"
#include "qeopenfiledialog.h"
#include "qesavefiledialog.h"
#include "qesession.h"
#include "qestatusbar.h"

#include "qeevent.h"
#include "qenonblockingevent.h"
#include "qeplayevent.h"
#include "qesaveevent.h"
#include "qechainopevent.h"
#include "qecopyevent.h"
#include "qepasteevent.h"
#include "qecutevent.h"
#include "qefadeinevent.h"
#include "qefadeoutevent.h"

QESession::QESession (QMainWindow *parent,
		      const char *name)
  : QWidget(parent, name),
    state_rep(state_no_file),
    fileview_repp(0),
    audio_io_repp(0),
    buttonrow_repp(0),
    buttonrow2_repp(0),
    nb_event_repp(0),
    mainwindow_repp(parent),
    menubar_repp(parent->menuBar()),
    statusbar_repp(0),
    vlayout_repp(0),
    refresh_toggle_rep(false),
    wcache_toggle_rep(true),
    direct_mode_rep(false) {

  esession_repp = new ECA_SESSION();
  ectrl_repp = new ECA_CONTROL(esession_repp);

  auto_ptr<ECA_SESSION> p (esession_repp);
  auto_esession_rep = p;

  auto_ptr<ECA_CONTROL> q (ectrl_repp);
  auto_ectrl_rep = q;

  startTimer(100);
  QTimer *timer = new QTimer( this );
  connect( timer, SIGNAL(timeout()), this, SLOT(position_update()));
  timer->start(500, false);

  QAccel* a = new QAccel (this);
  a->connectItem(a->insertItem(ALT+CTRL+Key_D), this, SLOT(debug_event()));

  init_menubar();
  init_layout();

  // --------
  ENSURE(ectrl_repp != 0);
  ENSURE(esession_repp != 0);
  // --------
}

void QESession::open_file(const string& filename) {
  stop_event();
  remove_temps();

  active_filename_rep = orig_filename_rep = filename;
  if (orig_filename_rep.empty() == true) {
    active_filename_rep = string(tmpnam(NULL)) + ".wav";
    state_rep = state_new_file;
    emit state_change_new_file();
  }
  else {
    fileview_repp = new QEFileView(active_filename_rep,
				   this, 
				   "fileview");
    state_rep = state_orig_file;
    emit state_change_original_file();
  }

  if (direct_mode_rep == true) {
    state_rep = state_orig_direct;
    emit state_change_original_file_direct();
    statusbar_repp->toggle_editing(true);
  }

  start_pos_rep = 0;
  sel_length_rep = 0;

  init_layout();

  // --------
  ENSURE(active_filename_rep.empty() == false);
  // --------
}

void QESession::timerEvent( QTimerEvent * ) {
  if (nb_event_repp != 0) {
    if (nb_event_repp->is_finished() == true &&
	nb_event_repp->is_triggered() == true)
      nb_event_repp->stop();
  }
}

QESession::~QESession(void) {
  if (nb_event_repp != 0) {
    if (nb_event_repp->is_triggered() &&
  	ectrl_repp->is_running()) nb_event_repp->stop();
    delete nb_event_repp;
  }

  remove_temps();
}

void QESession::remove_temps(void) {
  if (state_rep == state_edit_file || 
      state_rep == state_new_file) {
    assert(active_filename_rep.empty() != true);
    remove(active_filename_rep.c_str());
    remove((active_filename_rep + ".ews").c_str());
  }
}

void QESession::position_update(void) {
  static bool toggle = false;
  long int pos = 0;
  if (ectrl_repp->is_running() == true) {
    toggle = true;
    if (nb_event_repp != 0)
      pos = nb_event_repp->position_in_samples();
    else
      pos = ectrl_repp->position_in_samples() - ectrl_repp->get_chainsetup()->buffersize();
  }
  if (toggle == true) {
    if (pos > 0) fileview_repp->current_position(pos);
    else fileview_repp->current_position(0);
  }
  if (ectrl_repp->is_running() != true) toggle = false;
}

void QESession::init_menubar(void) {

  QPopupMenu *file_menu = new QPopupMenu();
  file_menu->insertItem("New", this, SLOT(new_file()), CTRL+Key_F );
  file_menu->insertItem("Open", this, SLOT(open_file()), CTRL+Key_O );
  menubar_repp->insertItem("File", file_menu);

  menubar_repp->setSeparator(QMenuBar::InWindowsStyle);
//    menubar_repp->show();
}

/**
 * Initializes the widget layout. Depending on the current session 
 * state, different layouts are used. If init_layout() is called 
 * multiple times, the result should be same if the session state
 * remains the same between individual calls.
 */
void QESession::init_layout(void) {
  if (vlayout_repp != 0) {
    delete vlayout_repp;
  }
  vlayout_repp = new QVBoxLayout(this);

  if (buttonrow_repp == 0) {
    buttonrow_repp = new QEButtonRow(this, "buttonrow");
    buttonrow_repp->set_font(QFont("times", 12));
    buttonrow_repp->add_button(new QPushButton("(N)ew session",buttonrow_repp), 
			  CTRL+Key_N,
			  this, SLOT(new_session()));
    buttonrow_repp->add_button(new QPushButton("New (f)ile",buttonrow_repp), 
			  CTRL+Key_F, this, SLOT(new_file()));
    buttonrow_repp->add_button(new QPushButton("(O)pen",buttonrow_repp), 
			  CTRL+Key_O, this, SLOT(open_file()));
    buttonrow_repp->add_button(new QPushButton("Sa(v)e",buttonrow_repp), 
			  CTRL+Key_V, this, SLOT(save_event()));
    buttonrow_repp->add_button(new QPushButton("Save (a)s",buttonrow_repp), 
			  CTRL+Key_A, this, SLOT(save_as_event()));
    buttonrow_repp->add_button(new QPushButton("(Q)uit",buttonrow_repp), 
			  CTRL+Key_Q, this, SLOT(close_session()));

  }
  vlayout_repp->addWidget(buttonrow_repp);

  if (buttonrow2_repp == 0) {
    buttonrow2_repp = new QEButtonRow(this, "buttonrow2_repp");
    buttonrow2_repp->set_font(QFont("times", 12));
    buttonrow2_repp->add_button(new QPushButton("S(t)art",buttonrow2_repp), CTRL+Key_T,
				this, SLOT(play_event()));
    buttonrow2_repp->add_button(new QPushButton("(S)top",buttonrow2_repp), CTRL+Key_S,
				this, SLOT(stop_event()));
    buttonrow2_repp->add_button(new QPushButton("(E)ffect",buttonrow2_repp), CTRL+Key_E,
				this, SLOT(effect_event()));
    buttonrow2_repp->add_button(new QPushButton("Fade (i)n",buttonrow2_repp), CTRL+Key_I,
				this, SLOT(fade_in_event()));
    buttonrow2_repp->add_button(new QPushButton("Fa(d)e out",buttonrow2_repp), CTRL+Key_D,
				this, SLOT(fade_out_event()));
    buttonrow2_repp->add_button(new QPushButton("Cop(y)",buttonrow2_repp), CTRL+Key_Y,
				this, SLOT(copy_event()));
    buttonrow2_repp->add_button(new QPushButton("C(u)t",buttonrow2_repp), CTRL+Key_U,
				this, SLOT(cut_event()));
    buttonrow2_repp->add_button(new QPushButton("(P)aste",buttonrow2_repp), CTRL+Key_P,
				this, SLOT(paste_event()));
  }
  
  vlayout_repp->addWidget(buttonrow2_repp);

  if (state_rep != state_no_file &&
      state_rep != state_new_file) {
    QObject::connect(this, 
		     SIGNAL(filename_changed(const string&)), 
		     fileview_repp,
		     SLOT(title(const string&)));
    vlayout_repp->addWidget(fileview_repp, 1);
  }

  if (statusbar_repp == 0) {
    statusbar_repp = new QEStatusBar(mainwindow_repp, "statusbar");
  }

  if (state_rep != state_no_file &&
      state_rep != state_new_file) {
    statusbar_repp->visible_area(ECA_AUDIO_TIME(0, fileview_repp->samples_per_second()),
				 ECA_AUDIO_TIME(fileview_repp->length(), 
						fileview_repp->samples_per_second()));

    QObject::connect(fileview_repp, 
		     SIGNAL(visible_area_changed(ECA_AUDIO_TIME,
						 ECA_AUDIO_TIME)), 
		     statusbar_repp, 
		     SLOT(visible_area(ECA_AUDIO_TIME, ECA_AUDIO_TIME)));
    QObject::connect(fileview_repp, 
		     SIGNAL(marked_area_changed(ECA_AUDIO_TIME,
						ECA_AUDIO_TIME)), 
		     statusbar_repp, 
		     SLOT(marked_area(ECA_AUDIO_TIME, ECA_AUDIO_TIME)));
    QObject::connect(fileview_repp, 
		     SIGNAL(current_position_changed(ECA_AUDIO_TIME)), 
		     statusbar_repp,
		     SLOT(current_position(ECA_AUDIO_TIME)));
  }
    
}

void QESession::debug_event(void) {
  cerr << "----------------" << endl;
  cerr << "- ecawave-debug:" << endl;
  cerr << "orig_filename_rep: " << orig_filename_rep << endl;;
  cerr << "active_filename_rep: " << active_filename_rep << endl;;

  cerr << "start_pos_rep: " << start_pos_rep << endl;
  cerr << "sel_length_rep: " << sel_length_rep << endl;

  if (fileview_repp != 0) cerr << "qefile-filename: " <<
			    fileview_repp->filename() << endl;
  else cerr << "qefile not created." << endl;

  if (refresh_toggle_rep) cerr << "refresh_toggle_rep: true" << endl;
  else cerr << "refresh_toggle_rep: false" << endl;

  if (wcache_toggle_rep) cerr << "wcache_toggle_rep: true" << endl;
  else cerr << "wcache_toggle_rep: false" << endl;

  if (direct_mode_rep) cerr << "direct_mode_rep: true" << endl;
  else cerr << "direct_mode_rep: false" << endl;
 
  if (ectrl_repp == 0) cerr << "ectrl not created." << endl;
  else {
    cerr << "ectrl status: " << ectrl_repp->engine_status() << endl;
    cerr << "ectrl position: " << ectrl_repp->position_in_seconds_exact()
	 << "sec" << endl;
  }
}

void QESession::new_session(void) { emit new_session_request(); }
void QESession::close_session(void) { emit session_closed(); }

void QESession::new_file(void) {
  stop_event();
  remove_temps();
  int old_height_hint = 0;
  // FIXME: create a new file if no file exists
  if (fileview_repp == 0) return;

  old_height_hint = fileview_repp->sizeHint().height();
  fileview_repp->new_file();
  orig_filename_rep = "";
  active_filename_rep = string(tmpnam(NULL)) + ".wav";
  state_rep = state_new_file;
  emit state_change_new_file();
  emit filename_changed(active_filename_rep);
  resize(width(), height() + fileview_repp->sizeHint().height() - old_height_hint);
}


void QESession::open_file(void) {
  // FIXME: doesn't work at all

  QEOpenFileDialog* fdialog = new QEOpenFileDialog();
  if (fdialog->exec() == QEOpenFileDialog::Accepted) {
    stop_event();
    remove_temps();

    ECA_AUDIO_FORMAT frm (fdialog->result_channels(), 
			  (long int)fdialog->result_srate(), 
			  ECA_AUDIO_FORMAT::sfmt_s16);
    if (fdialog->result_bits() == 8)
      frm.set_sample_format(ECA_AUDIO_FORMAT::sfmt_u8);

    fileview_repp->toggle_wave_cache(fdialog->result_wave_cache_toggle());
    fileview_repp->toggle_cache_refresh(fdialog->result_cache_refresh_toggle());
    direct_mode_rep = fdialog->result_direct_mode_toggle();
    int old_height_hint = fileview_repp->sizeHint().height();
    remove_temps();
    fileview_repp->new_file(fdialog->result_filename());
    state_rep = state_orig_file;
    emit state_change_original_file();
    if (direct_mode_rep == true) {
      state_rep = state_orig_direct;
      emit state_change_original_file_direct();
      statusbar_repp->toggle_editing(true);
    }
    orig_filename_rep = fileview_repp->filename();
    active_filename_rep = orig_filename_rep;
    emit filename_changed(orig_filename_rep);
    resize(width(), height() + fileview_repp->sizeHint().height() - old_height_hint);
  }
}

void QESession::prepare_event(void) { 
  // --------
  REQUIRE(fileview_repp != 0);
  // --------

  start_pos_rep = 0;
  sel_length_rep = 0;
  if (fileview_repp->is_valid() == true) {
    if (fileview_repp->is_marked() == true) {
      start_pos_rep = fileview_repp->marked_area_start();
      sel_length_rep = fileview_repp->marked_area_end() - fileview_repp->marked_area_start(); 
    }
    else {
      start_pos_rep = fileview_repp->current_position();
      sel_length_rep = fileview_repp->length();
    }
    
    if (start_pos_rep > fileview_repp->length() ||
	start_pos_rep < 0) {
      start_pos_rep = 0;
    }
    if (sel_length_rep == 0 ||
	start_pos_rep + sel_length_rep > fileview_repp->length()) {
      sel_length_rep = fileview_repp->length() - start_pos_rep;
    }
  }
  else {
    state_rep = state_no_file;
    emit state_change_no_file();
  }

  // --------
  ENSURE(start_pos_rep >= 0);
  ENSURE(sel_length_rep >= 0);
  ENSURE(start_pos_rep + sel_length_rep <= fileview_repp->length());
  ENSURE((state_rep == state_no_file && fileview_repp->is_valid() != true) ||
	 (state_rep != state_no_file && fileview_repp->is_valid() == true));
  // --------
}

void QESession::update_wave_data(void) {
  // --------
  // require:
  assert(fileview_repp != 0);
  // --------

  if (fileview_repp != 0) {
    // FIXME: is the below code correct?
//      if (fileview_repp->filename() != active_filename_rep) {
    fileview_repp->new_file(active_filename_rep);
//        state_rep = state_new_file;
    //        emit state_change_new_file();
    fileview_repp->update_wave_form_data();
    fileview_repp->emit_status();
  }
  else {
    fileview_repp = new QEFileView(active_filename_rep,
				   this, 
				   "fileview");
  }
}

bool QESession::temp_file_created(void) {
  struct stat stattemp1;
  struct stat stattemp2;
  stat(active_filename_rep.c_str(), &stattemp1);
  stat(active_filename_rep.c_str(), &stattemp2);
  if (stattemp1.st_size != stattemp2.st_size) return(false);
  return(true);
}

void QESession::prepare_temp(void) {
  if (state_rep == state_orig_file) {
    string temp = string(tmpnam(NULL)) + ".wav";

    struct stat stattemp1;
    struct stat stattemp2;
    stat(active_filename_rep.c_str(), &stattemp1);
    stat(temp.c_str(), &stattemp2);
    if (stattemp1.st_size != stattemp2.st_size) {
      QECopyEvent p (ectrl_repp, active_filename_rep, temp, 0, 0);
      p.status_info("Creating temporary file for processing...");
      if (p.is_valid() == true) {
	p.start();
      }
    }
    if (stattemp1.st_size != stattemp2.st_size) 
      copy_file(active_filename_rep + ".ews", temp + ".ews");
    stat(temp.c_str(), &stattemp2);
    if (stattemp2.st_size == 0) {
      QMessageBox* mbox = new QMessageBox(this, "mbox");
      mbox->information(this, "ecawave", QString("Error while creating temporary file ") + QString(temp.c_str()),0);
      state_rep = state_no_file;
      emit state_change_no_file();
    }
    else {
      active_filename_rep = temp;
      int old_height_hint = fileview_repp->sizeHint().height();
      fileview_repp->new_file(active_filename_rep);
      state_rep = state_edit_file;
      emit state_change_editing_file();
      statusbar_repp->toggle_editing(true);
      resize(width(), height() + fileview_repp->sizeHint().height() - old_height_hint);
    }
  }
}

void QESession::play_event(void) { 
  stop_event();
  prepare_event();
  if (fileview_repp->is_valid() == false) return;

  QEPlayEvent* p;
  p = new QEPlayEvent(ectrl_repp, active_filename_rep, ecawaverc_rep.resource("default-output"), start_pos_rep, sel_length_rep);

  if (p->is_valid() == true) {
    p->start();
    nb_event_repp = p;
  }
  else 
    nb_event_repp = 0;
}

void QESession::save_event(void) { 
  if (state_rep == state_orig_file) {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", "File not modified, save file cancelled.",0);
    return;
  }

  if (state_rep == state_new_file) save_as_event();
  else {
    stop_event();
    QESaveEvent p (ectrl_repp, active_filename_rep, orig_filename_rep);
    if (p.is_valid() == true) p.start();
  }
}

void QESession::save_as_event(void) { 
  QESaveFileDialog* fdialog = new QESaveFileDialog();
  if (fdialog->exec() == QESaveFileDialog::Accepted) {
    stop_event();

    QESaveEvent p (ectrl_repp, active_filename_rep, fdialog->result_filename());
    if (p.is_valid() == true) {
      p.start();
    }
  }
}

void QESession::effect_event(void) {
  stop_event();
  prepare_event();
  if (fileview_repp->is_valid() == false) return;
  prepare_temp();
  if (state_rep == state_no_file) return;

  QEChainopEvent* p = new QEChainopEvent(ectrl_repp, active_filename_rep, active_filename_rep,
					 start_pos_rep, sel_length_rep);

  QObject::connect(p, SIGNAL(finished()), this, SLOT(update_wave_data()));
  p->show();
  nb_event_repp = p;
}

void QESession::copy_event(void) { 
  stop_event();
  prepare_event();
  if (fileview_repp->is_valid() == false) return;

  QECopyEvent p (ectrl_repp, active_filename_rep, ecawaverc_rep.resource("clipboard-file"), start_pos_rep, sel_length_rep);
  
  if (p.is_valid() == true) {
    p.start();
  }
}

void QESession::paste_event(void) { 
  stop_event();
  prepare_event();
  prepare_temp();
  if (state_rep == state_no_file) return;
  assert(state_rep != state_orig_file);

  QEPasteEvent p (ectrl_repp, 
		  ecawaverc_rep.resource("clipboard-file"),
		  active_filename_rep, 
		  start_pos_rep);
  if (p.is_valid() == true) {
    p.start();
    update_wave_data();
  }
}

void QESession::cut_event(void) { 
  stop_event();
  prepare_event();
  prepare_temp();
  if (state_rep == state_no_file) return;
  assert(state_rep != state_orig_file);

  QECutEvent p (ectrl_repp, 
		active_filename_rep,
		ecawaverc_rep.resource("clipboard-file"),
		start_pos_rep,
		sel_length_rep);
  if (p.is_valid() == true) {
    p.start();
    fileview_repp->unmark();
    update_wave_data();
  }
}

void QESession::fade_in_event(void) { 
  stop_event();
  prepare_event();
  prepare_temp();
  if (state_rep == state_no_file) return;
  assert(state_rep != state_orig_file);

  QEFadeInEvent p (ectrl_repp, 
		   active_filename_rep,
		   active_filename_rep,
		   start_pos_rep,
		   sel_length_rep);
  if (p.is_valid() == true) {
    p.start();
    update_wave_data();
  }
}

void QESession::fade_out_event(void) { 
  stop_event();
  prepare_event();
  prepare_temp();
  if (state_rep == state_no_file) return;
  assert(state_rep != state_orig_file);

  QEFadeOutEvent p (ectrl_repp, 
		   active_filename_rep,
		   active_filename_rep,
		   start_pos_rep,
		   sel_length_rep);
  if (p.is_valid() == true) {
    p.start();
    update_wave_data();
  }
}

void QESession::stop_event(void) { 
  if (nb_event_repp != 0) {
    if (ectrl_repp->is_running()) nb_event_repp->stop();
    delete nb_event_repp;
  }
  nb_event_repp = 0;
}

void QESession::copy_file(const string& a, const string& b) {
  FILE *f1, *f2;
  f1 = fopen(a.c_str(), "r");
  f2 = fopen(b.c_str(), "w");
  char buffer[16384];

  if (!f1 || !f2) {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecawave", QString("Error while creating temporary file ") + QString(b.c_str()),0);
    return;
  }

  fseek(f1, 0, SEEK_END);
  long int len = ftell(f1);
  fseek(f1, 0, SEEK_SET);

  QProgressDialog progress ("Creating temporary file for processing...", 0,
			    (int)(len / 1000), 0, 0, true);
  progress.setProgress(0);
  progress.show();

  while(!feof(f1) && !ferror(f2)) {
    fwrite(buffer, fread(buffer, 1, 16384, f1), 1, f2);
    progress.setProgress((int)(ftell(f1) / 1000));
  }
  fclose(f1);
  fclose(f2);
}
