// ------------------------------------------------------------------------
// audioio-raw.cpp: Raw/headerless audio file format input/output
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>
#include <string>
#include <cstring>
#include <cstdio>
#include <cassert>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-raw.h"

#include "eca-error.h"
#include "eca-debug.h"

RAWFILE::RAWFILE(const string& name, const SIMODE mode, const AIO_PARAMS& fmt, int bsize, bool double_buffering) 
  :  AUDIO_IO_DEVICE(name, mode, fmt, bsize) {
  double_buffering_rep = double_buffering;
  position_in_samples(0);
}

RAWFILE::~RAWFILE(void) {
  close_device();
}

void RAWFILE::open_device(void) { 
  switch(io_mode()) {
  case si_read:
    {
      if (label().at(0) == '-') {
	fio = new ECA_FILE_IO_STREAM();
	fio->open_stdin();
      }
      else {
	if (double_buffering_rep) fio = new ECA_FILE_IO_MMAP();
	else fio = new ECA_FILE_IO_STREAM();
	fio->open_file(label(),"rb");
      }
      break;
    }
  case si_write: 
    {
      fio = new ECA_FILE_IO_STREAM();
      if (label().at(0) == '-') {
	cerr << "(audioio-raw) Outputting to standard output.\n";
	fio->open_stdout();
      }
      else {
	fio->open_file(label(),"wb");
      }
      break;
    }
  case si_readwrite: 
    {
      fio = new ECA_FILE_IO_STREAM();
      if (label().at(0) == '-') {
	cerr << "(audioio-raw) Outputting to standard output.\n";
	fio->open_stdout();
      }
      else {
	fio->open_file(label(),"r+b", false);
	if (fio->file_mode() == "") {
	  fio->open_file(label(),"w+b", true);
	}
      }
    }
  }

  set_open_state(true);
  set_length_in_bytes();
  position_in_samples(position_in_samples(), true);
}

void RAWFILE::close_device(void) {
  fio->close_file();
  delete fio;
  set_open_state(false);
}

void RAWFILE::get_sample(SAMPLE_BUFFER* t) {
  t->reserve_buffer(buffersize());
  fio->read_to_buffer(t->iobuf_uchar, format().align * buffersize());

  if (!fio->is_file_error()) {
    t->iobuf_to_buffer(fio->file_bytes_processed() / format().align, 
		       SAMPLE_BUFFER::is_system_littleendian,
		       format().bits,
		       format().channels, format().srate);
    position_in_samples_advance(t->length_in_samples());
    if (!fio->is_file_ready()) finished(true);
  }
  else
    throw(new ECA_ERROR("AUDIOIO-RAW","Read error!"));
}

void RAWFILE::put_sample(SAMPLE_BUFFER* t) {
  t->buffer_to_iobuf(SAMPLE_BUFFER::is_system_littleendian, format().bits, format().channels, format().srate);
  position_in_samples_advance(t->length_in_samples());
  length_in_samples(position_in_samples());
  fio->write_from_buffer(t->iobuf_uchar, format().align * t->length_in_samples());
  if (!fio->is_file_ready()) finished(true);
}

void RAWFILE::set_length_in_bytes(void) {
  long int savetemp = fio->get_file_position();

  fio->set_file_position_end();
  length_in_samples(fio->get_file_position() / format().align);

  fio->set_file_position(savetemp);
}

void RAWFILE::position_in_samples_hook(void) {
  if (position_in_samples() <= length_in_samples()) finished(false);
  fio->set_file_position(position_in_samples() * format().align);
}
