// ------------------------------------------------------------------------
// audioio-wave.cpp: RIFF WAVE audio file input/output.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <cstring>
#include <cmath>

#include <kvutils.h>

#include "samplebuffer.h"
#include "audioio-wave_impl.h"
#include "audioio-wave.h"

#include "eca-error.h"
#include "eca-debug.h"

WAVEFILE::WAVEFILE (const string& name, const SIMODE mode, const AIO_PARAMS& fmt, int bsize) 
  :  AUDIO_IO_DEVICE(name, mode, fmt, bsize)
{
  switch(mode) {
  case si_read:
    {
      fobject=fopen(name.c_str(),"rb");
      if (!fobject) {
	//      cerr << "(audioio-wave) Going to throw an exception.\n";
	throw(new ECA_ERROR("AUDIOIO-WAVE", "unable to open file " + name + " in mode rb.", ECA_ERROR::retry));
      }
      read_riff_header();
      read_riff_fmt();     // also sets format()
      set_length_in_bytes();
      find_riff_datablock();
      break;
    }
  case si_write:
    {
      fobject=fopen(name.c_str(),"w+b");
      if (!fobject) {
	throw(new ECA_ERROR("AUDIOIO-WAVE", "unable to open file " +
			    name + " in mode w+b .", ECA_ERROR::retry));
      }
      length_in_bytes_value = 0;
      write_riff_header();
      write_riff_fmt();
      write_riff_datablock();
      break;
    }

  case si_readwrite:
    {
      fobject=fopen(name.c_str(),"r+b");
      if (fobject) {
	set_length_in_bytes();
	read_riff_fmt();     // also sets format()
	find_riff_datablock();
      }
      else {
	fobject=fopen(name.c_str(),"w+b");
	if (!fobject) {
	  throw(new ECA_ERROR("AUDIOIO-WAVE", "unable to open file " +
			      name + " in mode w+b .", ECA_ERROR::retry));
	}
	write_riff_header();
	write_riff_fmt();
	write_riff_datablock();
      }
      length_in_bytes_value = 0;
    }
  }
  
  position_in_samples(0);
}

WAVEFILE::~WAVEFILE(void) {
  ecadebug->msg(1,"(file-io) Closing file " + label());
  update();
  fclose(fobject);
}

void WAVEFILE::update (void) {
  update_riff_datablock();
  write_riff_header();
  set_length_in_bytes();
}

void WAVEFILE::find_riff_datablock (void) {
  if (find_block("data")==-1) {
    throw(new ECA_ERROR("AUDIOIO-WAVE", "no RIFF data block found", ECA_ERROR::retry));
  }
  fseek_start_position = ftell(fobject);
}

void WAVEFILE::read_riff_header (void) {
  ecadebug->msg(5, "(program flow: read_riff_header())");
   
  long save = ftell(fobject);

  fseek(fobject,0,SEEK_END);
  riff_header.size=ftell(fobject);             
  fseek(fobject,0,SEEK_SET);

  fread(&riff_header,1,sizeof(riff_header),fobject);
  if (strncmp("RIFF",riff_header.id,4)!=0 || strncmp("WAVE",riff_header.wname,4)!=0) {
    throw(new ECA_ERROR("AUDIOIO-WAVE", "invalid RIFF-header", ECA_ERROR::stop));
  }

  fseek(fobject,save,SEEK_SET);
}

void WAVEFILE::write_riff_header (void) {
  ecadebug->msg(5, "(program flow: write_riff_header())");
    
  long save = ftell(fobject);

  strncpy(riff_header.id,"RIFF",4);
  strncpy(riff_header.wname,"WAVE",4);

  fseek(fobject,0,SEEK_END);
  riff_header.size=ftell(fobject);             
  fseek(fobject,0,SEEK_SET);

  fwrite(&riff_header,1,sizeof(riff_header),fobject);
  if (strncmp("RIFF",riff_header.id,4)!=0 || strncmp("WAVE",riff_header.wname,4)!=0)
    throw(new ECA_ERROR("AUDIOIO-WAVE", "invalid RIFF-header", ECA_ERROR::stop));

  //    ecadebug->msg(1, "Riff ID",string(riff_header.id));
  //    ecadebug->msg(1, "Wave data size",riff_header.size);
  //    ecadebug->msg(1, "Riff type",riff_header.wname);

  fseek(fobject,save,SEEK_SET);
}

void WAVEFILE::read_riff_fmt(void)
{
  ecadebug->msg(5, "(program flow: read_riff_fmt())");
    
  long save = ftell(fobject);

  if (find_block("fmt ")==-1)
    throw(new ECA_ERROR("AUDIOIO-WAVE", "no riff fmt-block found",  ECA_ERROR::stop));
  else {
    fread(&riff_format,1,sizeof(riff_format),fobject);

    if (riff_format.format != 1) {
      throw(new ECA_ERROR("AUDIOIO-WAVE", "Only WAVE_FORMAT_PCM is supported."));
      //      ecadebug->msg("(audioio-wave) WARNING: wave-format not '1'.");
    }

    AIO_PARAMS temp;
    temp.channels = riff_format.channels;
    temp.bits = riff_format.bits;
    temp.srate = riff_format.srate;
    format(temp);
  }

  fseek(fobject,save,SEEK_SET);
}

void WAVEFILE::write_riff_fmt(void)
{
  RB fblock;

  fseek(fobject,0,SEEK_END);

  riff_format.channels = format().channels;
  riff_format.bits = format().bits;
  riff_format.srate = format().srate;
  riff_format.byte_second = format().byte_second;
  riff_format.align = format().align;
  riff_format.format = 1;     // WAVE_FORMAT_PCM (0x0001) Microsoft Pulse Code
                              //                          Modulation (PCM) format

  strncpy(fblock.sig,"fmt ",4);
  fblock.bsize=16;
  fwrite(&fblock,1,sizeof(fblock),fobject);

  fwrite(&riff_format,1,sizeof(riff_format),fobject);
  ecadebug->msg(5, "Wrote RIFF format header.");
}


void WAVEFILE::write_riff_datablock(void) {
  RB fblock;

  ecadebug->msg(5, "(program flow: write_riff_datablock())");
    
  fseek(fobject,0,SEEK_END);

  strncpy(fblock.sig,"data",4);
  fblock.bsize=(0);
  fwrite(&fblock,1,sizeof(fblock),fobject);
  fseek_start_position = ftell(fobject);   
}

void WAVEFILE::update_riff_datablock(void) {
  RB fblock;
  ecadebug->msg(5, "(program flow: update_riff_datablock())");
    
  strncpy(fblock.sig,"data",4);

  find_block("data");
  long save = ftell(fobject);
  fseek(fobject,0,SEEK_END);
  fblock.bsize = ftell(fobject) - save;

  save = save - sizeof(fblock);
  if (save > 0) {
    fseek(fobject,save,SEEK_SET);
    fwrite(&fblock,1,sizeof(fblock),fobject);
  }
}

int WAVEFILE::next_riff_block(RB *t, unsigned long int *offtmp)
{
  ecadebug->msg(5, "(program flow: next_riff_block())");

  if (fread(t,1,sizeof(RB),fobject) != sizeof(RB)) {
    ecadebug->msg(2, "invalid RIFF block!");
    return(1);
  }
    
  //    for(int n = 0; n < 4; n++) printf("%c", t->sig[n]);
                                      
  if (feof(fobject) || ferror(fobject)) return(1);
  *offtmp=t->bsize+ftell(fobject);
  return (0);
}

signed long int WAVEFILE::find_block(char *fblock) {
  char *sig;
  unsigned long int offset;
  int ret=0;
  RB *block;

  ecadebug->msg(5, "(program flow: find_block())");
    
  sig = new char[10];
  block = new RB;

  fseek(fobject,sizeof(riff_header),SEEK_SET);
  do {
    ret = next_riff_block(block,&offset);
    //        if (ret == -1) break;
    if (ret != 0) break;
    ecadebug->msg(5, "AUDIOIO-WAVE: found RIFF-block ");
    if (strncmp(block->sig,fblock,4) == 0) break;
    fseek(fobject,offset,SEEK_SET);
  }
  while (ret == 0);
  offset = block->bsize;

  delete[] sig;
  delete block;

  return(offset);
}

void WAVEFILE::get_sample(SAMPLE_BUFFER* t) {
  t->reserve_buffer(buffersize());
  samples_read = fread(t->iobuf_uchar, format().align, buffersize(), fobject);
  if (!ferror(fobject)) {
    t->iobuf_to_buffer(samples_read, true, format().bits, format().channels, format().srate);
    //    cerr << "|setting buffersize to " << samples_read << ".";
    curpos_value += t->length_in_samples() * format().align;
    if (feof(fobject)) finished(true);
  }
  else {
    //    cerr << "!read error!";
    throw(new ECA_ERROR("AUDIOIO-WAVE","Read error!"));
  }
}

void WAVEFILE::put_sample(SAMPLE_BUFFER* t) {
  t->buffer_to_iobuf(true, format().bits, format().channels, format().srate);
  length_in_bytes_value = curpos_value = curpos_value + (t->length_in_samples() * format().align);
  fwrite(t->iobuf_uchar, format().align, t->length_in_samples(), fobject);
  if (ferror(fobject)) finished(true);
}

void WAVEFILE::set_length_in_bytes(void) {
  long save = ftell(fobject);
  long t = find_block("data");
  fseek(fobject,save,SEEK_SET);
  length_in_bytes_value = t;
  MESSAGE_ITEM mitem;
  mitem << "(audioio-wave) data length " << t << "bytes.";
}

long WAVEFILE::length_in_samples(void) const {
  //    get_length_in_bytes();
  return((long)ceil((double)length_in_bytes_value / (double)format().align));
}

long WAVEFILE::position_in_samples(void) const { return(curpos_value / format().align); }

void WAVEFILE::position_in_samples(long pos) {
  if (pos <= length_in_samples()) finished(false);
  if (pos < 0) pos = 0;
  
  curpos_value = pos * format().align;
  fseek(fobject, fseek_start_position + curpos_value, SEEK_SET);
}







