// ------------------------------------------------------------------------
// audioio-alsalb.cpp: ALSA (/dev/snd/pcm*) loopback input.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>
#ifdef COMPILE_ALSA

#include <string>
#include <cstring>
#include <cstdio>
#include <dlfcn.h>  

#include <kvutils.h>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-alsalb.h"

#include "eca-error.h"
#include "eca-debug.h"

static const char* (*dl_snd_strerror)( int errnum );
static int (*dl_snd_pcm_loopback_open)(snd_pcm_loopback_t **handle, int card, int device, int mode);
static int (*dl_snd_pcm_loopback_close)(snd_pcm_loopback_t *handle);
// static int (*dl_snd_pcm_loopback_file_descriptor)(snd_pcm_loopback_t *handle);
static int (*dl_snd_pcm_loopback_block_mode)(snd_pcm_loopback_t *handle, int enable);
static int (*dl_snd_pcm_loopback_stream_mode)(snd_pcm_loopback_t *handle, int mode);
static int (*dl_snd_pcm_loopback_format)(snd_pcm_loopback_t *handle, snd_pcm_format_t * format);
static ssize_t (*dl_snd_pcm_loopback_read)(snd_pcm_loopback_t *handle, void *buffer, size_t size);

static int eca_alsalbdevice_dynlib_count = 0;
static void *eca_alsalbdevice_dynlib_handle;

ALSALBDEVICE::ALSALBDEVICE (int card, int device, const SIMODE
			mode, const AIO_PARAMS& form, int bsize,
			bool playback_mode)
  :  AUDIO_IO_DEVICE(string("alsa_loopback,") + kvu_numtostr(card) +
		     string(",") + kvu_numtostr(device) , mode, form, bsize)
{
  card_number = card;
  device_number = device;
  finished(false);
  pb_mode = playback_mode;

  if (eca_alsalbdevice_dynlib_count == 0) load_dynlibs();
  eca_alsalbdevice_dynlib_count++;

  open_device();

  is_triggered = false;
}

void ALSALBDEVICE::load_dynlibs(void) {
  ecadebug->msg("(audioio-alsalb) Loading libasound shared library.");
  
  eca_alsalbdevice_dynlib_handle = dlopen ("libasound.so", RTLD_LAZY);
  if (!eca_alsalbdevice_dynlib_handle) {
    throw(new ECA_ERROR("AUDIOIO-ALSALB", "Unable to load asound.so dynamic library."));
  }

  dl_snd_pcm_loopback_open = 
    (int (*)(snd_pcm_loopback_t **, int, int, int))dlsym(eca_alsalbdevice_dynlib_handle, "snd_pcm_loopback_open");
  dl_snd_pcm_loopback_close = 
    (int (*)(snd_pcm_loopback_t *))dlsym(eca_alsalbdevice_dynlib_handle, "snd_pcm_loopback_close");
  dl_snd_pcm_loopback_block_mode =
    (int (*)(snd_pcm_loopback_t *, int))dlsym(eca_alsalbdevice_dynlib_handle, "snd_pcm_loopback_block_mode");
  dl_snd_pcm_loopback_stream_mode = 
    (int (*)(snd_pcm_loopback_t *, int))dlsym(eca_alsalbdevice_dynlib_handle, "snd_pcm_loopback_stream_mode");
  dl_snd_pcm_loopback_format =
    (int (*)(snd_pcm_loopback_t *handle, snd_pcm_format_t *))dlsym(eca_alsalbdevice_dynlib_handle, "snd_pcm_loopback_format");

  dl_snd_pcm_loopback_read = 
    (int (*)(snd_pcm_loopback_t *, void *, size_t))dlsym(eca_alsalbdevice_dynlib_handle, "snd_pcm_loopback_read");

  dl_snd_strerror = (const char* (*)(int))dlsym(eca_alsalbdevice_dynlib_handle, "snd_strerror");

  if (dlerror() != NULL) {
    throw(new ECA_ERROR("AUDIOIO-ALSALB", "Error while loading asound.so dynamic library."));
  }
}

void ALSALBDEVICE::open_device(void) {
  if (is_open == true) return;
  int err;
  if (io_mode() == si_read) {
    if (pb_mode) {
      if ((err = dl_snd_pcm_loopback_open(&audio_fd, card_number, device_number,
					  SND_PCM_LB_OPEN_PLAYBACK)) < 0) {
	throw(new ECA_ERROR("AUDIOIO-ALSALB", "unable to open ALSA-device for reading; error: " + string(dl_snd_strerror(err))));
      }
    }
    else {
      if ((err = dl_snd_pcm_loopback_open(&audio_fd, card_number, device_number,
					  SND_PCM_LB_OPEN_CAPTURE)) < 0) {
	throw(new ECA_ERROR("AUDIOIO-ALSALB", "unable to open ALSA-device for reading; error: " + string(dl_snd_strerror(err))));
      }
    }
  }    
  else {
      throw(new ECA_ERROR("AUDIOIO-ALSALB", "Only readinng support with a loopback device."));
  }
  
  // ---
  // Set blocking mode.
  // ---
  dl_snd_pcm_loopback_block_mode(audio_fd, 1);    // enable block mode

  // ---
  // Set fragment size.
  // ---
  if (buffersize() == 0) 
    throw(new ECA_ERROR("AUDIOIO-ALSALB", "Buffersize() is 0!", ECA_ERROR::stop));
  
  // ---
  // Select audio format
  // ---

  snd_pcm_format_t pf;
  memset(&pf, 0, sizeof(pf));
  if (format().bits == 16)
    pf.format = SND_PCM_SFMT_S16_LE; // DL_SND_PCM_FMT_U16_LE
  else 
    pf.format = SND_PCM_SFMT_U8;

  pf.rate = format().srate;
  pf.channels = format().channels; // Stereo

  if (io_mode() == si_read) {
    err = dl_snd_pcm_loopback_format(audio_fd, &pf);
    if (err < 0) {
    throw(new ECA_ERROR("AUDIOIO-ALSALB", "Error when setting up record parameters: " + string(dl_snd_strerror(err))));
    }
    //    AIO_PARAMS temp;
    //    if (pf.format = SND_PCM_SFMT_S16_LE)
    //      temp.bits = 16;
    //    else 
    //      temp.bits = 8;
    //
    //    temp.srate = pf.rate;
    //    temp.channels = pf.channels;
    //    format(temp);
  }
  is_open = true;
}

void ALSALBDEVICE::rt_stop(void) { }
void ALSALBDEVICE::rt_ready(void) { }
void ALSALBDEVICE::rt_activate(void) { }

void ALSALBDEVICE::close_device(void) {
  if (is_open) {
    dl_snd_pcm_loopback_close(audio_fd);
  }    
  is_open = false;
}

void ALSALBDEVICE::get_sample(SAMPLE_BUFFER* t) {
  t->reserve_buffer(buffersize());
  if (!is_open) throw(new ECA_ERROR("AUDIOIO-ALSALB","get_sample(): trying to read from a closed device!"));

  bytes_read = dl_snd_pcm_loopback_read(audio_fd, t->iobuf_uchar, format().align * buffersize());

  if (bytes_read < 0)
    throw(new ECA_ERROR("AUDIOIO-ALSALB","get_sample(): read error! - " + string((*dl_snd_strerror)(bytes_read)), ECA_ERROR::stop));

  //    t->length_in_samples(samples_read / format().align);
  //  else 

  t->iobuf_to_buffer(bytes_read / format().align, true, format().bits, format().channels, format().srate);
}

ALSALBDEVICE::~ALSALBDEVICE(void) { 
  close_device();
  //  eca_alsalbdevice_dynlib_count--;
  if (eca_alsalbdevice_dynlib_count == 0) {
    ecadebug->msg("(audioio-alsalb) Closing libasound shared library.");
    dlclose(eca_alsalbdevice_dynlib_handle);
  }
}

#endif // COMPILE_ALSA

