// ------------------------------------------------------------------------
// audiofx_timebased.cpp: Routines for time-based effects.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>

#include <kvutils.h>

#include "eca-debug.h"

#include "samplebuffer.h"
#include "audiofx_timebased.h"

EFFECT_DELAY::EFFECT_DELAY (DYNAMIC_PARAMETERS::parameter_type delay_time, int surround_mode, 
			    int num_of_delays, DYNAMIC_PARAMETERS::parameter_type mix_percent) 
{
  // map_parameters();

  laskuri = 0.0;

  set_parameter(1, delay_time);
  set_parameter(2, surround_mode);
  set_parameter(3, num_of_delays);
  set_parameter(4, mix_percent);
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_DELAY::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(dtime / (DYNAMIC_PARAMETERS::parameter_type)SAMPLE_BUFFER::sample_rate * 1000.0);
  case 2:
    return(surround);
  case 3:
    return(dnum);
  case 4:
    return(mix * 100.0);
  }
  return(0.0);
}

void EFFECT_DELAY::set_parameter(int param, DYNAMIC_PARAMETERS::parameter_type value) {
  switch (param) {
  case 1: 
    {
      dtime = value * (DYNAMIC_PARAMETERS::parameter_type)SAMPLE_BUFFER::sample_rate / 1000;
      vector<SINGLE_BUFFER>::iterator p = buffer.begin();
      while(p != buffer.end()) {
	p->resize(dtime);
	++p;
      }
    }
    break;

  case 2: 
    surround = value;
    break;

  case 3: 
    if (value != 0.0) dnum = value;
    else dnum = 1.0;
    buffer.resize(dnum);
    set_parameter(1, get_parameter(1));
    break;

  case 4: 
    mix = value / 100.0;
    break;
  }
}

void EFFECT_DELAY::process(SAMPLE_BUFFER::sample_type *insample) {
  temp2.make_silent();

  for(nm2 = 0; nm2 < dnum; nm2++) {
    if (laskuri >= dtime * (nm2 + 1)) {
      temp = buffer[nm2].front();
      switch ((int)surround) {
      case 1: 
	{
	// surround
	temp.swap_channels();
	break;
	}
      case 2: 
	{
	if (nm2 % 2 == 0) {
	  temp.put_left(( temp.get_left() + temp.get_right()) / 2.0);
	  temp.put_right( 0.0 );
	}
	else {
	  temp.put_left( 0.0 );
	  temp.put_right(( temp.get_left() + temp.get_right()) / 2.0);
	}
	break;
	}
      } // switch
      buffer[nm2].pop_front();
    }
    buffer[nm2].push_back(*insample);
    
    temp2.put_left( temp2.get_left() + (temp.get_left() / dnum));
    temp2.put_right( temp2.get_right() + (temp.get_right() / dnum));
  }
  laskuri++;

  insample->mix(temp2, mix);
}

EFFECT_FAKE_STEREO::EFFECT_FAKE_STEREO (DYNAMIC_PARAMETERS::parameter_type delay_time) {
  laskuri = 0;
  // map_parameters();

  set_parameter(1, delay_time);
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_FAKE_STEREO::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(dtime / (DYNAMIC_PARAMETERS::parameter_type)SAMPLE_BUFFER::sample_rate * 1000.0);
  }
  return(0.0);
}

void EFFECT_FAKE_STEREO::set_parameter(int param, DYNAMIC_PARAMETERS::parameter_type value) {
  switch (param) {
  case 1: 
    dtime = value * (DYNAMIC_PARAMETERS::parameter_type)SAMPLE_BUFFER::sample_rate / 1000;
    buffer.resize(dtime);
    break;
  }
}

void EFFECT_FAKE_STEREO::process(SAMPLE_BUFFER::sample_type *insample) {

  if (laskuri >= dtime) {
    temp = buffer.front();
    temp.put_left((insample->get_left() + insample->get_right()) / 2.0);
    temp.put_right((temp.get_left() + temp.get_right()) / 2.0);
    buffer.pop_front();
  }
  else {
    temp.put_left(insample->get_left() + insample->get_right() / 2.0);        
    temp.put_right(0.0);
    laskuri++;
  }
  buffer.push_back(*insample);
  *insample = temp;
}

EFFECT_REVERB::EFFECT_REVERB (DYNAMIC_PARAMETERS::parameter_type delay_time, int surround_mode, 
			      DYNAMIC_PARAMETERS::parameter_type feedback_percent) 
{
  laskuri = 0.0;
  // map_parameters();

  set_parameter(1, delay_time);
  set_parameter(2, surround_mode);
  set_parameter(3, feedback_percent);
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_REVERB::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(dtime / (DYNAMIC_PARAMETERS::parameter_type)SAMPLE_BUFFER::sample_rate * 1000.0);
  case 2:
    return(surround);
  case 3:
    return(feedback * 100.0);
  }
  return(0.0);
}

void EFFECT_REVERB::set_parameter(int param, DYNAMIC_PARAMETERS::parameter_type value) {
  switch (param) {
  case 1: 
    dtime = value * (DYNAMIC_PARAMETERS::parameter_type)SAMPLE_BUFFER::sample_rate / 1000;
    buffer.resize(dtime);
    break;

  case 2: 
    surround = value;
    break;

  case 3: 
    feedback = value / 100.0;
    break;
  }
}

void EFFECT_REVERB::process(SAMPLE_BUFFER::sample_type *insample) {
  if (laskuri >= dtime) {
    temp = buffer.front();
    
    if (surround == 0) {
      insample->mix(temp, feedback);
    }
    else {
      temp.swap_channels();
      insample->mix(temp, feedback);
    }
    buffer.pop_front();
  }
  else {
    insample->put_left( insample->get_left() * (1.0 - feedback) );
    insample->put_right( insample->get_right() * (1.0 - feedback) );
    laskuri++;
  }
  buffer.push_back(*insample);
}

