#ifndef _AUDIOFX_AMPLITUDE_H
#define _AUDIOFX_AMPLITUDE_H

#include <vector>

#include "samplebuffer.h"
#include "audiofx.h"

/**
 * Virtual base for amplitude effects and dynamic processors.
 * @author Kai Vehmanen
 */
class EFFECT_AMPLITUDE : public EFFECT_BASE {

 public:
  virtual ~EFFECT_AMPLITUDE(void) { }
};

#include "audiofx_compressor.h"

/**
 * Normal amplifier
 * @author Kai Vehmanen
 */
class EFFECT_AMPLIFY: public EFFECT_AMPLITUDE {

  parameter_type kerroin;

 public:

  string name(void) { return("Amplify"); }

  vector<string> parameter_names(void) const  {
    vector<string> t;
    t.push_back("amp-%");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER* sbuf);

  EFFECT_AMPLIFY (parameter_type multiplier_percent = 100.0);
  EFFECT_AMPLIFY* clone(void)  { return new EFFECT_AMPLIFY(*this); }
};

/**
 * Amplifier with clip control.
 * @author Kai Vehmanen
 */
class EFFECT_AMPLIFY_CLIPCOUNT : public EFFECT_AMPLITUDE {

  parameter_type kerroin;
  int nm, num_of_clipped, maxnum_of_clipped;

 public:

  string name(void) { return("Amplify with clipping control"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("amp-%");
    t.push_back("max-clipped");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER* sbuf);

  EFFECT_AMPLIFY_CLIPCOUNT* clone(void)  { return new EFFECT_AMPLIFY_CLIPCOUNT(*this); }
  EFFECT_AMPLIFY_CLIPCOUNT (parameter_type multiplier_percent = 100.0, int max_clipped = 0);
};

/**
 * Channel amplifier
 * @author Kai Vehmanen
 */
class EFFECT_AMPLIFY_CHANNEL: public EFFECT_AMPLITUDE {

  parameter_type kerroin;
  int channel_rep;

 public:

  string name(void) { return("Channel amplify"); }

  vector<string> parameter_names(void) const  {
    vector<string> t;
    t.push_back("amp-%");
    t.push_back("channel");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_AMPLIFY_CHANNEL* clone(void)  { return new EFFECT_AMPLIFY_CHANNEL(*this); }
  EFFECT_AMPLIFY_CHANNEL (parameter_type multiplier_percent = 100.0, int channel = 0);
};

/**
 * Dynamic compressor.
 * @author Kai Vehmanen
 */
class EFFECT_COMPRESS : public EFFECT_AMPLITUDE {

  parameter_type crate;
  parameter_type threshold;

  unsigned short int nm;  // temp value (channel)
  parameter_type delta, new_value;
  bool first_time;
  SAMPLE_BUFFER::single_type s[SAMPLE_BUFFER::ch_count];
  SAMPLE_BUFFER::sample_type temp;     // temporary values
  SAMPLE_BUFFER::sample_type lastin, lastout;

 public:

  string name(void) { return("Compress"); }

  vector<string> parameter_names(void) const  {
    vector<string> t;
    t.push_back("compression_rate");
    t.push_back("threshold");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_COMPRESS* clone(void)  { return new EFFECT_COMPRESS(*this); }
  EFFECT_COMPRESS (const EFFECT_COMPRESS& x);
  EFFECT_COMPRESS (parameter_type compress_rate, parameter_type thold);
  EFFECT_COMPRESS (void) : first_time(true) { 
    map_parameters();
  }
};

/**
 * Noise gate. A mono-summed is used to control the gate.
 * @author Kai Vehmanen
 */
class EFFECT_NOISEGATE_MONO : public EFFECT_AMPLITUDE {

  parameter_type th_level;
  parameter_type th_time;
  parameter_type atime, htime, rtime;
  
  parameter_type th_time_lask;
  parameter_type attack_lask;
  parameter_type hold_lask;
  parameter_type release_lask;
  
  SAMPLE_BUFFER::sample_type temp_sample;
  
  parameter_type kerroin;
  
  unsigned short int nm;
  
  enum { ng_waiting, ng_attacking, ng_active, ng_holding, ng_releasing } ng_status;
  
 public:
  
  string name(void) { return("Noisegate mono"); }

  vector<string> parameter_names(void) const  {
    vector<string> t;
    t.push_back("threshold_level_%");
    t.push_back("th_time_msec");
    t.push_back("attack_msec");
    t.push_back("hold_msec");
    t.push_back("release_msec");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;
  
  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_NOISEGATE_MONO* clone(void)  { return new EFFECT_NOISEGATE_MONO(*this); }
  EFFECT_NOISEGATE_MONO (parameter_type thlevel_percent, parameter_type thtime, parameter_type atime, parameter_type htime, parameter_type rtime);
  EFFECT_NOISEGATE_MONO (void) { 
    ng_status = ng_waiting;  
    map_parameters();
  }
};

/**
 * Panning effect for controlling the stereo image.
 * @author Kai Vehmanen
 */
class EFFECT_NORMAL_PAN : public EFFECT_AMPLITUDE {

private:

  parameter_type right_percent_rep;
  parameter_type l_kerroin, r_kerroin;
  
public:

  string name(void) { return("Normal pan"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("right-%");
    return(t);
  }
    
  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);
    
  EFFECT_NORMAL_PAN* clone(void)  { return new EFFECT_NORMAL_PAN(*this); }
  EFFECT_NORMAL_PAN(parameter_type right_percent = 50.0);
};

#endif
