#include <vector>
#include <string>

#include <kvutils.h>

#include "osc-gen.h"
#include "oscillator.h"
#include "eca-debug.h"
#include "eca-error.h"

double GENERIC_OSCILLATOR::value(void) {
  if (ienvelope.size() == 0) curval = 0.0;

  if (linear) {
    if (current + 1 < ienvelope.size()) { 
      curval = pcounter / pdistance * ienvelope[current + 1];
      curval += (1.0 - pcounter / pdistance) * ienvelope[current];
    }
  }
  else {
    curval = ienvelope[current];
  }

  pcounter += phase_step();
  if (pcounter > pdistance) {
    pcounter -= pdistance;
    ++current;
  }

  if (linear) {
    if (current + 1 == ienvelope.size()) current = 0;
  }
  else {
    if (current == ienvelope.size()) current = 0;
  }

  //  cerr << "(gen-osc) new value: " << curval << ".\n";
  return(curval);
}

GENERIC_OSCILLATOR::GENERIC_OSCILLATOR(double step, double freq, int
				       preset_number, const string& filename) :
OSCILLATOR(step, freq) 
{
  preset_found = false;
  linear = false;
  read_envelope(filename, preset_number);
  preset = preset_number;

  if (preset_found == false) {
    throw(new ECA_ERROR("OSC-GEN", "Preset " +
			kvu_numtostr(preset_number) + " not found from envelope file " + 
			filename + "."));
  }

  L = 1.0 / frequency();   // length of one wave in seconds

  if (linear) {
    if (ienvelope.size() > 1) pdistance = L / (ienvelope.size() - 1);
    else pdistance = L;
  }
  else {
    if (ienvelope.size() > 0) pdistance = L / ienvelope.size();
  }
  pcounter = 0.0;

  MESSAGE_ITEM m;
  m << "(osc-gen) Generic oscillator created using envelope preset number " << preset_number << ".";
  ecadebug->msg(m.to_string());

  current = 0;
}

GENERIC_OSCILLATOR::~GENERIC_OSCILLATOR (void) {
   while(ienvelope.size() > 0) ienvelope.pop_back();
}

void GENERIC_OSCILLATOR::read_envelope(const string& filename, int preset_number) {
  ifstream fin (filename.c_str());

  if (!fin) {
    throw(new ECA_ERROR("OSC-GEN", "Unable to open envelope file"
			    + filename + "."));
  }

  int curpreset;
  string sana;
  while(fin >> sana) {
    if (sana.size() > 0 && sana[0] == '#') {
      while(fin.get() != '\n' && fin.eof() == false);
      continue;
    }
    else {
      ecadebug->msg(5, "(osc-gen) Next preset is " + sana + ".");
      curpreset = atoi(sana.c_str());
      if (curpreset == preset_number) {
	ecadebug->msg(5, "(osc-gen) Found the right preset!");
	preset_found = true;
	fin >> sana; 
	if (fin.eof()) break; 
	else if (sana.size() > 0 && sana[0] == 'L') {
	  linear = true;
	  ecadebug->msg(2,"(osc-gen) Using linear-interpolation between
		   envelope points.");
	}
	else {
	  linear = false;
	  ecadebug->msg(2,"(osc-gen) Using static envelope points.");
	}
	double newpoint; 
	while(fin >> newpoint) {
	  ecadebug->msg(5, "(osc-gen) Added value: " + kvu_numtostr(newpoint));
	  ienvelope.push_back(newpoint);
	}
      }
      else 
	while(fin.get() != '\n' && fin.eof() == false);
    }
  }
}

double GENERIC_OSCILLATOR::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return((double)preset);
  }
}
