#ifndef _ECA_CONTROLLER_H
#define _ECA_CONTROLLER_H

#include <pthread.h>

#include "audioio.h"
#include "eca-chain.h"
#include "eca-session.h"
#include "dynamic-parameters.h"

class CHAIN_OPERATOR;
class ECA_CHAINSETUP;

extern string ecasound_lockfile;

enum {
  ECA_QUIT = 1
};

class ECA_CONTROLLER {

  ECA_SESSION* session; 
  int retcode;
  pthread_t th_cqueue;
  bool engine_started;

  void start_engine(bool ignore_lock = false);
  void close_engine(void);

 public:

  enum DIRECTION { input, output };

  /**
   * Parse string mode command and act accordingly.
   */
  void command(const string& cmd);

  // -------------------------------------------------------------------
  // Runtime control
  // -------------------------------------------------------------------

  /**
   * Start the processing engine
   *
   * require:
   *   is_engine_started() == true
   *
   * ensure:
   *   is_running() == true
   */
  void start(bool ignore_lock = false);

  /**
   * Stop the processing engine
   *
   * ensure:
   *   is_running() == false
   */
  void stop(void);

  /**
   * Stop the processing engine and throw an ECA_QUIT exception.
   */
  void quit(void);

  // -------------------------------------------------------------------
  // Session info / global resources (~/.ecasoundrc)
  // -------------------------------------------------------------------

  /**
   * Get resource values from ~/.ecasoundrc
   */
  const string& resource_value(const string& key) { return session->ecaresources.resource(key); }

  // -------------------------------------------------------------------
  // Session info / output to ecadebug
  // -------------------------------------------------------------------
  
  void print_general_status(void);
  void print_chain_status(void);
  void print_effect_status(void);
  void print_file_status(void);

  // -------------------------------------------------------------------
  // Session info / functions
  // -------------------------------------------------------------------
  
  /**
   * Get a string containing a comma separated list of all chains 
   * attached to input with index 'aiod'. 
   */
  string connected_chains_input(CHAIN::aio_id_type aiod);

  /**
   * Get a string containing a comma separated list of all chains 
   * attached to output with index 'aiod'. 
   */
  string connected_chains_output(CHAIN::aio_id_type aiod);

  /**
   * Get a string containing a comma separated list of all chains 
   * attached to audio object with name 'filename'. 
   */
  vector<string> connected_chains(const string& name);

  // -------------------------------------------------------------------
  // Session info / master input
  // -------------------------------------------------------------------

  long length_in_samples(void) const { return session->length_in_samples(); }
  double length_in_seconds_exact(void) const { return session->length_in_seconds_exact(); }
  long position_in_samples(void) const { return session->position_in_samples(); }
  double position_in_seconds_exact(void) const { return session->position_in_seconds_exact(); }

  // -------------------------------------------------------------------
  // Modify session
  // -------------------------------------------------------------------

  void toggle_interactive_mode(bool v) { session->iactive = v; } 
  void toggle_sfx(bool v) { session->sfx = v; } 
  void toggle_looping(bool v) { session->loop_active = v; } 
  void toggle_multitrack_mode(bool v) { session->multitrack_mode = v; } 

  /**
   * Set the default buffersize (in samples).
   *
   * require:
   *   is_editable() == true
   */
  void set_buffersize(int bsize);

  /**
   * Toggle whether raised priority mode is enabled or not.
   *
   * require:
   *   is_editable() == true
   */
  void toggle_raise_priority(bool v);

  // -------------------------------------------------------------------
  // Session status
  // -------------------------------------------------------------------

  /**
   * Returns true if processing engine is running.
   */
  bool is_running(void);

  /**
   * Returns true if active chainsetup exists and is connected.
   */
  bool is_connected(void) const;

  /**
   * Returns true if active chainsetup has been set. 
   */
  bool is_editable(void) const;

  /**
   * Returns true if engine has been started. 
   */
  bool is_engine_started(void) const { return(engine_started); }

  /**
   * Return info about engine status.
   */
  string engine_status(void);

  // -------------------------------------------------------------------
  // Chainsetups  (if not specified, active chainsetup is used)
  // -------------------------------------------------------------------

  /**
   * Activate chainsetup.
   *
   * @param name chainsetup name 
   * @param handle_errors whether to handle exceptions
   *
   * require:
   *   is_engine_started() == false
   *
   * ensure:
   *   name == active_chainsetup()
   */
  void activate_chainsetup(const string& name, bool handle_errors = true);

  /**
   * Name of currently active chainsetup.
   */
  string active_chainsetup(void);

  /**
   * Deactivate chainsetup.
   *
   * @param name chainsetup name 
   * @param handle_errors whether to handle exceptions
   *
   * require:
   *   name == active_chainsetup()
   *
   * ensure:
   *   active_chainsetup() == ""
   */
  void deactivate_chainsetup(const string& name, bool handle_errors = true);

  /**
   * Connect activate chainsetup.
   *
   * require:
   *   is_editable() == true
   *
   * ensure:
   *   is_connected() == true
   */
  void connect_active_chainsetup(void);

  /**
   * Name of connected chainsetup.
   */
  string connected_chainsetup(void);

  /**
   * Disconnect activate chainsetup.
   *
   * require:
   *   active_chainsetup() == connected_chainsetup()
   *
   * ensure:
   *   connected_chainsetup() == ""
   */
  void disconnect_active_chainsetup(void);

  /**
   * Add a new chainsetup.
   */
  void new_chainsetup(const string& name);

  /**
   * Remove chainsetup.
   *
   * @param name chainsetup name 
   * @param handle_errors whether to handle exceptions
   *
   * require:
   *   name != active_chainsetup()
   */
  void remove_chainsetup(const string& name, bool handle_errors = true);

  /**
   * Get a const pointer to chainsetup 'name'.
   */
  const ECA_CHAINSETUP* get_chainsetup(const string& name);

  /**
   * Load chainsetup from file 'filename' and set it active.
   *
   * @param name chainsetup filename 
   * @param handle_errors whether to handle exceptions
   */
  void load_chainsetup(const string& filename, bool handle_errors = true);

  /**
   * Save chainsetup to file 'filename'.
   *
   * @param csetup name of the chainsetup
   * @param name chainsetup filename 
   * @param handle_errors whether to handle exceptions
   */
  void save_chainsetup(const string& csetup, const string filename, bool set_filename, bool handle_errors = true);

  /**
   * Save active chainsetup to file 'filename'.
   *
   * @param name chainsetup filename 
   * @param handle_errors whether to handle exceptions
   *
   * require:
   *   is_connected() == true
   */
  void save_active_chainsetup(const string& filename, bool handle_errors = true);

  // -------------------------------------------------------------------
  // Chains (if not specified, active chainsetup is used)
  // -------------------------------------------------------------------

  /**
   * Add a new chain to the currently active chainsetup.
   *
   * @param name chain name 
   *
   * require:
   *   is_engine_started() == false &&
   *   is_editable() == true
   *
   * ensure:
   *   get_active_chains().size() > 0 &&
   *   get_active_chains()[0] == name
   */
  void add_chain(const string& name);

  /**
   * Remove chain from the currently active chainsetup.
   *
   * @param name chain name 
   *
   * require:
   *   is_engine_started() == false &&
   *   is_editable() == true
   *
   * ensure:
   *   get_active_chains().size() == 0
   */
  void remove_chain(const string& name);

  void set_active_chains(const vector<string>& chains);
  const vector<string>& get_active_chains(void) const;

  // -------------------------------------------------------------------
  // Audio-devices  (active chainsetup is edited)
  // -------------------------------------------------------------------

  void set_audio_format(int bits, int channels, long int srate);
  void set_audio_format(const AIO_PARAMS* format);
  void get_audio_format(const string& name, AIO_PARAMS* format);
  void set_explicit_format(bool enabled);
  void set_io_mode(SIMODE iomode);
  void add_audio_device(const string& filename, DIRECTION dir);
  AUDIO_IO* get_audio_device(const string& csname, const string& name);
  void add_default_output(void);
  void remove_audio_device(const string& filename);
  void attach_iodev_to_active_chains(const string& filename);

  // -------------------------------------------------------------------
  // Chain operators  (if not specified, active chainsetup is used)
  // -------------------------------------------------------------------

  void add_chain_operator(const string& chainop_params);
  void add_chain_operator(CHAIN_OPERATOR* cotmp);
  const CHAIN_OPERATOR* get_chain_operator(const string& chain, int chainop_id);
  void remove_chain_operator(const string& chain, int chainop_id);
  void set_chain_operator_parameter(const string& chain, 
				    int chainop_id,
				    int param,
				    DYNAMIC_PARAMETERS::parameter_type value);
  void add_general_controller(const string& gcontrol_params);

  ECA_CONTROLLER (ECA_SESSION* psession);
};

void show_controller_help(void);
void show_controller_help_more(void);

void start_normal_thread(ECA_SESSION* param, int retcode, pthread_t* th_cqueue);
void* start_normal(void* param);
void start_normal(ECA_SESSION* param);

#endif

